;; minlog.el Proof General instance for Minlog
;;
;; Copyright (C) 2004 Stefan Schimanski, Freiric Barral
;;
;; Author: Stefan Schimanski <schimans@math.lmu.de>
;;         Freiric Barral <barral@math.lmu.de>
;;

(require 'proof-easy-config)
(require 'minlog-config)
(require 'minlog-syntax)
(require 'minlog-abbrev)

(setq minlog-mzfork-base-filename "mzfork")
(setq minlog-mzfork-extension-filename "so")
(require 'minlog-mzfork "minlog-mzfork.el" t)

(require 'holes)

(defvar used-minlog-scheme nil
  "The currently used minlog scheme")

(defun detect-minlog-scheme ()
  "Find out whether mzscheme or guile is installed"
  (when (not used-minlog-scheme)
    (if (eq minlog-scheme 'auto)
        ;; detect the available scheme version         
        (cond
         ((not(string= "" (shell-command-to-string "which mzscheme")))
          (setq used-minlog-scheme 'mzscheme))
         ((not(string= "" (shell-command-to-string "which guile")))
          (setq used-minlog-scheme 'guile))
         (t (error "Neither mzscheme nor guile installed.")))
      (setq used-minlog-scheme minlog-scheme)))
  used-minlog-scheme)

(defun minlog-send-undo-point ()
  "Send (undo-point) before each command"
  (if (eql action 'proof-done-advancing)
      (setq string
	    (if (eq (detect-minlog-scheme) 'guile)
		(concat "(begin (if (pg-guile-start)\n" string "\n)(pg-end))")
	      (concat "(pg-start)\n" 
		      string "\n(pg-end)")))))

(defun minlog-find-and-forget (span)
  (let ((c 0))
    (while span
      (if (not (eq (span-property span 'type) 'comment))
	  (let ((nestedundos (span-property span 'nestedundos)))
	    (setq c (+ c (if nestedundos (+ 2 nestedundos) 1))))) 
      (setq span (next-span span 'type)))
    (format "(undo-steps %d) " c)))

(defun minlog-handle-error ()
  "Go back to state before the last error"
  (proof-shell-insert "(undo-error)\n" 'proof-done-retracting))

(defun minlog-init-unicode ()
  "Initialize minlog buffer"
  (save-excursion 
    (set-buffer "*minlog*")
    (toggle-enable-multibyte-characters 1)))

(proof-easy-config 'minlog "Minlog" 
		   proof-prog-name                 (if (eq (detect-minlog-scheme) 'guile)
						       "guile" "mzscheme")
		   proof-assistant-home-page       "http://www.minlog-system.de/"
		   
		   proof-script-sexp-commands      t
		   proof-script-comment-start      ";"

		   proof-save-command-regexp       "(save"
		   proof-goal-command-regexp       "(set-goal"
		   proof-save-with-hole-regexp     "(save[:space:]*\"\\([^\"]*\\)\"[:space:]*)"
					; proof-save-with-hole-result	 1

		   proof-showproof-command         "(display-proof)"
		   proof-context-command           "(dcg)"
		   proof-auto-multiple-files       t
 
		   proof-undo-n-times-cmd          "(undo-steps %s)"
		   proof-kill-goal-command         nil
		   proof-nested-undo-regexp        "."
		   proof-find-and-forget-fn        'minlog-find-and-forget
		   proof-count-undos-fn            'minlog-find-and-forget

		   proof-shell-cd-cmd              (if (eq (detect-minlog-scheme) 'guile)
                                                       "(chdir \"%s\")" 
                                                     "(current-directory \"%s\")")
		   proof-shell-interrupt-regexp    "user break"

		   proof-shell-pre-sync-init-cmd   (concat
						    "(define display-prompt? #t)\n"
;;; bug fix : If Proof general say pg-start not defined 
;;; uncomment the following line:
;				  "(define COQ-GOAL-DISPLAY #t)"
						    (if (eq (detect-minlog-scheme) 'guile)
							""
						      "(current-prompt-read 
                                      (lambda () 
                                       (begin 
(if display-prompt? (display \"guile> \")) 
                                         (read-syntax current-input-port))))\n"))

		   proof-activate-scripting-hook   'minlog-init-unicode

		   proof-shell-annotated-prompt-regexp "guile> "
		   proof-shell-prompt-pattern      "guile> "

		   proof-shell-start-goals-regexp   "; [0-9]* open goals?:"
;		   proof-shell-start-goals-regexp   "; ZU ZEIGEN:"
		   proof-shell-proof-completed-regexp "Proof finished"

		   proof-shell-truncate-before-error nil
		   proof-shell-error-regexp        "ABORT:\\|Minlog \"sorry\""
		   proof-shell-handle-error-or-interrupt-hook 'minlog-handle-error

		   proof-shell-insert-hook         'minlog-send-undo-point
		   proof-shell-strip-crs-from-input nil

		   proof-shell-restart-cmd         "(undo-max)"
		   proof-shell-quit-cmd            "(exit)"

                   proof-script-font-lock-keywords  minlog-font-lock-keywords
                   proof-goals-font-lock-keywords  minlog-font-lock-keywords
                   proof-response-font-lock-keywords  minlog-font-lock-keywords
                   proof-trace-font-lock-keywords  minlog-font-lock-keywords

		   proof-shell-init-cmd
		   (concat "(begin (load \"" minlog-path "/init.scm\")"
(if (eq (detect-minlog-scheme) 'guile) 
    ""
  (concat 
   "(if (< (string->number (version)) 300) (load-extension " 
   "(string-append \""
   proof-home-directory "minlog/" 
   minlog-mzfork-base-filename ".\"(version)\"." minlog-mzfork-extension-filename "\"))"
   "(load \"" proof-home-directory "minlog/mzfork.scm\"))" 
 "(error-display-handler 
  (lambda (message exception) 
    (display \"ABORT: \") 
    (display message) (newline)))"))
"
(set! COQ-GOAL-DISPLAY #t)

(define undo-level 0)
(define undo-steps 
  (lambda (steps) 
    (do ((n steps (- n 1)))
		((or (= undo-level 0) (= n 0))
		 (if (= n 0) (display-goals) (undo-point)))
	  (if (car fork-history)
		  (begin
			(exit n))
		  (let ((undo-steps (- PPROOF-STATE-HISTORY-LENGTH 
				       (car undo-history))))
		    (if (> undo-steps 0) (undo undo-steps))
		    (set! undo-history (cdr undo-history))
		    (set! fork-history (cdr fork-history))
		    (set! undo-level (- undo-level 1)))))))
"
(if (eq (detect-minlog-scheme) 'guile)
    "
(define (wait-and-get-ret pid) 
  (begin 
   (drain-input (current-input-port)) 
   (status:exit-val (cdr (waitpid pid)))))"
  "(define wait-and-get-ret waitpid)")"
	
(define undo-max 
  (lambda () 
	(if (not (eq? undo-level 0)) (exit 1))))

(define fork-history '())
(define undo-history '())

(define undo-error (lambda () (undo-steps 1)))

(define undo-point
  (lambda ()
	(begin
	 (set! undo-level (+ undo-level 1))
	 (if (null? (pproof-state-to-num-goals))
		 (let ((pid (primitive-fork)))
		   (if (not (= pid 0))
			   (begin
				(set! undo-level (- undo-level 1))
				(let ((ret (wait-and-get-ret pid)))
				  (if (= ret 0) 
					  (exit 0)
					  (undo-steps (- ret 1))))
				1)
			   (begin (set! fork-history (cons #t fork-history)) 0)))
		 (begin 
		  (set! fork-history (cons #f fork-history))
		  (set! undo-history 
			(cons PPROOF-STATE-HISTORY-LENGTH undo-history))
		  0)))))

(define (display-goals)
  (if (not (null? (pproof-state-to-num-goals)))
      (let ((num-of-goals (length (pproof-state-to-num-goals))))
	    (display-comment (number-to-string num-of-goals)
				 \" open goal\" (if (> num-of-goals 1) \"s:\" \":\"))
;				 \" ZU ZEIGEN:\" )
		(newline)
		(newline)
		(display-num-goal (car (pproof-state-to-num-goals)))
		(newline)
		(do ((num-goals (cdr (pproof-state-to-num-goals)) (cdr num-goals)))
		    ((null? num-goals) (display \"\"))
			(let* ((num-goal (car num-goals))
			 (formula (goal-to-formula (num-goal-to-goal num-goal)))
				 (number (num-goal-to-number num-goal))
				 (prefix (string-append DEFAULT-GOAL-NAME \"_\"
							(number-to-string number) \": \")))
			(newline)
			(display-comment
			 prefix
			 (pretty-print-string (string-length prefix) 
					      (- pp-width (string-length COMMENT-STRING))
					      (fold-formula formula)))
			(newline))))))

(define (pg-guile-start) (if (= (undo-point) 0)
			     (begin 
			      (set! display-prompt? #f)
			      (set! COQ-GOAL-DISPLAY #t)
			      #t)
			   #f))
(define (pg-start) (pg-guile-start) (display \"\"))
(define (pg-end) 
  (set! display-prompt? #t) 
  (display-goals))

(undo-point))
"
)
proof-find-theorems-command      "(display-theorems)")

;;; Some documentation of minlog can be launched within  proofgeneral

 
(defun minlog-help ()
  (save-window-excursion
    (save-excursion 
      (shell-command 
       (concat pdf-viewer  " " minlog-path "/doc/ref.pdf&"))))
  )


 (defpgdefault help-menu-entries
   '(     
["Reference Manual" (minlog-help)  t])
     
)

;; the base mode of every proofgeneral mode should be scheme-mode
(setq proof-proof-parent-mode 'scheme-mode)

;; comment out minlog code
(global-set-key [?\C-c ?\;] 'comment-region)
;; uncomment minlog code
(global-set-key [?\C-c ?:] 'uncomment-region)

(provide 'minlog)
