;; $Id: axiom.scm 2688 2014-01-24 09:18:17Z schwicht $
;; 8. Assumption variables and axioms
;; ==================================
;; To be renamed into avars scheme, with the axioms section transferred
;; into the new aconst.scm (was globalas.scm)

;; 8-1. Assumption variables
;; =========================

;; Assumption variables are implemented as lists ('avar formula index name).

;; To make sure that assumption variables generated by the system are
;; different from all user introduced assumption variables, we maintain a
;; global counter MAXAVARINDEX.  Whenever the user introduces an
;; assumption variable, e.g. by (make-avar formula 27 ""), then MAXAVARINDEX
;; is incremented to at least 27.

(define MAXAVARINDEX -1)

;; Constructor, accessors and tests for assumption variables:

(define (make-avar formula index name)
  (set! MAXAVARINDEX (max index MAXAVARINDEX))
  (list 'avar formula index name))

(define avar-to-formula cadr)
(define avar-to-index caddr)
(define avar-to-name cadddr)

;; avar-form? moved next to compose-substitutions in typ.scm, where it
;; is used already.

;; (define (avar-form? x) (and (pair? x) (eq? 'avar (car x))))

(define (avar? x)
  (and (avar-form? x)
       (list? x)
       (= 4 (length x))
       (let ((formula (cadr x))
	     (index (caddr x))
	     (name (cadddr x)))
	 (and (formula? formula)
	      (<= -1 index)
	      (<= index MAXAVARINDEX)
	      (string? name)))))

(define (avar=? avar1 avar2)
  (or (eq? avar1 avar2)
      (and (avar-form? avar1) (avar-form? avar2)
	   (= (avar-to-index avar1) (avar-to-index avar2))
	   (string=? (avar-to-name avar1) (avar-to-name avar2)))))

;; For display we use

(define (avar-to-string avar)
  (let ((name (avar-to-name avar))
	(index (avar-to-index avar)))
    (string-append
     (if (string=? "" name) DEFAULT-AVAR-NAME name)
     (if (= -1 index) "" (number-to-string index)))))

;; For automatic generation of assumption variables (e.g. for bound
;; renaming) we provide

(define (formula-to-new-avar formula . optional-name)
  (if (null? optional-name)
      (make-avar formula (+ 1 MAXAVARINDEX) "")
      (let ((string (car optional-name)))
	(if (string? string)
	    (make-avar formula (+ 1 MAXAVARINDEX) string)
	    (myerror "formula-to-new-avar" "string expected"
		     (car optional-name))))))

(define DEFAULT-AVAR-NAME "u")

;; For convenience we add mk-avar with options.  Options are index (default
;; -1) and name (default DEFAULT-AVAR-NAME)

(define (mk-avar formula . options)
  (let ((index -1)
	(name DEFAULT-AVAR-NAME))
    (if (pair? options)
	(begin (set! index (car options))
	       (set! options (cdr  options))))
    (if (pair? options)
	(begin (set! name (car options))
	       (set! options (cdr  options))))
    (if (pair? options)
	 (myerror "make-avar" "unexpected argument" options))
  (cond ((not (and (integer? index) (<= -1 index)))
	 (myerror "make-avar" "index >= -1 expected" index))
	((not (formula-form? formula))
	 (myerror "make-avar" "formula expected" formula))
	((not (string? name))
	 (myerror "make-avar" "string expected" name))
	(else (make-avar formula index name)))))

(define (normalize-avar avar)
  (make-avar (normalize-formula (avar-to-formula avar))
	     (avar-to-index avar)
	     (avar-to-name avar)))

;; 8-2. Assumption constants
;; =========================

;; An assumption constant appears in a proof, as an axiom, a theorem or
;; a global assumption.  Its formula is given as an "uninstantiated
;; formula", where only type and predicate variables can occur free;
;; these are considered to be bound in the assumption constant.  An
;; exception are the Elim and Gfp aconsts, where the argument variables
;; xs^ of the (co)inductively defined predicate are formally free in the
;; uninstantiated formula;; however, they are considered bound as well.
;; In the proof the bound type variables are implicitely instantiated
;; by types, and the bound predicate variables by cterms.  Since we do
;; not have type and predicate quantification in formulas, the aconst
;; contains these parts left implicit in the proof: tpsubst.

;; To normalize a proof we will first translate it into a term, then
;; normalize the term and finally translate the normal term back into a
;; proof.  To make this work, in case of axioms we pass to the term
;; appropriate "reproduction data" to be used when after normalization
;; the axiom in question is to be reconstructed: all-formulas for
;; induction, a number i and an inductively defined predicate constant
;; idpc for its ith clause, imp-formulas for elimination, an
;; existential formula for existence introduction, and an existential
;; formula together with a conclusion for existence elimination.
;; During normalization of the term these formulas are passed along.
;; When the normal form is reached, we have to translate back into a
;; proof.  Then these reproduction data are used to reconstruct the
;; axiom in question, via

;; all-formulas-to-ind-aconst
;; number-and-idpredconst-to-intro-aconst
;; imp-formulas-to-elim-aconst
;; all-formula-to-cases-aconst
;; ex-formula-to-ex-intro-aconst
;; ex-formula-and-concl-to-ex-elim-aconst
;; exnc-formula-to-exnc-intro-aconst ;obsolete
;; exnc-formula-and-concl-to-exnc-elim-aconst ;obsolete

;; The reproduction data can be computed from the name, the
;; uninstantiated formula, the tpsubst of an axiom, by
;; aconst-to-computed-repro-data .  However, to avoid recomputations we
;; carry them along.

(define (make-aconst name kind uninst-formula tpsubst . repro-data)
  (append (list 'aconst name kind uninst-formula tpsubst)
	  repro-data))

(define aconst-to-name cadr)
(define aconst-to-kind caddr)
(define aconst-to-uninst-formula cadddr)
(define (aconst-to-tpsubst x) (car (cddddr x)))
(define (aconst-to-repro-data x) (cdr (cddddr x)))

;; To construct the formula associated with an aconst, it is useful to
;; separate the instantiated formula from the variables to be
;; generalized.  The latter can be obtained as free variables in
;; inst-formula.

(define (aconst-to-inst-formula aconst)
  (let* ((uninst-formula (aconst-to-uninst-formula aconst))
	 (tpsubst (aconst-to-tpsubst aconst))
	 (name (aconst-to-name aconst)))
    (if (or (string=? "Elim" name) (string=? "Gfp" name))
	(all-form-to-final-kernel
	 (formula-substitute
	  (apply mk-all (append (formula-to-free uninst-formula)
				(list uninst-formula)))
	  tpsubst))
	(formula-substitute uninst-formula tpsubst))))

(define (aconst-to-formula aconst)
  (let* ((inst-formula (aconst-to-inst-formula aconst))
	 (free (formula-to-free inst-formula)))
    (apply mk-allnc (append free (list inst-formula)))))

(define (aconst-form? x) (and (pair? x) (eq? 'aconst (car x))))

;; The reproduction data can be computed from the name, the
;; uninstantiated formula, the tpsubst of an axiom, by
;; aconst-to-computed-repro-data .  However, to avoid recomputations we
;; carry them along.

(define (aconst-to-computed-repro-data aconst)
  (let* ((name (aconst-to-name aconst))
	 (uninst-formula (aconst-to-uninst-formula aconst))
	 (tpsubst (aconst-to-tpsubst aconst))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (psubst (list-transform-positive tpsubst
		  (lambda (x) (pvar-form? (car x))))))
    (cond
     ((string=? name "Ind")
      (let ((cterms (map cadr psubst)))
	(map (lambda (cterm)
	       (let* ((vars (cterm-to-vars cterm))
		      (formula (cterm-to-formula cterm))
		      (var (if (= 1 (length vars)) (car vars)
			       (myerror "aconst-to-computed-repro-data"
					"unary cterm expected" cterm))))
		 (if (t-deg-zero? (var-to-t-deg var))
		     (myerror "aconst-to-computed-repro-data"
			      "cterm with total variable expected" cterm)
		     (make-all var formula))))
	     cterms)))
     ((string=? name "Cases")
      (let* ((cterms (map cadr psubst))
	     (cterm (if (= 1 (length cterms)) (car cterms)
			(myerror "aconst-to-computed-repro-data"
				 "only one cterm expected" cterms)))
	     (vars (cterm-to-vars cterm))
	     (formula (cterm-to-formula cterm))
	     (var (if (= 1 (length vars)) (car vars)
		      (myerror "aconst-to-computed-repro-data"
			       "unary cterm expected" cterm))))
	(if (t-deg-zero? (var-to-t-deg var))
	    (myerror "aconst-to-computed-repro-data"
		     "cterm with total variable expected" cterm)
	    (list (make-all var formula)))))
     ((string=? name "Ex-Intro")
      (let* ((cterms (map cadr psubst))
	     (cterm (if (= 1 (length cterms)) (car cterms)
			(myerror "aconst-to-computed-repro-data"
				 "only one cterm expected" cterms)))
	     (vars (cterm-to-vars cterm))
	     (formula (cterm-to-formula cterm))
	     (var (if (= 1 (length vars)) (car vars)
		      (myerror "aconst-to-computed-repro-data"
			       "unary cterm expected" cterm))))
	(list (make-ex var formula))))
     ((string=? name "Ex-Elim")
      (let* ((cterms (map cadr psubst))
	     (cterm1 (if (= 2 (length cterms)) (car cterms)
			 (myerror "aconst-to-computed-repro-data"
				  "two cterms expected" cterms)))
	     (cterm2 (cadr cterms))
	     (vars1 (cterm-to-vars cterm1))
	     (var1 (if (= 1 (length vars1)) (car vars1)
		       (myerror "aconst-to-computed-repro-data"
				"unary cterm expected" cterm1)))
	     (formula1 (cterm-to-formula cterm1))
	     (vars2 (cterm-to-vars cterm2))
	     (formula2 (if (null? vars2) (cterm-to-formula cterm2)
		       (myerror "aconst-to-computed-repro-data"
				"nullary cterm expected" cterm2))))
	(list (make-ex var1 formula1) formula2)))
     ((string=? name "Exnc-Intro") ;obsolete
      (let* ((cterms (map cadr psubst))
	     (cterm (if (= 1 (length cterms)) (car cterms)
			(myerror "aconst-to-computed-repro-data"
				 "only one cterm expected" cterms)))
	     (vars (cterm-to-vars cterm))
	     (formula (cterm-to-formula cterm))
	     (var (if (= 1 (length vars)) (car vars)
		      (myerror "aconst-to-computed-repro-data"
			       "unary cterm expected" cterm))))
	(list (make-exnc var formula))))
     ((string=? name "Exnc-Elim") ;obsolete
      (let* ((cterms (map cadr psubst))
	     (cterm1 (if (= 2 (length cterms)) (car cterms)
			 (myerror "aconst-to-computed-repro-data"
				  "two cterms expected" cterms)))
	     (cterm2 (cadr cterms))
	     (vars1 (cterm-to-vars cterm1))
	     (var1 (if (= 1 (length vars1)) (car vars1)
		       (myerror "aconst-to-computed-repro-data"
				"unary cterm expected" cterm1)))
	     (formula1 (cterm-to-formula cterm1))
	     (vars2 (cterm-to-vars cterm2))
	     (formula2 (if (null? vars2) (cterm-to-formula cterm2)
		       (myerror "aconst-to-computed-repro-data"
				"nullary cterm expected" cterm2))))
	(list (make-exnc var1 formula1) formula2)))
     ((member name '("Intro"))
      (intro-aconst-to-computed-repro-data aconst))
     ((string=? name "Elim")
      (elim-aconst-to-computed-repro-formulas aconst))
     ((string=? name "GInd")
      (if (= 1 (length psubst))
	  (let* ((cterm (cadar psubst))
		 (vars (cterm-to-vars cterm))
		 (formula (cterm-to-formula cterm)))
	    (list (apply mk-all (append vars (list formula)))))
	  (myerror "aconst-to-computed-repro-data"
		   "a single pvar instantiation expected" psubst)))
     ((string=? name "ElimMR")
      (mr-elim-aconst-to-computed-repro-formulas aconst))
     (else '()))))

(define (mr-elim-aconst-to-computed-repro-formulas aconst)
  (let* ((name (aconst-to-name aconst))
	 (uninst-formula (aconst-to-uninst-formula aconst))
	 (tpsubst (aconst-to-tpsubst aconst))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (psubst (list-transform-positive tpsubst
		  (lambda (x) (pvar-form? (car x)))))
	 (uninst-idpc-formula
	  (imp-form-to-premise (all-form-to-kernel uninst-formula)))
	 (uninst-idpc (predicate-form-to-predicate uninst-idpc-formula))
	 (idpc-name (if (idpredconst-form? uninst-idpc)
			(idpredconst-to-name uninst-idpc)
			(myerror "mr-elim-aconst-to-computed-repro-formulas"
				 "idpredconst expected" uninst-idpc)))
	 (uninst-types (idpredconst-to-types uninst-idpc))
	 (uninst-param-cterms (idpredconst-to-cterms uninst-idpc))
	 (idpc-names-with-pvars-and-opt-alg-names
	   (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	    idpc-name))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (relevant-pvars ;in the given order, as determined by psubst
	  (list-transform-positive (map car psubst)
	    (lambda (pvar) (member pvar pvars))))
	 (pvar-name-alist (map (lambda (x) (list (cadr x) (car x)))
			       idpc-names-with-pvars-and-opt-alg-names))
	 (relevant-idpc-names
	  (map (lambda (pvar)
		 (let ((info (assoc pvar pvar-name-alist)))
		   (if info (cadr info)
		       (myerror "mr-elim-aconst-to-computed-repro-formulas"
				"unexpected pvar" pvar))))
	       relevant-pvars))
	 (relevant-uninst-idpcs
	  (map (lambda (name)
		 (make-idpredconst name uninst-types uninst-param-cterms))
	       relevant-idpc-names))
	 (relevant-cterms
	  (map cadr (list-transform-positive psubst
		      (lambda (x) (member (car x) relevant-pvars)))))
	 (inst-formula (aconst-to-inst-formula aconst))
	 (inst-idpc-formula
	  (imp-form-to-premise (all-form-to-kernel inst-formula)))
	 (inst-idpc (predicate-form-to-predicate inst-idpc-formula))
	 (inst-types (idpredconst-to-types inst-idpc))
	 (inst-param-cterms (idpredconst-to-cterms inst-idpc))
	 (relevant-inst-idpcs
	  (map (lambda (name)
		 (make-idpredconst name inst-types inst-param-cterms))
	       relevant-idpc-names))
	 (pvars (map idpredconst-name-to-pvar relevant-idpc-names))
	 (cterms (map (lambda (pvar) (cadr (assoc pvar psubst))) pvars))
	 (var-lists (map cterm-to-vars cterms))
	 (relevant-inst-idpc-formulas
	  (map (lambda (idpc vars)
		 (apply make-predicate-formula
			idpc (map make-term-in-var-form vars)))
	       relevant-inst-idpcs var-lists)))
    (map (lambda (idpc-formula concl)
	   (make-imp idpc-formula concl))
	 relevant-inst-idpc-formulas (map cterm-to-formula relevant-cterms))))

(define (intro-aconst-to-computed-repro-data aconst)
  (let* ((uninst-formula (aconst-to-uninst-formula aconst))
	 (tpsubst (aconst-to-tpsubst aconst))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (uninst-idpc (predicate-form-to-predicate
		       (imp-impnc-all-allnc-form-to-final-conclusion
			uninst-formula)))
	 (name (idpredconst-to-name uninst-idpc))
	 (params (idpredconst-name-to-params name))
	 (uninst-clause
	  (allnc-form-to-final-kernel uninst-formula (length params)))
	 (orig-clauses (idpredconst-name-to-clauses name))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (cterms (map (lambda (pvar)
			(let ((info (assoc pvar tpsubst)))
			  (if info (cadr info)
			      (if (member name '("ExDT" "ExLT" "ExRT" "ExUT"))
				  (predicate-to-cterm-with-total-vars pvar)
				  (predicate-to-cterm pvar)))))
		      param-pvars))
	 (param-pvar-cterms
	  (if (member name '("ExDT" "ExLT" "ExRT" "ExUT"))
	      (map predicate-to-cterm-with-total-vars param-pvars)
	      (map predicate-to-cterm param-pvars)))
	 (idpc-names-with-pvars-and-opt-alg-names
	  (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	   name))
	 (names (map car idpc-names-with-pvars-and-opt-alg-names))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (tvars (idpredconst-name-to-tvars name))
	 (uninst-idpcs (map (lambda (name)
			       (make-idpredconst name tvars param-pvar-cterms))
			     names))
	 (uninst-idpc-cterms (map predicate-to-cterm uninst-idpcs))
	 (psubst-for-pvars (make-substitution-wrt pvar-cterm-equal?
						  pvars uninst-idpc-cterms))
	 (i (do ((n 0 (+ 1 n))
		 (l orig-clauses (cdr l)))
		((or (classical-formula=?
		      (formula-substitute (car l) psubst-for-pvars)
		      uninst-clause)
		     (= n (length orig-clauses)))
		 (if (= n (length orig-clauses))
		     (myerror "intro-aconst-to-computed-repro-data"
			      "clause not found"
			      uninst-clause)
		     n))))
	 (types (map (lambda (type) (type-substitute type tsubst)) tvars))
	 (idpc (make-idpredconst name types cterms)))
    (list i idpc)))

(define (elim-aconst-to-computed-repro-formulas aconst)
  (let* ((name (aconst-to-name aconst))
	 (uninst-formula (aconst-to-uninst-formula aconst))
	 (tpsubst (aconst-to-tpsubst aconst))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (psubst (list-transform-positive tpsubst
		  (lambda (x) (pvar-form? (car x)))))
	 (uninst-idpc-formula (imp-form-to-premise uninst-formula))
	 (uninst-idpc (predicate-form-to-predicate uninst-idpc-formula))
	 (idpc-name (if (idpredconst-form? uninst-idpc)
			(idpredconst-to-name uninst-idpc)
			(myerror "elim-aconst-to-computed-repro-formulas"
				 "idpredconst expected" uninst-idpc)))
	 (uninst-types (idpredconst-to-types uninst-idpc))
	 (uninst-param-cterms (idpredconst-to-cterms uninst-idpc))
	 (idpc-names-with-pvars-and-opt-alg-names
	   (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	    idpc-name))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (relevant-pvars ;in the given order, as determined by psubst
	  (list-transform-positive (map car psubst)
	    (lambda (pvar) (member pvar pvars))))
	 (pvar-name-alist (map (lambda (x) (list (cadr x) (car x)))
			       idpc-names-with-pvars-and-opt-alg-names))
	 (relevant-idpc-names
	  (map (lambda (pvar)
		 (let ((info (assoc pvar pvar-name-alist)))
		   (if info (cadr info)
		       (myerror "elim-aconst-to-computed-repro-formulas"
				"unexpected pvar" pvar))))
	       relevant-pvars))
	 (relevant-uninst-idpcs
	  (map (lambda (name)
		 (make-idpredconst name uninst-types uninst-param-cterms))
	       relevant-idpc-names))
	 (relevant-cterms
	  (map cadr (list-transform-positive psubst
		      (lambda (x) (member (car x) relevant-pvars)))))
	 (inst-formula (aconst-to-inst-formula aconst))
	 (inst-idpc-formula (imp-form-to-premise inst-formula))
	 (inst-idpc (predicate-form-to-predicate inst-idpc-formula))
	 (inst-types (idpredconst-to-types inst-idpc))
	 (inst-param-cterms (idpredconst-to-cterms inst-idpc))
	 (relevant-inst-idpcs
	  (map (lambda (name)
		 (make-idpredconst name inst-types inst-param-cterms))
	       relevant-idpc-names))
	 (pvars (map idpredconst-name-to-pvar relevant-idpc-names))
	 (cterms (map (lambda (pvar) (cadr (assoc pvar psubst))) pvars))
	 (var-lists (map cterm-to-vars cterms))
	 (relevant-inst-idpc-formulas
	  (map (lambda (idpc vars)
		 (apply make-predicate-formula
			idpc (map make-term-in-var-form vars)))
	       relevant-inst-idpcs var-lists)))
    (map (lambda (idpc-formula concl)
	   (make-imp idpc-formula concl))
	 relevant-inst-idpc-formulas (map cterm-to-formula relevant-cterms))))

(define (uniform-non-recursive-clause? formula . pvars)
  (and
   (null? (formula-to-free formula))
   (letrec
       ((impnc-param-prem-clause?
	 (lambda (formula)
	   (if (impnc-form? formula)
	       (let ((prem (impnc-form-to-premise formula))
		     (conc (impnc-form-to-conclusion formula)))
		 (and (null? (intersection (formula-to-pvars prem) pvars))
		      (impnc-param-prem-clause? conc)))
	       (and (predicate-form? formula)
		    (pair? (member (predicate-form-to-predicate formula)
				   pvars)))))))
     (impnc-param-prem-clause?
      (allnc-form-to-final-kernel formula)))))

(define (check-aconst x . opt-ignore-deco-flag)
  (if (not (aconst-form? x))
      (myerror "check-aconst" "aconst expected" x))
  (if (not (list? x))
      (myerror "check-aconst" "list expected" x))
  (if (not (<= 5 (length x)))
      (myerror "check-aconst" "list of length at least 5 expected" x))
  (let* ((name (cadr x))
	 (kind (caddr x))
	 (uninst-formula (cadddr x))
	 (tpsubst (car (cddddr x)))
	 (repro-data (cdr (cddddr x)))
	 (tsubst (if (and (list? tpsubst) (apply and-op (map pair? tpsubst)))
		     (list-transform-positive tpsubst			    
		       (lambda (x) (tvar-form? (car x))))
		     (myerror "check-aconst"
			      "tpsubst as list of pairs expected"
			      tpsubst)))
	 (psubst (list-transform-positive tpsubst
		  (lambda (x) (pvar-form? (car x)))))
	 (tvars (map car tsubst))
	 (pvars (map car psubst)))
    (if (not (string? name))
	(myerror "check-aconst" "string expected" name))
    (if (not (member kind (list 'axiom 'theorem 'global-assumption)))
	(myerror "check-aconst"
		 "kind axiom, theorem or global-assumption expected"
		 kind))
    (if (not (formula? uninst-formula))
	(myerror "check-aconst" "formula expected" uninst-formula))
    (if (not (tpsubst? tpsubst))
	(apply myerror "check-aconst" "tpsubst expected" tpsubst))
    (if (not (= (+ (length tsubst) (length psubst)) (length tpsubst)))
	(myerror "check-aconst" "tpsubst expected" tpsubst))
    (if (pair? (set-minus tvars (formula-to-tvars uninst-formula)))
	(myerror "check-aconst" "tsubst has superfluous tvars"
		 (set-minus tvars (formula-to-tvars uninst-formula))))
    (if (pair? (set-minus pvars (formula-to-pvars uninst-formula)))
	(myerror "check-aconst" "psubst has superfluous pvars"
		 (set-minus pvars (formula-to-pvars uninst-formula))))
    (if (not (admissible-substitution? ;for elim-aconst generalize idpc-args
	      tpsubst (apply mk-all (append (formula-to-free uninst-formula)
					    (list uninst-formula)))))
	(apply myerror "check-aconst" "admissible substitution expected"
	       (cons uninst-formula tpsubst)))
    (let ((ignore-deco-flag (if (pair? opt-ignore-deco-flag)
				 (car opt-ignore-deco-flag)
				 #t)))
      (if (not ignore-deco-flag)
	  (let* ((et-type (formula-to-et-type uninst-formula))
		 (et-tvars (type-to-tvars et-type))
		 (rel-pvars
		  (list-transform-positive (formula-to-pvars uninst-formula)
		    (lambda (pvar)
		      (member (PVAR-TO-TVAR pvar) et-tvars))))
		 (rel-psubst (list-transform-positive psubst
			       (lambda (p) (member (car p) rel-pvars))))
		 (violating-psubst
		  (list-transform-positive rel-psubst
		    (lambda (p)
		      (and (h-deg-zero? (pvar-to-h-deg (car p)))
			   (formula-of-nulltype?
			    (cterm-to-formula (cadr p))))))))
	    (if (and (pair? violating-psubst)
		     (not (member name (list "Ex-Intro" "Ex-Elim" "Gfp"))))
		(apply myerror "check-aconst" name
		       "computationally relevant formulas expected"
		       (map cterm-to-formula (map cadr violating-psubst)))))))
    (let ((violating-pvars
	   (list-transform-positive (formula-to-pvars uninst-formula)
	     (lambda (pvar)
	       (let ((info (assoc pvar tpsubst)))
		 (and
		  info
		  (if DIALECTICA-FLAG
		      (or (and (not (pvar-with-positive-content? pvar))
			       (not (nulltype?
				     (formula-to-etdp-type
				      (cterm-to-formula (cadr info))))))
			  (and (not (pvar-with-negative-content? pvar))
			       (not (nulltype?
				     (formula-to-etdn-type
				      (cterm-to-formula (cadr info)))))))
		      (and (not (pvar-with-positive-content? pvar))
			   (not (nulltype?
				 (formula-to-et-type
				  (cterm-to-formula (cadr info)))))))))))))
      (if (pair? violating-pvars)
	  (apply myerror
		 "check-aconst" "incorrect substitution for pvars"
		 (append pvars (list "in aconst" name)))))
    (let ((free (formula-to-free uninst-formula)))
      (if (and (or (string=? name "Elim") (string=? name "Gfp"))
	       (imp-form? uninst-formula)
	       (pair? (set-minus free (formula-to-free (imp-form-to-premise
							uninst-formula)))))
	  (apply myerror
		 "check-aconst" name "uninstantiated formula"
		 uninst-formula
		 "has unexpected free variables"
		 (set-minus free (formula-to-free
				  (imp-form-to-premise uninst-formula)))))
      (if (and (string=? name "ElimMR")
	       (all-form? uninst-formula)
	       (pair? (set-minus free (formula-to-free
				       (imp-form-to-premise
					(all-form-to-kernel
					 uninst-formula))))))
	  (apply myerror
		 "check-aconst" name "uninstantiated formula"
		 uninst-formula
		 "has unexpected free variables"
		 (set-minus free (formula-to-free
				  (imp-form-to-premise
				   (all-form-to-kernel uninst-formula))))))
      (if (and (not (or (string=? name "Elim") (string=? name "Gfp")
			(string=? name "ElimMR")))
	       (pair? free))
	  (apply myerror
		 "check-aconst" name "uninstantiated formula" uninst-formula
		 "has unexpected free variables" free)))
    (if (and (eq? kind 'axiom)
	     (not
	      (or
	       (member
		name
		'("AndAtomToLeft" "AndAtomToRight" "AtomToImp"
		  "Intro" "Elim" "Closure" "Gfp" "ElimMR"
		  "Ind" "Cases" "GInd"
		  "Ex-Intro" "Ex-Elim"
		  "Exnc-Intro" "Exnc-Elim" ;obsolete
		  "Comp" ;added 2013-12-08
		  "Truth-Axiom" ;obsolete
		  "Eq-Refl" "Eq-Sym" "Eq-Trans" "Eq-Compat" "Ext" ;obsolete
		  "Pair-Elim" "Total"
		  "InhabTotal" "InhabTotalMR"
		  "AllTotal" "AllncTotal" "ExTotal" ;obsolete
		  "ExTotalElim"
		  "ExDTotalElim" "ExLTotalElim" "ExRTotalElim" "ExUTotalElim"
		  "ExDTotal" "ExLTotal" "ExRTotal" "ExUTotal" ;obsolete
		  "AllTotalRev" "AllncTotalRev" "ExTotalRev" ;obsolete
		  "ExTotalIntro"
		  "ExDTotalIntro" "ExLTotalIntro"
		  "ExRTotalIntro" "ExUTotalIntro"
		  "ExDTotalRev" "ExLTotalRev" ;obsolete
		  "ExRTotalRev" "ExUTotalRev" ;obsolete
		  "Constr-Total" "Constr-Total-Args" ;obsolete
		  "Total-Pair" "Total-Proj"
		  "AllTotalIntro" "AllTotalElim"
		  "AllncTotalIntro" "AllncTotalElim"
		  "All-AllPartial" "AllPartial-All" ;obsolete
		  "Allnc-AllncPartial" "AllncPartial-Allnc" ;obsolete
		  "Ex-ExPartial" "ExPartial-Ex" ;obsolete
		  "Exnc-ExncPartial" "ExncPartial-Exnc" ;obsolete
		  "AtomToEqDTrue" "EqDTrueToAtom"))
	       (apply
		or-op
		(map
		 (lambda (string)
		   (and (<= (string-length string) (string-length name))
			(string=? (substring name 0 (string-length string))
				  string)))
		 '("Eq-to-=-1-"
		   "Eq-to-=-2-"
		   "=-to-Eq-"
		   "=-to-E-"
		   "=-to-E-"
		   "E-to-Total-"
		   "SE-to-E-"
		   "Total-to-E-"
		   "All-AllPartial-"
		   "Allnc-AllncPartial-"
		   "ExPartial-Ex-"
		   "ExncPartial-Exnc-" ;obsolete
		   ))))))
	(myerror "check-aconst" "axiom expected" name))
    (if (and (eq? kind 'theorem)
	     (not (assoc name THEOREMS)))
	(myerror "check-aconst" "theorem expected" name))
    (if (and (eq? kind 'global-assumption)
	     (not (assoc name GLOBAL-ASSUMPTIONS)))
	(myerror "check-aconst" "global-assumption expected" name))
    (if
     (string=? "Intro" name)
     (let ((computed-repro-data (intro-aconst-to-computed-repro-data x)))
       (if (not (= 2 (length repro-data)))
	   (myerror "check-aconst" "repro data of length 2 expected"
		    repro-data))
       (if (not (= (car repro-data) (car computed-repro-data)))
	   (myerror "check-aconst" "equal clause numbers expected"
		    (car repro-data) (car computed-repro-data)))
       (if (not (idpredconst-equal?
		 (cadr repro-data) (cadr computed-repro-data)))
	   (myerror "check-aconst" "equal idpredconsts expected"
		    (cadr repro-data) (cadr computed-repro-data))))
     (let ((computed-repro-data (aconst-to-computed-repro-data x)))
       (if (not (= (length repro-data) (length computed-repro-data)))
	   (myerror "check-aconst" "aconst with name" name "has"
		    (length repro-data) "repro-data but"
		    (length computed-repro-data)
		    "computed repro-data"))
       (for-each
	(lambda (rfla crfla)
	  (let ((test (if (not (member name (list "Elim" "Gfp" "ElimMR")))
			  (not (classical-formula=? rfla crfla))
			  (not (classical-formula=?
				(apply mk-all
				       (append (formula-to-free rfla)
					       (list rfla)))
				(apply mk-all
				       (append (formula-to-free crfla)
					       (list crfla))))))))
	    (if test (myerror "check-aconst"
			      "equal formulas expected for aconst"
			      name
			      "repro formula" rfla
			      "computed repro formula"
			      crfla))))
	repro-data computed-repro-data)))
    #t))

(define (idpredconst-equal? idpc1 idpc2)
  (or (equal? idpc1 idpc2)
      (let* ((name1 (idpredconst-to-name idpc1))
	     (types1 (idpredconst-to-types idpc1))
	     (cterms1 (idpredconst-to-cterms idpc1))
	     (name2 (idpredconst-to-name idpc2))
	     (types2 (idpredconst-to-types idpc2))
	     (cterms2 (idpredconst-to-cterms idpc2)))
	(and (string=? name1 name2)
	     (equal? types1 types2)
	     (= (length cterms1) (length cterms2))
	     (apply and-op (map (lambda (x y) (classical-cterm=? x y))
				cterms1 cterms2))))))

(define (avar-full=? avar1 avar2 . ignore-deco-flag)
  (or (eq? avar1 avar2)
      (and (avar-form? avar1) (avar-form? avar2)
	   (= (avar-to-index avar1) (avar-to-index avar2))
	   (string=? (avar-to-name avar1) (avar-to-name avar2))
	   (apply
	    classical-formula=?
	    (avar-to-formula avar1) (avar-to-formula avar2)
	    (append ignore-deco-flag)))))

;; Complete test tpsubst? for type and predicate substitutions.

(define (tpsubst? x)
  (and
   (list? x)
   (apply and-op
	  (map (lambda (item)
		 (and (list? item)
		      (= 2 (length item))
		      (or (tvar-form? (car item))
			  (pvar-form? (car item)))))
	       x))
   (= (length (remove-duplicates (map car x)))
      (length x))
   (let ((tsubst (list-transform-positive x
		   (lambda (item) (tvar-form? (car item)))))
	 (psubst (list-transform-positive x
		   (lambda (item) (pvar-form? (car item))))))
     (and
      (tsubst? tsubst)
      (apply and-op
	     (map (lambda (item)
		    (and (admissible-substitution? x (car item))
			 (not (pvar-cterm-equal? (car item) (cadr item)))))
		  psubst))))))

(define (aconst=? aconst1 aconst2)
  (and (aconst-form? aconst1) (aconst-form? aconst2)
       (string=? (aconst-to-name aconst1) (aconst-to-name aconst2))
       (eq? (aconst-to-kind aconst1) (aconst-to-kind aconst2))
       (classical-formula=? (aconst-to-formula aconst1)
			    (aconst-to-formula aconst2))))

(define (aconst-without-rules? aconst)
  (let ((name (aconst-to-name aconst))
	(kind (aconst-to-kind aconst)))
    (or
     (eq? 'theorem kind)
     (and (eq? 'global-assumption kind)
          (not (string=? "Efq" name))) ;This is a hack.
     (and
      (eq? 'axiom kind)
      (not (member
            name
            '("Ind" "Cases" "GInd" "Intro" "Elim"
	      "Ex-Intro" "Ex-Elim"
	      "Exnc-Intro" "Exnc-Elim" ;obsolete
	      )))))))

(define (aconst-to-string aconst)
  (let* ((name (aconst-to-name aconst))
	 (repro-data (aconst-to-repro-data aconst))
	 (repro-string
	  (if
	   (string=? "Intro" name)
	   (string-append " " (number-to-string (car repro-data))
			  " " (idpredconst-to-string (cadr repro-data)))
	   (apply string-append
		  (map (lambda (x) (string-append " " (formula-to-string x)))
		       repro-data)))))
    (cond
     ((string=? "Ind" name) (string-append "(Ind" repro-string ")"))
     ((string=? "Cases" name) (string-append "(Cases" repro-string ")"))
     ((string=? "Intro" name) (string-append "(Intro" repro-string ")"))
     ((string=? "Elim" name) (string-append "(Elim" repro-string ")"))
     ((string=? "Ex-Intro" name) (string-append "(Ex-Intro" repro-string ")"))
     ((string=? "Ex-Elim" name) (string-append "(Ex-Elim" repro-string ")"))
     ((string=? "Exnc-Intro" name) ;obsolete
      (string-append "(Exnc-Intro" repro-string ")"))
     ((string=? "Exnc-Elim" name) ;obsolete
      (string-append "(Exnc-Elim" repro-string ")"))
     (else name))))

;; pvar-to-cterm is superseded by the more general predicate-to-cterm.
;; It is kept temporarily for backward compatibility.

(define (pvar-to-cterm pvar) (predicate-to-cterm pvar))

(define (number-and-pconst-to-comp-aconst i pconst)
  (let* ((name (const-to-name pconst))
	 (tsubst (const-to-tsubst pconst))
	 (comprules (pconst-name-to-comprules name))
	 (comprule (list-ref comprules i))
	 (lhs (rule-to-lhs comprule))
	 (rhs (rule-to-rhs comprule))
	 (vars (term-to-free lhs))
	 (partial-vars (map (lambda (var)
			      (make-var (var-to-type var)
					(var-to-index var)
					t-deg-zero
					(var-to-name var)))
			    vars))
	 (subst (make-substitution-wrt
		 var-term-equal? vars (map make-term-in-var-form partial-vars)))
	 (uninst-eqd-formula (formula-substitute (make-eqd lhs rhs) subst)))
    (make-aconst "Comp" 'axiom
		 (apply mk-all (append partial-vars (list uninst-eqd-formula)))
		 tsubst)))

(define eq-refl-aconst ;obsolete
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (formula-of-eq-refl-aconst
	  (mk-allnc var (make-eq varterm varterm))))
    (make-aconst "Eq-Refl" 'axiom formula-of-eq-refl-aconst empty-subst)))

(define eq-sym-aconst ;obsolete
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var1 (make-var tvar 1 t-deg-zero name))
	 (var2 (make-var tvar 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (formula-of-eq-sym-aconst
	  (mk-allnc var1 var2 (mk-imp (make-eq varterm1 varterm2)
				      (make-eq varterm2 varterm1)))))
    (make-aconst "Eq-Sym" 'axiom formula-of-eq-sym-aconst empty-subst)))

(define eq-trans-aconst ;obsolete
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var1 (make-var tvar 1 t-deg-zero name))
	 (var2 (make-var tvar 2 t-deg-zero name))
	 (var3 (make-var tvar 3 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (varterm3 (make-term-in-var-form var3))
	 (formula-of-eq-trans-aconst
	  (mk-allnc var1 var2 var3 (mk-imp (make-eq varterm1 varterm2)
					   (make-eq varterm2 varterm3)
					   (make-eq varterm1 varterm3)))))
    (make-aconst "Eq-Trans" 'axiom formula-of-eq-trans-aconst empty-subst)))

(define ext-aconst ;obsolete
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (arrow-type (make-arrow tvar1 tvar2))
	 (fname (default-var-name arrow-type))
	 (fvar1 (make-var arrow-type 1 t-deg-zero fname))
	 (fvar2 (make-var arrow-type 2 t-deg-zero fname))
	 (name (default-var-name tvar1))
	 (var (make-var tvar1 -1 t-deg-zero name))
	 (fterm1 (make-term-in-app-form
		  (make-term-in-var-form fvar1)
		  (make-term-in-var-form var)))
	 (fterm2 (make-term-in-app-form
		  (make-term-in-var-form fvar2)
		  (make-term-in-var-form var)))
	 (prem-eq-fla (make-eq fterm1 fterm2))
	 (concl-eq-fla (make-eq (make-term-in-var-form fvar1)
				(make-term-in-var-form fvar2)))
	 (formula-of-ext-aconst
	  (mk-allnc fvar1 fvar2 (mk-imp (mk-allnc var prem-eq-fla)
					concl-eq-fla))))
    (make-aconst "Ext" 'axiom formula-of-ext-aconst empty-subst)))

(define eq-compat-aconst ;obsolete
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var1 (make-var tvar 1 t-deg-zero name))
	 (var2 (make-var tvar 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (eq-fla (make-eq varterm1 varterm2))
	 (fla1 (make-predicate-formula pvar varterm1))
	 (fla2 (make-predicate-formula pvar varterm2))
	 (formula-of-eq-compat-aconst
	  (mk-allnc var1 var2 (mk-imp eq-fla fla1 fla2))))
    (make-aconst "Eq-Compat" 'axiom formula-of-eq-compat-aconst empty-subst)))

(define pair-elim-aconst
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (pairtype (make-star tvar1 tvar2))
	 (pairname (default-var-name pairtype))
	 (pairvar (make-var pairtype -1 t-deg-zero pairname))
	 (pairvarterm (make-term-in-var-form pairvar))
	 (name1 (default-var-name tvar1))
	 (var1 (make-var tvar1 1 t-deg-zero name1))
	 (name2 (default-var-name tvar2))
	 (var2 (make-var tvar2 2 t-deg-zero name2))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (pairterm
	  (make-term-in-pair-form varterm1 varterm2))
	 (pvar (make-pvar (make-arity pairtype) -1 h-deg-zero n-deg-zero ""))
	 (fla1 (mk-all var1 var2
		       (make-predicate-formula pvar pairterm)))
	 (fla2 (mk-all pairvar
		       (make-predicate-formula pvar pairvarterm)))
	 (formula-of-pair-elim-aconst (mk-imp fla1 fla2)))
    (make-aconst "Pair-Elim" 'axiom formula-of-pair-elim-aconst empty-subst)))

(define (all-pair-formula-to-pair-elim-aconst all-pair-formula)
  (let* ((var (all-form-to-var all-pair-formula))
	 (kernel (all-form-to-kernel all-pair-formula))
	 (pairtype (var-to-type var))
	 (type1 (star-form-to-left-type pairtype))
	 (type2 (star-form-to-right-type pairtype))
	 (types (list type1 type2))
	 (fixed-tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (fixed-tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (fixed-pairtype (make-star fixed-tvar1 fixed-tvar2))
	 (fixed-tvars (list fixed-tvar1 fixed-tvar2))	 
	 (tsubst (make-substitution fixed-tvars types))
	 (cterm (make-cterm var kernel))
	 (fixed-pvar
	  (make-pvar (make-arity fixed-pairtype) -1 h-deg-zero n-deg-zero ""))
	 (psubst (make-subst-wrt pvar-cterm-equal? fixed-pvar cterm)))
    (make-aconst (aconst-to-name pair-elim-aconst)
		 (aconst-to-kind pair-elim-aconst)
		 (aconst-to-uninst-formula pair-elim-aconst)
		 (append tsubst psubst))))

;; The following axioms involve the obsolete Equal predconst and should
;; be removed.  However, some of them are still used in simp-with.  One
;; should first replace them there, using EqD.

(define (finalg-to-eq-to-=-1-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 t-deg-zero name))
	 (var2 (make-var finalg 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (eq-fla (make-eq varterm1 varterm2))
	 (e-fla1 (make-e varterm1))
	 (=-fla (make-= varterm1 varterm2))
	 (formula-of-eq-to-=-1-aconst 
	  (mk-allnc var1 var2 (mk-imp eq-fla e-fla1 =-fla)))
	 (aconst-name (string-append "Eq-to-=-1-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-eq-to-=-1-aconst empty-subst)))

(define (finalg-to-eq-to-=-2-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 t-deg-zero name))
	 (var2 (make-var finalg 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (eq-fla (make-eq varterm1 varterm2))
	 (e-fla2 (make-e varterm2))
	 (=-fla (make-= varterm1 varterm2))
	 (formula-of-eq-to-=-2-aconst 
	  (mk-allnc var1 var2 (mk-imp eq-fla e-fla2 =-fla)))
	 (aconst-name (string-append "Eq-to-=-2-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-eq-to-=-2-aconst empty-subst)))

(define (finalg-to-=-to-eq-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 t-deg-zero name))
	 (var2 (make-var finalg 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (=-fla (make-= varterm1 varterm2))
	 (eq-fla (make-eq varterm1 varterm2))
	 (formula-of-=-to-eq-aconst 
	  (mk-allnc var1 var2 (mk-imp =-fla eq-fla)))
	 (aconst-name (string-append "=-to-Eq-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-=-to-eq-aconst empty-subst)))
    
;; The following two axioms are almost special cases of the assertion
;; that two pconsts defined by the same computation rules are equal on
;; all (possibly partial) arguments.  Since we do not require such a
;; property, we mark these axioms as obsolete.

;; Code to be discarded later
(define (finalg-to-=-to-e-1-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 t-deg-zero name))
	 (var2 (make-var finalg 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (=-fla (make-= varterm1 varterm2))
	 (e-fla (make-e varterm1))
	 (formula-of-=-to-e-1-aconst 
	  (mk-allnc var1 var2 (mk-imp =-fla e-fla)))
	 (aconst-name (string-append "=-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-=-to-e-1-aconst empty-subst)))
    
(define (finalg-to-=-to-e-2-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 t-deg-zero name))
	 (var2 (make-var finalg 2 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (=-fla (make-= varterm1 varterm2))
	 (e-fla (make-e varterm2))
	 (formula-of-=-to-e-2-aconst 
	  (mk-allnc var1 var2 (mk-imp =-fla e-fla)))
	 (aconst-name (string-append "=-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-=-to-e-2-aconst empty-subst)))
    
;; The following table is obsolete and to be discarded.
;; Relations between totality concepts for the different types.  Because
;; of inclusions relations are inherited from alg to sfinalg to finalg.

;; finalg     | sfinalg    | alg      | rho=>sigma
;; -----------------------------------------------
;; E      | ^ |            |          |  
;; SE     | | | SE     | ^ |          |
;; STotal |   | STotal v | | STotal ^ | STotal
;; Total  v   | Total      | Total  | | Total

;; The formula allnc n^(E n^ -> TotalNat n^) has no realizer and hence
;; should not be an axiom.

;; The following axiom concerns the almost obsolete SE, and again is a
;; proposition on arbitrary (possibly partial) arguments.  Since we do
;; not require such properties, we mark the axiom as obsolete.

;; Code to be discarded later
(define (finalg-to-se-to-e-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (se-fla (make-se varterm))
	 (e-fla (make-e varterm))
	 (formula-of-se-to-e-aconst 
	  (mk-allnc var (mk-imp se-fla e-fla)))
	 (aconst-name (string-append "SE-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-se-to-e-aconst empty-subst)))

(define (finalg-to-total-to-e-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (total-fla (make-total varterm))
	 (e-fla (make-e varterm))
	 (formula-of-total-to-e-aconst 
	  (mk-allnc var (mk-imp total-fla e-fla)))
	 (aconst-name (string-append "Total-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-total-to-e-aconst empty-subst)))

;; Notice that finalg-to-total-to-e-aconst might be removed, because
;; its formula can be derived from alg-to-total-to-stotal-aconst
;; sfinalg-to-stotal-to-se-aconst and finalg-to-se-to-e-aconst .

(define (alg-to-total-to-stotal-aconst alg)
  (let* ((name (default-var-name alg))
	 (var (make-var alg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (total-fla (make-total varterm))
	 (stotal-fla (make-stotal varterm))
	 (formula-of-total-to-stotal-aconst 
	  (mk-allnc var (mk-imp total-fla stotal-fla)))
	 (aconst-name (string-append "Total-to-STotal-" (type-to-string alg))))
    (make-aconst
     aconst-name 'axiom formula-of-total-to-stotal-aconst empty-subst)))

(define alltotal-elim-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (vartotal (make-var tvar -1 t-deg-one name))
	 (var (make-var tvar -1 t-deg-zero name))
	 (vartotalterm (make-term-in-var-form vartotal))
	 (varterm (make-term-in-var-form var))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (alltotal-fla
	  (mk-all vartotal (make-predicate-formula pvar vartotalterm)))
	 (allnc-fla
	  (mk-allnc var (mk-imp (make-total varterm)
				(make-predicate-formula pvar varterm))))
	 (formula-of-alltotal-elim-aconst (mk-imp alltotal-fla allnc-fla)))
    (make-aconst "AllTotalElim"
		 'axiom formula-of-alltotal-elim-aconst empty-subst)))

;; (pp (aconst-to-formula alltotal-elim-aconst))
;; all alpha (Pvar alpha)alpha -> 
;; allnc alpha^(Total alpha^ -> (Pvar alpha)alpha^)

;; Obsolete (but kept for backwards compatability)

(define all-allpartial-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (all-fla (mk-all var (make-predicate-formula pvar varterm)))
	 (allpartial-fla
	  (mk-allnc varpartial
		    (mk-imp (make-total varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-all-allpartial-aconst (mk-imp all-fla allpartial-fla)))
    (make-aconst "All-AllPartial"
		 'axiom formula-of-all-allpartial-aconst empty-subst)))

;; (pp (aconst-to-formula all-allpartial-aconst))
;; all x (Pvar alpha)x -> allnc x^(Total x^ -> (Pvar alpha)x^)

(define alltotal-intro-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (vartotal (make-var tvar -1 t-deg-one name))
	 (var (make-var tvar -1 t-deg-zero name))
	 (vartotalterm (make-term-in-var-form vartotal))
	 (varterm (make-term-in-var-form var))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (alltotal-fla
	  (mk-all vartotal (make-predicate-formula pvar vartotalterm)))
	 (allnc-fla
	  (mk-allnc var (mk-imp (make-total varterm)
				(make-predicate-formula pvar varterm))))
	 (formula-of-alltotal-intro-aconst (mk-imp allnc-fla alltotal-fla)))
    (make-aconst "AllTotalIntro"
		 'axiom formula-of-alltotal-intro-aconst empty-subst)))

;; (pp (aconst-to-formula alltotal-intro-aconst))
;; allnc alpha^(Total alpha^ -> (Pvar alpha)alpha^) -> 
;; all alpha (Pvar alpha)alpha

;; Obsolete (but kept for backwards compatability)

(define allpartial-all-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (all-fla (mk-all var (make-predicate-formula pvar varterm)))
	 (allpartial-fla
	  (mk-allnc varpartial
		    (mk-imp (make-total varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-allpartial-all-aconst (mk-imp allpartial-fla all-fla)))
    (make-aconst "AllPartial-All"
		 'axiom formula-of-allpartial-all-aconst empty-subst)))

;; (pp (aconst-to-formula allpartial-all-aconst))
;; allnc x^(Total x^ -> (Pvar alpha)x^) -> all x (Pvar alpha)x

(define allnctotal-intro-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (vartotal (make-var tvar -1 t-deg-one name))
	 (var (make-var tvar -1 t-deg-zero name))
	 (vartotalterm (make-term-in-var-form vartotal))
	 (varterm (make-term-in-var-form var))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (allnctotal-fla
	  (mk-allnc vartotal (make-predicate-formula pvar vartotalterm)))
	 (allnc-impnc-fla
	  (mk-allnc var (mk-impnc (make-total varterm)
				  (make-predicate-formula pvar varterm))))
	 (formula-of-allnctotal-intro-aconst
	  (mk-imp allnc-impnc-fla allnctotal-fla)))
    (make-aconst "AllncTotalIntro"
		 'axiom formula-of-allnctotal-intro-aconst empty-subst)))

;; (pp (aconst-to-formula allnctotal-intro-aconst))
;; allnc alpha^(Total alpha^ --> (Pvar alpha)alpha^) -> 
;; allnc alpha (Pvar alpha)alpha

(define allnctotal-elim-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (vartotal (make-var tvar -1 t-deg-one name))
	 (var (make-var tvar -1 t-deg-zero name))
	 (vartotalterm (make-term-in-var-form vartotal))
	 (varterm (make-term-in-var-form var))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (allnctotal-fla
	  (mk-allnc vartotal (make-predicate-formula pvar vartotalterm)))
	 (allnc-impnc-fla
	  (mk-allnc var (mk-impnc (make-total varterm)
				  (make-predicate-formula pvar varterm))))
	 (formula-of-allnctotal-elim-aconst
	  (mk-imp allnctotal-fla allnc-impnc-fla)))
    (make-aconst "AllncTotalElim"
		 'axiom formula-of-allnctotal-elim-aconst empty-subst)))

;; (pp (aconst-to-formula allnctotal-elim-aconst))
;; allnc alpha (Pvar alpha)alpha -> 
;; allnc alpha^(Total alpha^ --> (Pvar alpha)alpha^)

(define (finalg-to-all-allpartial-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (all-fla (mk-all var (make-predicate-formula pvar varterm)))
	 (allpartial-fla
	  (mk-all varpartial
		    (mk-imp (make-e varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-all-allpartial-aconst
	  (mk-imp all-fla allpartial-fla))
	 (name (string-append "All-AllPartial-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-all-allpartial-aconst empty-subst)))

(define (finalg-to-allnc-allncpartial-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (allnc-fla (mk-allnc var (make-predicate-formula pvar varterm)))
	 (allncpartial-fla
	  (mk-allnc varpartial
		    (mk-imp (make-e varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-allnc-allncpartial-aconst
	  (mk-imp allnc-fla allncpartial-fla))
	 (name (string-append "Allnc-AllncPartial-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-allnc-allncpartial-aconst
		 empty-subst)))

(define (finalg-to-expartial-ex-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (ex-fla (mk-ex var (make-predicate-formula pvar varterm)))
	 (expartial-fla
	  (mk-ex varpartial
		 (mk-and (make-e varpartialterm)
			 (make-predicate-formula pvar varpartialterm))))
	 (formula-of-expartial-ex-aconst (mk-imp expartial-fla ex-fla))
	 (name (string-append "ExPartial-Ex-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-expartial-ex-aconst empty-subst)))

(define (finalg-to-exncpartial-exnc-aconst finalg) ;obsolete
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 t-deg-zero name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (exnc-fla (mk-exnc var (make-predicate-formula pvar varterm)))
	 (exncpartial-fla
	  (mk-exnc varpartial
		   (mk-and (make-e varpartialterm)
			   (make-predicate-formula pvar varpartialterm))))
	 (formula-of-exncpartial-exnc-aconst (mk-imp exncpartial-fla exnc-fla))
	 (name (string-append "ExncPartial-Exnc-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-exncpartial-exnc-aconst empty-subst)))

;; Now for induction.  We define a procedure that takes all-formulas
;; and returns the corresponding induction axiom.

(define (all-formulas-to-ind-aconst . all-formulas)
  (if (nested-alg-name?
       (alg-form-to-name (var-to-type (all-form-to-var (car all-formulas)))))
      (myerror "all-formulas-to-ind-aconst"
	       "all-formula for an unnested algebra expected"
	       (car all-formulas)
	       "unfold all-formula and use imp-formulas-to-elim-aconst"))
  (let* ((uninst-imp-formula-and-tpsubst
	  (apply all-formulas-to-uninst-imp-formula-and-tpsubst all-formulas))
	 (uninst-imp-formula (car uninst-imp-formula-and-tpsubst))
	 (tpsubst (cadr uninst-imp-formula-and-tpsubst)))
    (apply make-aconst "Ind" 'axiom uninst-imp-formula tpsubst all-formulas)))

(define (all-formulas-to-uninst-imp-formulas-and-tpsubst . all-formulas)
  (if
   (null? all-formulas)
   (list '() empty-subst)
   (let* ((free (apply union (map formula-to-free all-formulas)))
	  (vars (map all-form-to-var all-formulas))
	  (totality-test
	   (if (not (apply and-op (map t-deg-one? (map var-to-t-deg vars))))
	       (apply myerror
		      "all-formulas-to-uninst-imp-formulas-and-tpsubst"
		      "total variables expected in all formulas"
		      all-formulas)))
	  (kernels (map all-form-to-kernel all-formulas))
	  (types (map var-to-type vars))
	  (alg-names
	   (map (lambda (type)
		  (if (alg-form? type)
		      (alg-form-to-name type)
		      (myerror
		       "all-formulas-to-uninst-imp-formulas-and-tpsubst"
		       "alg expected" type)))
		types))
	  (tparam-lists (map alg-form-to-types types))
	  (all-formula (car all-formulas))
	  (type (car types))
	  (alg-name (car alg-names))
	  (orig-tvars (alg-name-to-tvars alg-name))
	  (tvars (map (lambda (x) (new-tvar)) orig-tvars))
	  (tparams (car tparam-lists))
	  (tsubst (make-substitution tvars tparams))
	  (uninst-types (map (lambda (x) (apply make-alg (cons x tvars)))
			     alg-names)) 
	  (uninst-arities (map (lambda (x) (make-arity x)) uninst-types))
	  (cterms (map (lambda (x y) (make-cterm x y)) vars kernels))
	  (psubst (map (lambda (x y) (list (arity-to-new-general-pvar x) y))
		       uninst-arities cterms))
	  (pvars (map car psubst))
	  (uninst-vars (map (lambda (x y) (type-to-new-var x y))
			    uninst-types vars))
	  (uninst-all-formulas
	   (map (lambda (x pvar)
		  (make-all x (make-predicate-formula
			       pvar (make-term-in-var-form x))))
		uninst-vars pvars))
	  (uninst-kernel-formulas
	   (map (lambda (x pvar)
                  (make-predicate-formula
                   pvar (make-term-in-var-form x)))
                uninst-vars pvars))
	  (alg-names-with-uninst-all-formulas
	   (map list alg-names uninst-all-formulas))
	  (simalg-names (alg-name-to-simalg-names alg-name)))
     (if (not (equal? alg-names (remove-duplicates alg-names)))
	 (myerror "all-formulas-to-uninst-imp-formulas-and-tpsubst"
		  "distinct algs expected" alg-names))
     (if (pair? (set-minus alg-names simalg-names))
	 (myerror "all-formulas-to-uninst-imp-formulas-and-tpsubst"
		  "too many alg names" (set-minus alg-names simalg-names)))
     (if (< 1 (length (remove-duplicates tparam-lists)))
	 (myerror "all-formulas-to-uninst-imp-formulas-and-tpsubst"
		  "lists expected" tparam-lists))
     (let* ((relevant-simalg-names (list-transform-positive simalg-names
				     (lambda (x) (member x alg-names))))
	    (orig-typed-constr-names
	     (apply append (map alg-name-to-typed-constr-names
				relevant-simalg-names)))
	    (renaming-tsubst (make-substitution orig-tvars tvars))
	    (typed-constr-names
	     (map (lambda (x)
		    (list (car x) (type-substitute (cadr x) renaming-tsubst)))
		  orig-typed-constr-names))
	    (uninst-step-formulas
	     (map (lambda (x) (typed-constr-name-to-step-formula
			       x alg-names-with-uninst-all-formulas
			       renaming-tsubst))
		  typed-constr-names))
            (uninst-imp-formulas
             (map (lambda (uninst-var uninst-kernel-formula)
                    (make-all uninst-var
                              (apply mk-imp
                                     (append uninst-step-formulas
                                             (list uninst-kernel-formula)))))
                  uninst-vars uninst-kernel-formulas)))
       (list uninst-imp-formulas (append tsubst psubst))))))

(define (all-formulas-to-uninst-imp-formula-and-tpsubst . all-formulas)
  (let* ((uninst-imp-formulas-and-tpsubst
	  (apply all-formulas-to-uninst-imp-formulas-and-tpsubst all-formulas))
	 (uninst-imp-formulas (car uninst-imp-formulas-and-tpsubst))
	 (tpsubst (cadr uninst-imp-formulas-and-tpsubst)))
    (list (car uninst-imp-formulas) tpsubst)))

(define (typed-constr-name-to-step-formula
	 typed-constr-name alg-names-with-all-formulas renaming-tsubst)
  (let* ((constr-name (typed-constr-name-to-name typed-constr-name))
	 (type (typed-constr-name-to-type typed-constr-name))
	 (argtypes (arrow-form-to-arg-types type))
	 (alg-name (alg-form-to-name (arrow-form-to-final-val-type type)))
	 (rel-simalg-names (map car alg-names-with-all-formulas))
	 (simalg-names (alg-name-to-simalg-names alg-name))
	 (irrel-simalg-names (set-minus simalg-names rel-simalg-names))
	 (all-formula (cadr (assoc alg-name alg-names-with-all-formulas)))
	 (var (all-form-to-var all-formula))
	 (kernel (all-form-to-kernel all-formula))
	 (argvars (map type-to-new-var argtypes))
	 (constr (const-substitute (constr-name-to-constr constr-name)
				   renaming-tsubst #t))
	 (constr-app-term (apply mk-term-in-app-form
				 (make-term-in-const-form constr)
				 (map make-term-in-var-form argvars)))
	 (concl-of-step (formula-subst kernel var constr-app-term))
	 (arg-pdfla-lists
	  (map
	   (lambda (argvar argtype)
	     (cond ;argtype has irrelevant alg names: no stepformula no arg
	      ((pair? (intersection (type-to-alg-names argtype)
				    irrel-simalg-names))
	       '())
	      ((and ;unnested argtype: duplication
		(alg-form? (arrow-form-to-final-val-type argtype))
		(member (alg-form-to-name
			 (arrow-form-to-final-val-type argtype))
			rel-simalg-names))
	       (list
		argvar
		(let* ((argargtypes (arrow-form-to-arg-types argtype))
		       (argargvars (map type-to-new-var argargtypes))
		       (argvaltype (arrow-form-to-final-val-type argtype))
		       (argvaltype-name (if (alg-form? argvaltype)
					    (alg-form-to-name argvaltype)
					    ""))		       
		       (hyp-all-formula
			(cadr (assoc argvaltype-name
				     alg-names-with-all-formulas)))
		       (hyp-var (all-form-to-var hyp-all-formula))
		       (hyp-kernel (all-form-to-kernel hyp-all-formula))
		       (app-term (apply mk-term-in-app-form
					(make-term-in-var-form argvar)
					(map make-term-in-var-form
					     argargvars)))
		       (hyp-formula
			(formula-subst hyp-kernel hyp-var app-term)))
		  (apply mk-all (append argargvars (list hyp-formula))))))
	      (;argvaltype not with a rel-simalg-name.  Then take argvar
	       (let* ((argvaltype (arrow-form-to-final-val-type argtype))
		      (argvaltype-name (if (alg-form? argvaltype)
					   (alg-form-to-name argvaltype)
					   "")))
		 (not (member argvaltype-name rel-simalg-names)))
	       (list argvar))
	      (else ;nested argtype
	       (myerror "typed-constr-name-to-step-formula"
			"not implemented for nested argtype"
			argtype
			"Use elimination instead"))))
	   argvars argtypes)))
    (do ((l (reverse arg-pdfla-lists) (cdr l))
	 (res concl-of-step
	      (let ((arg-pdfla-list (car l)))
		(cond
		 ((null? arg-pdfla-list)
		  res)
		 ((= 1 (length arg-pdfla-list))
		  (make-all (car arg-pdfla-list) res))
		 ((= 2 (length arg-pdfla-list))
		  (let* ((argvar (caar l))
			 (pdfla (cadar l)))
		    (make-all argvar (make-imp pdfla res))))
		 (else (myerror "typed-constr-name-to-step-formula"
				"arg-pdfla-list expected"
				arg-pdfla-list))))))
	((null? l) res))))

;; We define a procedure that takes an all-formula and returns the
;; corresponding cases axiom.

(define (all-formula-to-cases-aconst all-formula)
  (if (nested-alg-name?
       (alg-form-to-name (var-to-type (all-form-to-var all-formula))))
      (myerror "all-formula-to-cases-aconst"
	       "all-formula for an unnested algebra expected"
	       all-formula
	       "unfold all-formula"))
  (let* ((uninst-imp-formula-and-tpsubst
	  (all-formula-to-uninst-cases-imp-formula-and-tpsubst all-formula))
	 (uninst-imp-formula (car uninst-imp-formula-and-tpsubst))
	 (tpsubst (cadr uninst-imp-formula-and-tpsubst)))
    (make-aconst "Cases" 'axiom uninst-imp-formula tpsubst all-formula)))

(define (all-formula-to-uninst-cases-imp-formula-and-tpsubst all-formula)
  (let* ((free (formula-to-free all-formula))
	 (var (all-form-to-var all-formula))
	 (partial-flag (t-deg-zero? (var-to-t-deg var)))
	 (kernel (if partial-flag
		     (imp-form-to-conclusion
		      (all-form-to-kernel all-formula))
		     (all-form-to-kernel all-formula)))
	 (type (var-to-type var))
	 (alg-name (if (alg-form? type)
		       (alg-form-to-name type)
		       (myerror
			"all-formula-to-uninst-cases-imp-formula-and-tpsubst"
			"alg expected" type)))
	 (orig-tvars (alg-name-to-tvars alg-name))
	 (tvars (map (lambda (x) (new-tvar)) orig-tvars))
	 (tparams (alg-form-to-types type))
	 (tsubst (make-substitution tvars tparams))
	 (uninst-type (apply make-alg alg-name tvars))
	 (uninst-arity (make-arity uninst-type))
	 (cterm (make-cterm var kernel))
	 (psubst (list (list (arity-to-new-general-pvar uninst-arity) cterm)))
	 (pvar (caar psubst))
	 (uninst-var (type-to-new-var uninst-type var))
	 (uninst-stotal-prem
	  (if partial-flag
	      (make-stotal (make-term-in-var-form uninst-var))
	      #f))
	 (uninst-all-formula
	  (make-all uninst-var
		    (if partial-flag
			(make-imp uninst-stotal-prem
				  (make-predicate-formula
				   pvar (make-term-in-var-form uninst-var)))
			(make-predicate-formula
			 pvar (make-term-in-var-form uninst-var)))))
	 (uninst-kernel-formula
          (make-predicate-formula pvar (make-term-in-var-form uninst-var)))
	 (orig-typed-constr-names (alg-name-to-typed-constr-names alg-name))
	 (renaming-tsubst (make-substitution orig-tvars tvars))
	 (typed-constr-names
	  (map (lambda (x)
		 (list (car x) (type-substitute (cadr x) renaming-tsubst)))
	       orig-typed-constr-names))
	 (uninst-step-formulas
	  (map (lambda (x) (typed-constr-name-to-cases-step-formula
			    x uninst-all-formula renaming-tsubst))
	       typed-constr-names))
	 (uninst-imp-formula
	  (make-all uninst-var
                    (apply mk-imp (append
                                   (if partial-flag
                                       (list uninst-stotal-prem)
                                       '())
                                   uninst-step-formulas
                                   (list uninst-kernel-formula))))))
    (list uninst-imp-formula (append tsubst psubst))))

(define (typed-constr-name-to-cases-step-formula
	 typed-constr-name all-formula renaming-tsubst)
  (let* ((constr-name (typed-constr-name-to-name typed-constr-name))
	 (type (typed-constr-name-to-type typed-constr-name))
	 (alg-name (alg-form-to-name (arrow-form-to-final-val-type type)))
	 (var (all-form-to-var all-formula))
	 (partial-flag (t-deg-zero? (var-to-t-deg var)))
	 (kernel
	  (if partial-flag
	      (imp-form-to-conclusion (all-form-to-kernel all-formula))
	      (all-form-to-kernel all-formula)))
	 (argtypes (arrow-form-to-arg-types type))
	 (orig-tvars (alg-name-to-tvars alg-name))
	 (subst-tvars (map (lambda (type)
			     (type-substitute type renaming-tsubst))
			   orig-tvars))
	 (argvars (if (and partial-flag (not (finalg? (var-to-type var))))
		      (map type-to-new-partial-var argtypes)
		      (map type-to-new-var argtypes)))
	 (constr (const-substitute (constr-name-to-constr constr-name)
				   renaming-tsubst #t))
	 (constr-app-term
	  (apply mk-term-in-app-form
		 (make-term-in-const-form constr)
		 (map make-term-in-var-form argvars)))
	 (concl-of-step (formula-subst kernel var constr-app-term))
	 (non-param-argvars
	  (list-transform-positive argvars
	    (lambda (var)
	      (not (member (arrow-form-to-final-val-type (var-to-type var))
			   subst-tvars)))))
	 (stotal-formulas ;as many as there are non-param-argvars
	  (if
	   (and partial-flag (not (finalg? (var-to-type var))))
	   (map (lambda (argvar)
		  (let* ((argtype (var-to-type argvar))
			 (argargtypes (arrow-form-to-arg-types argtype))
			 (argargvars (map type-to-new-var argargtypes))
			 (argvaltype (arrow-form-to-final-val-type argtype))
			 (app-term (apply mk-term-in-app-form
					  (make-term-in-var-form argvar)
					  (map make-term-in-var-form
					       argargvars))))
		    (apply mk-all (append argargvars
					  (list (make-stotal app-term))))))
		non-param-argvars)
	   '())))
    (apply mk-all
	   (append argvars
		   (list (apply mk-imp (append stotal-formulas
					       (list concl-of-step))))))))

;; GInd: all h,x(all x(all y(hy<hx -> Ry) -> Rx) -> all p(p -> Rx))
;; with h a measure function of type alpha1 => ... => alphan => nat.

(define (all-formula-to-uninst-gind-formula-and-tpsubst all-formula n)
  (let* ((h (make-fixed-measure-var n))
	 (x (make-fixed-vars 1 n))
	 (y (make-fixed-vars 2 n))
	 (R (make-fixed-pvar n))
	 (Rx (apply make-predicate-formula R (map make-term-in-var-form x)))
	 (Ry (apply make-predicate-formula R (map make-term-in-var-form y)))
	 (hx (apply mk-term-in-app-form
		    (make-term-in-var-form h)
		    (map make-term-in-var-form x)))
	 (hy (apply mk-term-in-app-form
		    (make-term-in-var-form h)
		    (map make-term-in-var-form y)))
	 (hy<hx (make-atomic-formula
		 (mk-term-in-app-form (make-term-in-const-form
				       (pconst-name-to-pconst "NatLt"))
				      hy hx)))
	 (prog-fla ;all x(all y(hy<hx -> Ry) -> Rx)
	  (apply
	   mk-all
	   (append
	    x (list (make-imp
		     (apply mk-all (append y (list (make-imp hy<hx Ry))))
		     Rx)))))
	 (boolevar (make-var (py "boole") -1 1 "")) ;p
	 (booleatom (make-atomic-formula (make-term-in-var-form boolevar)))
	 (concl (make-all boolevar (make-imp booleatom Rx)))
         (uninst-gind-formula
          (apply mk-all h (append x (list (make-imp prog-fla concl)))))
	 (free (formula-to-free all-formula))
         (vars (all-form-to-vars all-formula n))
         (kernel (all-form-to-final-kernel all-formula n))
         (types (map var-to-type vars))
	 (tsubst (make-substitution (map var-to-type x) types))
         (cterm (apply make-cterm (append vars (list kernel))))
         (psubst (list (list R cterm))))
    (if (apply and-op
               (map (lambda (x) (t-deg-one? (var-to-t-deg x))) vars))
        (list uninst-gind-formula (append tsubst psubst))
        (myerror "all-formula-to-uninst-gind-formula-and-tpsubst"
                 "total variables expected" vars))))

;; all-formula-and-number-to-gind-aconst takes an all-formula, a number
;; n for the arity of the measure function and an optional argument for
;; the name of a theorem proving gind from induction.  If
;; opt-gindthmname is not present, gind is viewed as an axiom (and grec
;; will be extracted).  Otherwise gind is viewed as proved from ind
;; (and rec is extracted):

;;                                    NatLtLtSuccTrans hy hx k v:hy<hx w:hx<Sk
;;                                    ----------------------------------------
;;                           IH  y                            hy<k
;;                           -------------------------------------
;;                                                   Ry
;;                                      ---------------------------
;;          Efq:bot->Rx u:hx<0          Prog^h  x  all y(hy<hx->Ry)
;;          ------------------          ---------------------------
;;                    Rx                           Rx
;;             ---------------    ----------------------------------------
;; Ind h S(hx) all x(hx<0->Rx)    all k(all x(hx<k->Rx)->all x(hx<Sk->Rx))
;; -----------------------------------------------------------------------
;;                          all x(hx<S(hx)->Rx)                             x T
;;                          ---------------------------------------------------
;;                                                     Rx
;;                                      ---------------------------------
;;                                      all h,x(Prog^h -> all p(p -> Rx))

(define (all-formula-and-number-to-gind-aconst all-formula n . opt-gindthmname)
  (let* ((uninst-gind-formula-and-tpsubst
	  (all-formula-to-uninst-gind-formula-and-tpsubst all-formula n))
	 (uninst-gind-formula (car uninst-gind-formula-and-tpsubst))
	 (tpsubst (cadr uninst-gind-formula-and-tpsubst)))
    (if
     (null? opt-gindthmname) ;use gind as axiom
     (make-aconst "GInd" 'axiom uninst-gind-formula tpsubst all-formula)
     (let* ((gind-name (string-append "GInd" (number-to-alphabetic-string n)))
	    (info (assoc gind-name THEOREMS)))
       (if (not (and (string? (car opt-gindthmname))
		     (string=? (car opt-gindthmname) gind-name)))
	   (begin (comment "warning: for clarity use the gind-name")
		  (comment gind-name)
		  (comment "rather than")
		  (comment (car opt-gindthmname))))
       (if
	info
	(let ((aconst (theorem-name-to-aconst gind-name)))
	  (make-aconst (aconst-to-name aconst)
		       (aconst-to-kind aconst)
		       (aconst-to-uninst-formula aconst)
		       tpsubst
		       all-formula))
	(let* ((h-and-x (all-form-to-vars uninst-gind-formula))
	       (h (car h-and-x))
	       (x (cdr h-and-x))
	       (kernel (all-form-to-final-kernel uninst-gind-formula))
	       (boolevar (all-form-to-var (imp-form-to-conclusion kernel)))
	       (booleatom
		(make-atomic-formula (make-term-in-var-form boolevar)))
	       (prog-fla (imp-form-to-premise kernel))
	       (prog-kernel  (all-form-to-final-kernel prog-fla))
	       (y (all-form-to-vars (imp-form-to-premise prog-kernel)))
	       (Rx (imp-form-to-conclusion prog-kernel))
	       (hx (apply mk-term-in-app-form
			  (make-term-in-var-form h)
			  (map make-term-in-var-form x)))
	       (hy (apply mk-term-in-app-form
			  (make-term-in-var-form h)
			  (map make-term-in-var-form y)))
	       (k (make-var (py "nat") -1 1 ""))
	       (hx<0 (make-atomic-formula
		      (mk-term-in-app-form (make-term-in-const-form
					    (pconst-name-to-pconst "NatLt"))
					   hx (pt "Zero"))))
	       (hx<k (make-atomic-formula
		      (mk-term-in-app-form (make-term-in-const-form
					    (pconst-name-to-pconst "NatLt"))
					   hx (make-term-in-var-form k))))
	       (hy<hx (make-atomic-formula
		       (mk-term-in-app-form (make-term-in-const-form
					     (pconst-name-to-pconst "NatLt"))
					    hy hx)))
	       (hx<Sk (make-atomic-formula
		       (mk-term-in-app-form (make-term-in-const-form
					     (pconst-name-to-pconst "NatLt"))
					    hx (make-term-in-app-form
						(pt "Succ")
						(make-term-in-var-form k)))))
	       (IH-fla ;all x(hx<k->Rx)
		(apply mk-all (append x (list (make-imp hx<k Rx)))))
	       (ind-fla (make-all k IH-fla))
	       (u (formula-to-new-avar hx<0))
	       (v (formula-to-new-avar hy<hx))
	       (w (formula-to-new-avar hx<Sk))
	       (IH (formula-to-new-avar IH-fla))
	       (prog (formula-to-new-avar prog-fla))
	       (udummy (formula-to-new-avar booleatom))
	       (efq (proof-of-efq-at Rx))
	       (proof
		(apply
		 mk-proof-in-intro-form
		 (append
		  (list h)
		  x
		  (list
		   prog
		   (make-proof-in-all-intro-form
		    boolevar ;p
		    (make-proof-in-imp-intro-form
		     udummy
		     (apply
		      mk-proof-in-elim-form
		      (append
		       (list
			(make-proof-in-aconst-form
			 (all-formulas-to-ind-aconst ind-fla))
			(make-term-in-var-form h)
			(make-term-in-app-form (pt "Succ") hx)
			(apply ;base
			 mk-proof-in-intro-form
			 (append
			  x (list u (mk-proof-in-elim-form
				     efq (make-proof-in-avar-form u)))))
			(apply ;step
			 mk-proof-in-intro-form
			 (append
			  (list k IH)
			  x (list
			     w (apply
				mk-proof-in-elim-form
				(append
				 (list (make-proof-in-avar-form prog))
				 (map make-term-in-var-form x)
				 (list
				  (apply
				   mk-proof-in-intro-form
				   (append
				    y (list
				       v (apply
					  mk-proof-in-elim-form
					  (append
					   (list (make-proof-in-avar-form IH))
					   (map make-term-in-var-form y)
					   (list
					    (mk-proof-in-elim-form
					     (make-proof-in-aconst-form
					      (theorem-name-to-aconst
					       "NatLtLtSuccTrans"))
					     hy hx
					     (make-term-in-var-form k)
					     (make-proof-in-avar-form v)
					     (make-proof-in-avar-form
					      w)))))))))))))))
		       (map make-term-in-var-form x)
		       (list (make-proof-in-aconst-form truth-aconst))))))))))
	       (aconst (begin (set! OLD-COMMENT-FLAG COMMENT-FLAG)
			      (set! COMMENT-FLAG #f)
			      (add-theorem gind-name proof)
			      (set! COMMENT-FLAG OLD-COMMENT-FLAG)
			      (theorem-name-to-aconst gind-name))))
	  (make-aconst (aconst-to-name aconst)
		       (aconst-to-kind aconst)
		       (aconst-to-uninst-formula aconst)
		       tpsubst
		       all-formula)))))))

(define (formula-to-efq-aconst formula)
  (let* ((efqaconst (global-assumption-name-to-aconst "Efq"))
         (uninst-efq-formula (aconst-to-uninst-formula efqaconst))
         (pvar (predicate-form-to-predicate
                (imp-form-to-conclusion uninst-efq-formula)))
         (cterm (make-cterm formula))
         (psubst (make-subst-wrt pvar-cterm-equal? pvar cterm)))
    (make-aconst "Efq" 'global-assumption uninst-efq-formula psubst)))

;; Now the introduction and elimination axioms for the existential quantifier.

;; We define a procedure that takes an existential formula and returns the
;; corresponding existence introduction axiom:
;; ex-intro: all zs,z(A -> ex z A)

(define (ex-formula-to-ex-intro-aconst ex-formula)
  (let* ((var (ex-form-to-var ex-formula))
	 (kernel (ex-form-to-kernel ex-formula))
	 (cterm (make-cterm var kernel))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity (make-arity tvar))
	 (pvar (if (nulltype? (cterm-to-formula cterm))
		   (arity-to-new-pvar arity)
		   (arity-to-new-general-pvar arity)))
	 (predicate-formula
	  (make-predicate-formula pvar (make-term-in-var-form new-var)))
	 (imp-formula (make-imp predicate-formula
				(make-ex new-var predicate-formula)))
	 (uninst-ex-intro-formula (make-all new-var imp-formula))
	 (tsubst (make-subst tvar type))
	 (psubst (make-subst-wrt pvar-cterm-equal? pvar cterm)))
    (make-aconst
     "Ex-Intro" 'axiom uninst-ex-intro-formula (append tsubst psubst)
     ex-formula)))

;; We define a procedure that takes an existential formula and a
;; conclusion, and returns the corresponding existence elimination axiom:
;; ex-elim: allnc zs(ex z A -> all z(A -> B) -> B)

(define (ex-formula-and-concl-to-ex-elim-aconst ex-formula concl)
  (let* ((var (ex-form-to-var ex-formula))
	 (kernel (ex-form-to-kernel ex-formula))
	 (cterm1 (make-cterm var kernel))
	 (cterm2 (make-cterm concl))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity1 (make-arity tvar))
	 (pvar1 (if (nulltype? (cterm-to-formula cterm1))
		    (arity-to-new-pvar arity1)
		    (arity-to-new-general-pvar arity1)))
	 (predicate-formula1
	  (make-predicate-formula pvar1 (make-term-in-var-form new-var)))
	 (arity2 (make-arity))
	 (pvar2 (if (nulltype? (cterm-to-formula cterm2))
		    (arity-to-new-pvar arity2)
		    (arity-to-new-general-pvar arity2)))
	 (predicate-formula2 (make-predicate-formula pvar2))
	 (imp-formula
	  (mk-imp
	   (make-ex new-var predicate-formula1)
	   (make-all new-var (make-imp predicate-formula1 predicate-formula2))
	   predicate-formula2))
	 (tsubst (make-subst tvar type))
	 (psubst (make-substitution-wrt pvar-cterm-equal?
					(list pvar1 pvar2)
					(list cterm1 cterm2))))
    (make-aconst "Ex-Elim" 'axiom imp-formula (append tsubst psubst)
		 ex-formula concl)))

;; Now the introduction and elimination axioms for the exnc quantifier.

;; We define a procedure that takes an exnc formula and returns the
;; corresponding existence introduction axiom:
;; exnc-intro: allnc zs,z(A -> exnc z A)

(define (exnc-formula-to-exnc-intro-aconst exnc-formula) ;obsolete
  (let* ((var (exnc-form-to-var exnc-formula))
	 (kernel (exnc-form-to-kernel exnc-formula))
	 (cterm (make-cterm var kernel))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity (make-arity tvar))
	 (pvar (if (nulltype? (cterm-to-formula cterm))
		   (arity-to-new-pvar arity)
		   (arity-to-new-general-pvar arity)))
	 (predicate-formula
	  (make-predicate-formula pvar (make-term-in-var-form new-var)))
	 (imp-formula (make-imp predicate-formula
				(make-exnc new-var predicate-formula)))
	 (uninst-exnc-intro-formula (make-allnc new-var imp-formula))
	 (tsubst (make-subst tvar type))
	 (psubst (make-subst-wrt pvar-cterm-equal? pvar cterm)))
    (make-aconst
     "Exnc-Intro" 'axiom uninst-exnc-intro-formula (append tsubst psubst)
     exnc-formula)))

;; We define a procedure that takes an exnc formula and a
;; conclusion, and returns the corresponding exnc elimination axiom:
;; exnc-elim: allnc zs(exnc z A -> allnc z(A -> B) -> B)

(define (exnc-formula-and-concl-to-exnc-elim-aconst ;obsolete
	 exnc-formula concl)
  (let* ((var (exnc-form-to-var exnc-formula))
	 (kernel (exnc-form-to-kernel exnc-formula))
	 (cterm1 (make-cterm var kernel))
	 (cterm2 (make-cterm concl))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity1 (make-arity tvar))
	 (pvar1 (if (nulltype? (cterm-to-formula cterm1))
		    (arity-to-new-pvar arity1)
		    (arity-to-new-general-pvar arity1)))
	 (predicate-formula1
	  (make-predicate-formula pvar1 (make-term-in-var-form new-var)))
	 (arity2 (make-arity))
	 (pvar2 (if (nulltype? (cterm-to-formula cterm2))
		    (arity-to-new-pvar arity2)
		    (arity-to-new-general-pvar arity2)))
	 (predicate-formula2 (make-predicate-formula pvar2))
	 (imp-formula
	  (mk-imp
	   (make-exnc new-var predicate-formula1)
	   (make-allnc new-var
		       (make-imp predicate-formula1 predicate-formula2))
	   predicate-formula2))
	 (tsubst (make-subst tvar type))
	 (psubst (make-substitution-wrt pvar-cterm-equal?
					(list pvar1 pvar2)
					(list cterm1 cterm2))))
    (make-aconst "Exnc-Elim" 'axiom imp-formula (append tsubst psubst)
		 exnc-formula concl)))

;; all boole^1,boole^2(boole^1 andb boole^2 -> boole^1) etc can be
;; proved in TCF+ from the computation rules for AndConst .

(define and-atom-to-left-aconst
  (let* ((name (default-var-name (make-alg "boole")))
	 (var1 (make-var (make-alg "boole") 1 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (var2 (make-var (make-alg "boole") 2 t-deg-zero name))
	 (varterm2 (make-term-in-var-form var2))
	 (and-term
	  (mk-term-in-app-form
	   (make-term-in-const-form and-const) varterm1 varterm2))
	 (and-formula (make-atomic-formula and-term))
	 (atom-formula1 (make-atomic-formula varterm1))
	 (formula-of-and-atom-to-left-aconst
	  (mk-all var1 var2 (make-imp and-formula atom-formula1))))
    (make-aconst "AndAtomToLeft" 'axiom
		 formula-of-and-atom-to-left-aconst empty-subst)))

(define and-atom-to-right-aconst
  (let* ((name (default-var-name (make-alg "boole")))
	 (var1 (make-var (make-alg "boole") 1 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (var2 (make-var (make-alg "boole") 2 t-deg-zero name))
	 (varterm2 (make-term-in-var-form var2))
	 (and-term
	  (mk-term-in-app-form
	   (make-term-in-const-form and-const) varterm1 varterm2))
	 (and-formula (make-atomic-formula and-term))
	 (atom-formula2 (make-atomic-formula varterm2))
	 (formula-of-and-atom-to-right-aconst
	  (mk-all var1 var2 (make-imp and-formula atom-formula2))))
    (make-aconst "AndAtomToRight" 'axiom
		 formula-of-and-atom-to-right-aconst empty-subst)))

;; all boole^1,boole^2(boole^1 impb boole^2 -> boole^1 -> boole^2) can
;; be proved in TCF+ from the computation rules for ImpConst .

(define atom-to-imp-aconst
  (let* ((name (default-var-name (make-alg "boole")))
	 (var1 (make-var (make-alg "boole") 1 t-deg-zero name))
	 (varterm1 (make-term-in-var-form var1))
	 (var2 (make-var (make-alg "boole") 2 t-deg-zero name))
	 (varterm2 (make-term-in-var-form var2))
	 (imp-term
	  (mk-term-in-app-form
	   (make-term-in-const-form imp-const) varterm1 varterm2))
	 (imp-formula (make-atomic-formula imp-term))
	 (atom-formula1 (make-atomic-formula varterm1))
	 (atom-formula2 (make-atomic-formula varterm2))
	 (formula-of-atom-to-imp-aconst
	  (mk-all var1 var2 (mk-imp imp-formula atom-formula1 atom-formula2))))
    (make-aconst "AtomToImp" 'axiom
		 formula-of-atom-to-imp-aconst empty-subst)))

(define total-pair-aconst
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (name1 (default-var-name tvar1))
	 (var1 (make-var tvar1 1 t-deg-zero name1))
	 (name2 (default-var-name tvar2))
	 (var2 (make-var tvar2 2 t-deg-zero name2))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (pairterm (make-term-in-pair-form varterm1 varterm2))
	 (totalpairfla (make-total pairterm))
	 (totalfla1 (make-total varterm1))
	 (totalfla2 (make-total varterm2))
	 (formula-of-total-pair-aconst
	  (mk-imp totalpairfla (mk-and totalfla1 totalfla2))))
    (make-aconst "TotalPair" 'axiom formula-of-total-pair-aconst empty-subst)))

;; (pp (aconst-to-formula total-pair-aconst))
;; allnc alpha1^1,alpha2^2(
;;  Total(alpha1^1@alpha2^2) -> Total alpha1^1 & Total alpha2^2)

;; (pp (formula-to-et-type (aconst-to-formula total-pair-aconst)))
;; alpha1@@alpha2=>alpha1@@alpha2

(define total-pair-rev-aconst
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (name1 (default-var-name tvar1))
	 (var1 (make-var tvar1 1 t-deg-zero name1))
	 (name2 (default-var-name tvar2))
	 (var2 (make-var tvar2 2 t-deg-zero name2))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (pairterm (make-term-in-pair-form varterm1 varterm2))
	 (totalpairfla (make-total pairterm))
	 (totalfla1 (make-total varterm1))
	 (totalfla2 (make-total varterm2))
	 (formula-of-total-pair-rev-aconst
	  (mk-imp (mk-and totalfla1 totalfla2) totalpairfla)))
    (make-aconst "TotalPairRev"
		 'axiom formula-of-total-pair-rev-aconst empty-subst)))

;; (pp (aconst-to-formula total-pair-rev-aconst))
;; '(
;; allnc alpha1^1,alpha2^2(
;;  Total alpha1^1 & Total alpha2^2 -> Total(alpha1^1@alpha2^2))
;; )

;; (define tsubst (make-subst (py "alpha") (py "nat")))
;; (define psubst
;;   (make-subst (make-pvar (make-arity (py "alpha")) -1 h-deg-zero n-deg-zero "")
;; 	      (make-cterm (pv "n^")
;; 			  (pf "(Pvar nat)n^"))))
;; (define tpsubst (append tsubst psubst))
;; (admissible-substitution? tpsubst (aconst-to-formula extotal-aconst))

;; (pp-subst tpsubst)

;; (define subst-aconst (aconst-substitute alltotal-aconst tpsubst))

;; (pp (rename-variables (aconst-to-formula subst-aconst)))
;; all n (Pvar nat)n -> allnc n^(Total n^ -> (Pvar nat)n^)

;; (pp (rename-variables (unfold-formula (aconst-to-formula subst-aconst))))
;; all n (Pvar nat)n -> allnc n^(TotalNat n^ -> (Pvar nat)n^)

;; Additional axioms with names "Intro" and "Elim"

(define (number-and-idpredconst-to-intro-aconst i idpc)
  (let* ((name (idpredconst-to-name idpc))
	 (types (idpredconst-to-types idpc))
	 (tpsubst (idpredconst-to-tpsubst idpc))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (param-pvar-cterms
	  (if (member name '("ExDT" "ExLT" "ExRT" "ExUT"))
	      (map predicate-to-cterm-with-total-vars param-pvars)
	      (map predicate-to-cterm param-pvars)))
	 (idpc-names-with-pvars-and-opt-alg-names
	  (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	   name))
	 (names (map car idpc-names-with-pvars-and-opt-alg-names))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (tvars (idpredconst-name-to-tvars name))
	 (uninst-idpcs (map (lambda (name)
			       (make-idpredconst name tvars param-pvar-cterms))
			     names))
	 (uninst-idpc-cterms (map predicate-to-cterm uninst-idpcs))
	 (psubst-for-pvars (make-substitution-wrt pvar-cterm-equal?
						  pvars uninst-idpc-cterms))
	 (orig-clauses (idpredconst-name-to-clauses name))
	 (orig-clause
	  (if (and (integer? i) (< i (length orig-clauses)))
	      (list-ref orig-clauses i)
	      (myerror "number-and-idpredconst-to-intro-aconst" i
		       "should be an index of a clause for" name)))
	 (uninst-clause (formula-substitute orig-clause psubst-for-pvars))
	 (params (idpredconst-name-to-params name)))
    (make-aconst "Intro" 'axiom
		 (apply mk-allnc (append params (list uninst-clause)))
		 tpsubst i idpc)))

;; Again, now parallel to induction, with repro-formulas.  Reason:
;; repro-formulas needed for proof-to-extracted-term-aux , and probably
;; also for normalization via terms.

;; Now for elimination.  

;; imp-formulas is a list of formulas I xs^ -> A(xs^), where the left
;; hand sides make up the relevant idpcs.  Then from the relevant
;; clauses (i.e., those implying relevant idpcs) we can form the step
;; formulas, shortened by omitting all of its premises containing
;; irrelevant idpcs.  For each relevant uninst-idpc and corresponding
;; rel-pvar we form an uninst-elim-formula assuming uninst-idpc
;; and the (fixed) step formulas and yielding rel-pvar.

(define (imp-formulas-to-uninst-elim-formulas-etc . imp-formulas)
  (if (null? imp-formulas)
      (myerror "imp-formulas-to-uninst-elim-formulas-etc"
	       "at least one imp-formula expected"))
  (for-each (lambda (imp-fla)
	      (if (not (imp-form? imp-fla))
		  (myerror "imp-formulas-to-uninst-elim-formulas-etc"
			   "implication formula expected" imp-fla)))
	    imp-formulas)
  (for-each (lambda (imp-fla)
	      (let ((prem (imp-form-to-premise imp-fla)))
		(if (not (and
			  (predicate-form? prem)
			  (let ((pred (predicate-form-to-predicate prem)))
			    (and (idpredconst-form? pred)
				 (assoc (idpredconst-to-name pred) IDS)))))
		    (myerror "imp-formulas-to-uninst-elim-formulas-etc"
			     "idpredconst formula expected" prem))))
	    imp-formulas)
  (let* ((prems (map imp-form-to-premise imp-formulas))
	 (concls (map imp-form-to-conclusion imp-formulas))
	 (rel-idpcs (map predicate-form-to-predicate prems))
	 (rel-idpc-names (map idpredconst-to-name rel-idpcs))
	 (idpc (car rel-idpcs))
	 (name (car rel-idpc-names))
	 (idpc-names-with-pvars-and-opt-alg-names
	  (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names name))
	 (names (map car idpc-names-with-pvars-and-opt-alg-names))
	 (sorted-rel-idpc-names (list-transform-positive names
				  (lambda (x) (member x rel-idpc-names))))
	 (sorted-rel-clauses ;with Xj, sorted
	  (apply append (map idpredconst-name-to-clauses
			     sorted-rel-idpc-names)))
	 (rel-pvars (map idpredconst-name-to-pvar rel-idpc-names))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (irrel-pvars (set-minus pvars rel-pvars))
	 (tvars (idpredconst-name-to-tvars name))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (param-pvar-cterms
	  (if (member name '("ExDT" "ExLT" "ExRT" "ExUT"))
	      (map predicate-to-cterm-with-total-vars param-pvars)
	      (map predicate-to-cterm param-pvars)))
	 (params (idpredconst-name-to-params name))
	 (new-var-lists
	  (map (lambda (pvar)
		 (append params (map type-to-new-partial-var
				     (list-tail
				      (arity-to-types (pvar-to-arity pvar))
				      (length params)))))
	       rel-pvars))
	 (rel-pvar-formulas
	  (map (lambda (pvar vars)
		 (apply make-predicate-formula
			pvar (map make-term-in-var-form vars)))
	       rel-pvars new-var-lists))
	 (rel-uninst-idpcs
	  (map (lambda (name) (make-idpredconst name tvars param-pvar-cterms))
	       rel-idpc-names))
	 (rel-uninst-idpc-formulas
	  (map (lambda (uninst-idpc vars)
		 (apply make-predicate-formula
			uninst-idpc (map make-term-in-var-form vars)))
	       rel-uninst-idpcs new-var-lists))
	 (rel-uninst-idpc-cterms
	  (map (lambda (vars fla) (apply make-cterm (append vars (list fla))))
	       new-var-lists rel-uninst-idpc-formulas))
	 (conjs (map make-andd rel-uninst-idpc-formulas rel-pvar-formulas))
	 (conj-cterms (map (lambda (vars conj)
			     (apply make-cterm (append vars (list conj))))
			   new-var-lists conjs))
	 (pvars-to-idpcs-psubst
	  (make-substitution-wrt pvar-cterm-equal?
				 rel-pvars rel-uninst-idpc-cterms))
	 (pvars-to-conjs-psubst
	  (make-substitution-wrt pvar-cterm-equal?
				 rel-pvars conj-cterms))
	 (sorted-clause-prems-list (map imp-impnc-all-allnc-form-to-premises
					sorted-rel-clauses))
	 (sorted-stepformula-prem-lists-list
	  (map
	   (lambda (clause-prems)
	     (map
	      (lambda (clause-prem)
		(cond ;prem has irrelevant pvars: no stepformula prem
		 ((pair? (intersection (formula-to-pvars clause-prem)
				       irrel-pvars))
		  '())
		 ((and ;unnested prem: duplication
		   (predicate-form?
		    (imp-impnc-all-allnc-form-to-final-conclusion clause-prem))
		   (member (predicate-form-to-predicate
			    (imp-impnc-all-allnc-form-to-final-conclusion
			     clause-prem))
			   rel-pvars))
		  (list (formula-substitute clause-prem pvars-to-idpcs-psubst)
			clause-prem))
		 (else ;nested prem: substitute conjunction of idpcs and pvars
		  (list (formula-substitute clause-prem
					    pvars-to-conjs-psubst)))))
	      clause-prems))
	   sorted-clause-prems-list))
	 (prem-to-prems-alists
	  (map (lambda (stepformula-prem-lists clause-prems)
		 (map list clause-prems stepformula-prem-lists))
	       sorted-stepformula-prem-lists-list sorted-clause-prems-list))
	 (sorted-simplified-strengthened-rel-clauses
	  (letrec
	      ((imp-all-formula-and-alist-to-prem-modified-formula
		(lambda (formula prem-to-prems-alist)
		  (cond
		   ((predicate-form? formula) formula)
		   ((imp-impnc-form? formula)
		    (let* ((prem (imp-impnc-form-to-premise formula))
			   (concl (imp-impnc-form-to-conclusion formula))
			   (prev
			    (imp-all-formula-and-alist-to-prem-modified-formula
			     concl prem-to-prems-alist))
			   (info (assoc prem prem-to-prems-alist)))
		      (if
		       info
		       (if (imp-form? formula)
			   (apply mk-imp (append (cadr info) (list prev)))
			   (apply mk-impnc (append (cadr info) (list prev))))
		       (apply
			myerror
			"imp-all-formula-and-alist-to-prem-modified-formula"
			"premise" prem
			"does not occur in prem-to-prems-alist"
			prem-to-prems-alist))))
		   ((all-allnc-form? formula)
		    (let* ((var (all-allnc-form-to-var formula))
			   (kernel (all-allnc-form-to-kernel formula))
			   (prev
			    (imp-all-formula-and-alist-to-prem-modified-formula
			     kernel prem-to-prems-alist)))
		      (if (all-form? formula)
			  (make-all var prev)
			  (make-allnc var prev))))
		   (else (myerror
			  "imp-all-formula-and-alist-to-prem-modified-formula"
			  "formula in imp-impnc-all-allnc-form expected"
			  formula))))))
	    (map imp-all-formula-and-alist-to-prem-modified-formula
		 sorted-rel-clauses prem-to-prems-alists)))
	 (uninst-elim-formulas-in-imp-form
	  (map (lambda (uninst-idpc-formula pvar-formula)
		 (apply mk-imp uninst-idpc-formula
			(append sorted-simplified-strengthened-rel-clauses
				(list pvar-formula))))
	       rel-uninst-idpc-formulas rel-pvar-formulas))
	 (uninst-elim-formulas
	  (if ;IMR case
	   (apply and-op (map mr-idpredconst-name? rel-idpc-names))
	   (map (lambda (new-var-list uninst-elim-formula-in-imp-form)
		  (make-all (car new-var-list)
			    uninst-elim-formula-in-imp-form))
		new-var-lists uninst-elim-formulas-in-imp-form)
	   uninst-elim-formulas-in-imp-form))
	 (arg-lists (map predicate-form-to-args prems))
	 (var-lists
	  (map (lambda (args)
		 (map (lambda (arg)
			(if (and
			     (term-in-var-form? arg)
			     (t-deg-zero?
			      (var-to-t-deg (term-in-var-form-to-var arg))))
			    (term-in-var-form-to-var arg)
			    (myerror
			     "imp-formulas-to-uninst-elim-formulas-etc"
			     "partial variable expected" arg)))
		      args))
	       arg-lists))
	 (var-lists-test
	  (for-each
	   (lambda (vars prem)
	     (if (not (= (length (remove-duplicates vars)) (length vars)))
		 (myerror "imp-formulas-to-uninst-elim-formulas-etc"
			  "distinct variables expected in" prem)))
	   var-lists prems))
	 (concl-cterms (map (lambda (vars concl)
			      (apply make-cterm (append vars (list concl))))
			    var-lists concls))
	 (psubst-for-pvars
	  (make-substitution-wrt pvar-cterm-equal? rel-pvars concl-cterms))
	 (tpsubst (idpredconst-to-tpsubst idpc))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (psubst-for-param-pvars (list-transform-positive tpsubst
				   (lambda (x) (pvar-form? (car x))))))
    (list uninst-elim-formulas
	  tsubst psubst-for-param-pvars psubst-for-pvars)))

(define (clause-to-simplified-clause clause irrelevant-pvars)
  (cond
   ((all-form? clause)
    (make-all (all-form-to-var clause)
	      (clause-to-simplified-clause
	       (all-form-to-kernel clause) irrelevant-pvars)))
   ((allnc-form? clause)
    (make-allnc (allnc-form-to-var clause)
		(clause-to-simplified-clause
		 (allnc-form-to-kernel clause) irrelevant-pvars)))
   ((impnc-form? clause)
    (make-impnc (impnc-form-to-premise clause)
		(clause-to-simplified-clause
		 (impnc-form-to-conclusion clause) irrelevant-pvars)))
   ((imp-form? clause)
    (let* ((prem (imp-form-to-premise clause))
	   (final-conc (imp-impnc-all-allnc-form-to-final-conclusion prem)))
      (if (and (predicate-form? final-conc)
	       (member (predicate-form-to-predicate final-conc)
		       irrelevant-pvars))
	  (clause-to-simplified-clause
	   (imp-form-to-conclusion clause) irrelevant-pvars)
	  (make-imp prem (clause-to-simplified-clause
			  (imp-form-to-conclusion clause) irrelevant-pvars)))))
   ((predicate-form? clause) clause)
   (else (myerror "clause-to-simplified-clause"
		  "clause expected" clause))))

(define (replace-final-conclusion clause new-conclusion)
  (cond
   ((all-form? clause)
    (make-all (all-form-to-var clause)
	      (replace-final-conclusion
	       (all-form-to-kernel clause) new-conclusion)))
   ((allnc-form? clause)
    (make-allnc (allnc-form-to-var clause)
		(replace-final-conclusion
		 (allnc-form-to-kernel clause) new-conclusion)))
   ((impnc-form? clause)
    (make-impnc (impnc-form-to-premise clause)
		(replace-final-conclusion
		 (impnc-form-to-conclusion clause) new-conclusion)))
   ((imp-form? clause)
    (make-imp (imp-form-to-premise clause)
	      (replace-final-conclusion
	       (imp-form-to-conclusion clause) new-conclusion)))
   ((predicate-form? clause) new-conclusion)
   (else (myerror "replace-final-conclusion"
		  "clause expected" clause))))

;; Will be obsolete once imp-formulas-to-mr-elim-proof is adapted to the
;; general form of clauses.

(define (idpc-clause-to-rec-premises idpc-clause idpc-pvars)
  (let* ((kernel (all-allnc-form-to-final-kernel idpc-clause))
	 (prems (imp-impnc-form-to-premises kernel)))
    (list-transform-positive prems
      (lambda (prem)
	(pair? (intersection
		(formula-to-pvars
		 (imp-impnc-all-allnc-form-to-final-conclusion prem))
		idpc-pvars))))))

;; We define uniform-non-rec-idpc-clause?  We require that it has the
;; form allnc xs(impnc-param-prems --> X rs).  Every param-prem has
;; only param-pvars free and they can only occur strictly positive.  X
;; is from idpc-pvars.

(define (uniform-non-rec-idpc-clause? formula param-pvars idpc-pvars)
  (and
   (null? (formula-to-free formula))
   (let* ((allnc-kernel (allnc-form-to-final-kernel formula))
	  (param-prems (impnc-form-to-premises allnc-kernel))
	  (final-concl (impnc-form-to-final-conclusion allnc-kernel)))
     (and (predicate-form? final-concl)
	  (member (predicate-form-to-predicate final-concl) idpc-pvars)
	  (apply and-op (map (lambda (fla)
			       (null? (set-minus param-pvars
						 (formula-to-spos-pvars fla))))
			     param-prems))))))

;; In non-computational-invariant formulas the following idpcs are
;; admitted.  (1) mr-nc-idpcs.  (2) uniform-one-clause-nc-idpcs with
;; non-computational-invariant parameters.  (3)
;; restricted-elim-nc-idpcs.

(define (non-computational-invariant? formula param-pvars)
  (cond
   ((atom-form? formula) #t)
   ((prime-predicate-form? formula)
    (let ((pred (predicate-form-to-predicate formula)))
      (cond
       ((pvar-form? pred) (member pred param-pvars))
       ((predconst-form? pred)
	(not (string=? (predconst-to-name pred) "Total")))
       ((idpredconst-form? pred)
	(let* ((name (idpredconst-to-name pred))
	       (cterms (idpredconst-to-cterms pred))
	       (formulas (map cterm-to-formula cterms)))
	  (and
	   (null? (idpredconst-name-to-opt-alg-name name))
	   (or (mr-nc-idpc-name? name)
	       (and (uniform-one-clause-nc-idpc-name? name)
		    (apply and-op (map (lambda (fla)
					 (non-computational-invariant?
					  fla param-pvars))
				       formulas)))
	       (restricted-elim-nc-idpc-name? name)))))
       (else (myerror "non-computational-invariant?"
		      "predicate expected" pred)))))
   ((imp-form? formula)
    (let ((prem (imp-form-to-premise formula))
	  (conc (imp-form-to-conclusion formula)))
      (and (non-computational-invariant? prem param-pvars)
	   (non-computational-invariant? conc param-pvars))))
   ((impnc-form? formula)
    (let ((prem (impnc-form-to-premise formula))
	  (conc (impnc-form-to-conclusion formula)))
      (and (non-computational-invariant? prem param-pvars)
	   (non-computational-invariant? conc param-pvars))))
   ((and-form? formula)
    (let ((left (and-form-to-left formula))
	  (right (and-form-to-right formula)))
      (and (non-computational-invariant? left param-pvars)
	   (non-computational-invariant? right param-pvars))))
   ((andd-form? formula)
    (let ((left (andd-form-to-left formula))
	  (right (andd-form-to-right formula)))
      (and (non-computational-invariant? left param-pvars)
	   (non-computational-invariant? right param-pvars))))
   ((andl-form? formula)
    (let ((right (andl-form-to-left formula)))
      (non-computational-invariant? left param-pvars)))
   ((andr-form? formula)
    (let ((right (andr-form-to-right formula)))
      (non-computational-invariant? right param-pvars)))
   ((andu-form? formula) #t)
   ((ord-form? formula) #f)
   ((orl-form? formula) #f)
   ((orr-form? formula) #f)
   ((oru-form? formula) #f)
   ((all-form? formula)
    (let ((kernel (all-form-to-kernel formula)))
      (non-computational-invariant? kernel param-pvars)))
   ((ex-form? formula) #f)
   ((allnc-form? formula)
    (let ((kernel (allnc-form-to-kernel formula)))
      (non-computational-invariant? kernel param-pvars)))
   ((exnc-form? formula) ;obsolete
    (let ((kernel (exnc-form-to-kernel formula)))
      (non-computational-invariant? kernel param-pvars)))
   ((exd-form? formula) #f)
   ((exl-form? formula) #f)
   ((exr-form? formula) #f)
   ((exu-form? formula)
    (let ((kernel (exu-form-to-kernel formula)))
      (non-computational-invariant? kernel param-pvars)))
   ((exdt-form? formula) #f)
   ((exlt-form? formula) #f)
   ((exrt-form? formula) #f)
   ((exut-form? formula)
    (let ((kernel (exut-form-to-kernel formula)))
      (non-computational-invariant? kernel param-pvars)))
   ((or (exca-form? formula) (excl-form? formula))
    (non-computational-invariant? (unfold-formula formula) param-pvars))
   (else (myerror "non-computational-invariant?" "formula expected" formula))))

(define (mr-nc-idpc-name? name) ;ends with "MR"
  (final-substring? "MR" name))

(define (uniform-one-clause-nc-idpc-name? name)
  (let* ((clauses (idpredconst-name-to-clauses name))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (idpc-pvars (idpredconst-name-to-pvars name)))
    (and (= 1 (length clauses))
	 (uniform-non-rec-idpc-clause? (car clauses) param-pvars idpc-pvars))))

(define (restricted-elim-nc-idpc-name? name)
  (let* ((clauses (idpredconst-name-to-clauses name))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (idpc-pvars (idpredconst-name-to-pvars name)))
    (apply and-op
	   (map (lambda (fla)
		  (non-computational-invariant?
		   fla (append param-pvars idpc-pvars)))
		clauses))))

(define (imp-formulas-to-uninst-elim-formula-etc . imp-formulas)
  (let* ((uninst-elim-formulas-etc
	  (apply imp-formulas-to-uninst-elim-formulas-etc imp-formulas))
	 (uninst-elim-formulas (car uninst-elim-formulas-etc))
	 (rest (cdr uninst-elim-formulas-etc)))
    (cons (car uninst-elim-formulas) rest)))

;; We define a procedure that takes imp-formulas and returns the
;; corresponding elimination axiom.

(define (imp-formulas-to-elim-aconst . imp-formulas)
  (let* ((uninst-elim-formula-etc
	  (apply imp-formulas-to-uninst-elim-formula-etc imp-formulas))
	 (uninst-elim-formula (car uninst-elim-formula-etc))
	 (tpsubst (apply append (cdr uninst-elim-formula-etc))))
    (if (all-form? uninst-elim-formula) ;IMR case
	(apply make-aconst "ElimMR" 'axiom uninst-elim-formula tpsubst
	       imp-formulas)
	(apply make-aconst "Elim" 'axiom uninst-elim-formula tpsubst
	       imp-formulas))))

;; Additional axioms with names "Closure" and "Gfp"

(define (coidpredconst-to-closure-aconst coidpc)
  (let* ((name (idpredconst-to-name coidpc))
	 (types (idpredconst-to-types coidpc))
	 (tpsubst (idpredconst-to-tpsubst coidpc))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (param-pvar-cterms (map predicate-to-cterm param-pvars))
	 (idpc-names-with-pvars-and-opt-alg-names
	  (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	   name))
	 (names (map car idpc-names-with-pvars-and-opt-alg-names))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (tvars (idpredconst-name-to-tvars name))
	 (uninst-idpcs (map (lambda (name)
			       (make-idpredconst name tvars param-pvar-cterms))
			     names))
	 (uninst-idpc-cterms (map predicate-to-cterm uninst-idpcs))
	 (psubst-for-pvars (make-substitution-wrt pvar-cterm-equal?
						  pvars uninst-idpc-cterms))
	 (orig-clause (car (idpredconst-name-to-clauses name)))
	 (uninst-clause (formula-substitute orig-clause psubst-for-pvars))
	 (params (idpredconst-name-to-params name)))
    (make-aconst "Closure" 'axiom
		 (apply mk-allnc (append params (list uninst-clause)))
		 tpsubst)))

;; Now for the greatest fixed point axioms.  

;; imp-formulas is a list of formulas A(xs^) -> J xs^ , where the right
;; hand sides make up the relevant coidpcs.  Then from the relevant
;; coclauses (i.e., those implying relevant coidpcs) we can form the
;; step formulas, shortened by omitting all of its disjuncts containing
;; irrelevant coidpcs.  Then the j-th uninst-gfp-formula is
;; R_j xs^ -> (allnc xs^(R_j xs^ -> disj))_{j<N} -> J_j xs^.

(define (imp-formulas-to-uninst-gfp-formulas-etc . imp-formulas)
  (if (null? imp-formulas)
      (myerror "imp-formulas-to-uninst-gfp-formulas-etc"
	       "at least one imp-formula expected"))
  (for-each (lambda (imp-fla)
	      (if (not (imp-form? imp-fla))
		  (myerror "imp-formulas-to-uninst-gfp-formulas-etc"
			   "implication formula expected" imp-fla)))
	    imp-formulas)
  (for-each (lambda (imp-fla)
	      (let ((concl (imp-form-to-conclusion imp-fla)))
		(if (not (and
			  (predicate-form? concl)
			  (let ((pred (predicate-form-to-predicate concl)))
			    (and (idpredconst-form? pred)
				 (assoc (idpredconst-to-name pred) COIDS)))))
		    (myerror "imp-formulas-to-uninst-gfp-formulas-etc"
			     "coidpredconst formula expected" concl))))
	    imp-formulas)
  (let* ((prems (map imp-form-to-premise imp-formulas))
	 (concls (map imp-form-to-conclusion imp-formulas))
	 (rel-coidpcs (map predicate-form-to-predicate concls))
	 (rel-coidpc-names (map idpredconst-to-name rel-coidpcs))
	 (coidpc (car rel-coidpcs))
	 (name (car rel-coidpc-names))
	 (coidpc-names-with-pvars-and-opt-alg-names
	  (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names name))
	 (names (map car coidpc-names-with-pvars-and-opt-alg-names))
	 (sorted-rel-coidpc-names (list-transform-positive names
				    (lambda (x) (member x rel-coidpc-names))))
	 (sorted-rel-coclauses ;with Xj, sorted
	  (apply append (map idpredconst-name-to-clauses
			     sorted-rel-coidpc-names)))
	 (rel-pvars (map idpredconst-name-to-pvar rel-coidpc-names))
	 (pvars (map cadr coidpc-names-with-pvars-and-opt-alg-names))
	 (irrel-pvars (set-minus pvars rel-pvars))
	 (tvars (idpredconst-name-to-tvars name))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (param-pvar-cterms
	  (if (member name '("CoExDT" "CoExLT" "CoExRT" "CoExUT"))
	      (map predicate-to-cterm-with-total-vars param-pvars)
	      (map predicate-to-cterm param-pvars)))
	 (params (idpredconst-name-to-params name))
	 (new-var-lists
	  (map (lambda (pvar)
		 (append params (map type-to-new-partial-var
				     (list-tail
				      (arity-to-types (pvar-to-arity pvar))
				      (length params)))))
	       rel-pvars))
	 (rel-pvar-formulas
	  (map (lambda (pvar vars)
		 (apply make-predicate-formula
			pvar (map make-term-in-var-form vars)))
	       rel-pvars new-var-lists))
	 (rel-uninst-coidpcs
	  (map (lambda (name) (make-idpredconst name tvars param-pvar-cterms))
	       rel-coidpc-names))
	 (rel-uninst-coidpc-formulas
	  (map (lambda (uninst-coidpc vars)
		 (apply make-predicate-formula
			uninst-coidpc (map make-term-in-var-form vars)))
	       rel-uninst-coidpcs new-var-lists))
					;now for sorted-strengthened-coclauses
	 (uninst-var-lists (map all-allnc-form-to-vars sorted-rel-coclauses))
	 (disjuncts-list (map (lambda (coclause)
				(or-form-to-disjuncts
				 (imp-impnc-form-to-conclusion
				  (all-allnc-form-to-final-kernel coclause))))
			      sorted-rel-coclauses))
	 (sorted-rel-pvars (map (lambda (sorted-rel-coclause)
				  (predicate-form-to-predicate
				   (imp-impnc-form-to-premise
				    (all-allnc-form-to-final-kernel
				     sorted-rel-coclause))))
				sorted-rel-coclauses))
	 (sorted-shortened-coclauses
	  (map
	   (lambda (vars pvars disjuncts)
	     (letrec
		 ((and-ex-fla-to-shortened-fla
		   (lambda (fla)
		     (cond
		      ((prime-predicate-form? fla) fla)
		      ((and (bicon-form? fla)
			    (memq (bicon-form-to-bicon fla)
				  '(andd andl andr andu and)))
		       (if
			(pair? (intersection (formula-to-pvars fla)
					     irrel-pvars))
			(and-ex-fla-to-shortened-fla (bicon-form-to-right fla))
			(make-bicon (bicon-form-to-bicon fla)
				    (bicon-form-to-left fla)
				    (and-ex-fla-to-shortened-fla
				     (bicon-form-to-right fla)))))
		      ((and (quant-form? fla)
			    (memq (quant-form-to-quant fla)
				  '(exd exl exr exu ex)))
		       (make-quant (quant-form-to-quant fla)
				   (quant-form-to-vars fla) ;check syntax
				   (and-ex-fla-to-shortened-fla
				    (quant-form-to-kernel fla))))
		      (else (myerror "and-ex-fla-to-shortened-fla"
				     "unexpected formula" fla))))))
	       (apply
		mk-allnc
		(append
		 vars (list (make-imp
			     (apply make-predicate-formula
				    pvars (map make-term-in-var-form vars))
			     (apply mk-ori (map and-ex-fla-to-shortened-fla
						disjuncts))))))))
	   uninst-var-lists sorted-rel-pvars disjuncts-list))
	 (sorted-rel-pvar-formulas
	  (map (lambda (pvar vars)
		 (apply make-predicate-formula
			pvar (map make-term-in-var-form vars)))
	       sorted-rel-pvars uninst-var-lists))
	 (sorted-rel-uninst-coidpcs
	  (map (lambda (name) (make-idpredconst name tvars param-pvar-cterms))
	       sorted-rel-coidpc-names))
	 (sorted-rel-uninst-coidpc-formulas
	  (map (lambda (uninst-coidpc vars)
		 (apply make-predicate-formula
			uninst-coidpc (map make-term-in-var-form vars)))
	       sorted-rel-uninst-coidpcs uninst-var-lists))
	 (arg-lists (map predicate-form-to-args concls))
	 (var-lists
	  (map (lambda (args)
		 (map (lambda (arg)
			(if (and
			     (term-in-var-form? arg)
			     (t-deg-zero?
			      (var-to-t-deg (term-in-var-form-to-var arg))))
			    (term-in-var-form-to-var arg)
			    (myerror "imp-formulas-to-uninst-gfp-formulas-etc"
				     "partial variable expected" arg)))
		      args))
	       arg-lists))
	 (var-lists-test
	  (for-each
	   (lambda (vars concl)
	     (if (not (= (length (remove-duplicates vars)) (length vars)))
		 (myerror "imp-formulas-to-uninst-gfp-formulas-etc"
			  "distinct variables expected in" concl)))
	   var-lists concls))
	 (prem-cterms (map (lambda (vars prem)
			     (apply make-cterm (append vars (list prem))))
			   var-lists prems))
	 (disjs
	  (map (lambda (rel-uninst-coidpc-formula rel-pvar-formula prem-cterm)
		 ((if (formula-of-nulltype? (cterm-to-formula prem-cterm))
		      make-orl make-ord)
		  rel-uninst-coidpc-formula rel-pvar-formula))
	       sorted-rel-uninst-coidpc-formulas
	       sorted-rel-pvar-formulas prem-cterms))
	 (disj-cterms (map (lambda (vars disj)
			     (apply make-cterm (append vars (list disj))))
			   uninst-var-lists disjs))
	 (pvars-to-disjs-psubst
	  (make-substitution-wrt pvar-cterm-equal?
				 sorted-rel-pvars disj-cterms))
	 (prelim-subst-sorted-shortened-coclauses
	  (map (lambda (shortened-coclause)
		 (formula-substitute shortened-coclause pvars-to-disjs-psubst))
	       sorted-shortened-coclauses))
	 (sorted-strengthened-coclauses
	  (map (lambda (fla) ;prelim-subst-sorted-shortened-coclause
		 (let* ((vars (all-allnc-form-to-vars fla))
			(kernel (all-allnc-form-to-final-kernel fla))
			(disj (imp-impnc-form-to-premise kernel))
			(str-concl (imp-impnc-form-to-conclusion kernel))
			(right-disjunct
			 (cond ((ord-form? disj) (ord-form-to-right disj))
			       ((orl-form? disj) (orl-form-to-right disj))
			       (else
				(myerror "ord od orl form expected" disj)))))
		   (apply
		    mk-allnc
		    (append vars (list (make-imp right-disjunct str-concl))))))
	       prelim-subst-sorted-shortened-coclauses))
	 (uninst-gfp-formulas
	  (map (lambda (pvar-formula uninst-coidpc-formula)
		 (apply mk-imp pvar-formula
			(append sorted-strengthened-coclauses
				(list uninst-coidpc-formula))))
	       rel-pvar-formulas rel-uninst-coidpc-formulas))
	 (psubst-for-pvars
	  (make-substitution-wrt pvar-cterm-equal? rel-pvars prem-cterms))
	 (tpsubst (idpredconst-to-tpsubst coidpc))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (psubst-for-param-pvars (list-transform-positive tpsubst
				   (lambda (x) (pvar-form? (car x))))))
    (list uninst-gfp-formulas
	  tsubst psubst-for-param-pvars psubst-for-pvars)))

(define (imp-formulas-to-uninst-gfp-formula-etc . imp-formulas)
  (let* ((uninst-gfp-formulas-etc
	  (apply imp-formulas-to-uninst-gfp-formulas-etc imp-formulas))
	 (uninst-gfp-formulas (car uninst-gfp-formulas-etc))
	 (rest (cdr uninst-gfp-formulas-etc)))
    (cons (car uninst-gfp-formulas) rest)))

;; We define a procedure that takes imp-formulas and returns the
;; corresponding greatest fixed point axiom.

(define (imp-formulas-to-gfp-aconst . imp-formulas)
  (let* ((uninst-gfp-formula-etc
	  (apply imp-formulas-to-uninst-gfp-formula-etc
		 imp-formulas))
	 (uninst-gfp-formula (car uninst-gfp-formula-etc))
	 (tpsubst (apply append (cdr uninst-gfp-formula-etc))))
    (make-aconst "Gfp" 'axiom uninst-gfp-formula tpsubst)))

;; Theorems

;; A theorem is a special assumption constant.  We maintain an
;; association list THEOREMS assigning to every name of a theorem the
;; assumption constant and its proof.

;; Format of THEOREMS 
;; ((name aconst proof <extracted-term>) ...)

(define (theorem-name-to-aconst name)
  (let ((info (assoc name THEOREMS)))
    (if info
	(cadr info)
	(myerror "theorem-name-to-aconst" "theorem name expected" name))))

(define (theorem-name-to-proof name)
  (let ((info (assoc name THEOREMS)))
    (if info
	(caddr info)
	(myerror "theorem-name-to-proof" "theorem name expected" name))))

(define (theorem-aconst-to-inst-proof aconst)
  (let* ((name (aconst-to-name aconst))
	 (kind (aconst-to-kind aconst))
	 (tpsubst (aconst-to-tpsubst aconst))
	 (proof-of-thm
	  (if (eq? 'theorem kind)
	      (theorem-name-to-proof name)
	      (myerror "theorem-aconst-to-inst-proof" "kind theorem expected"
		       kind (aconst-to-formula aconst)))))
    (proof-substitute proof-of-thm tpsubst)))

(define (theorem-or-global-assumption-name-to-pconst-name string)
  (string-append "c"
		 (list->string (remove-numerals (string->list string)))))

(define (remove-numerals charlist)
  (if (null? charlist)
      '()
      (append (let ((char (car charlist)))
		(cond ((char=? char #\-) (list #\X #\x))
		      ((char=? char #\() (list #\Y #\y)) 
		      ((char=? char #\)) (list #\y #\Y)) 
		      ((char=? char #\1) (list #\O #\n #\e)) 
		      ((char=? char #\2) (list #\T #\w #\o)) 
		      ((char=? char #\3) (list #\T #\h #\r #\e #\e)) 
		      ((char=? char #\4) (list #\F #\o #\u #\r))
		      ((char=? char #\5) (list #\F #\i #\v #\e))
		      ((char=? char #\6) (list #\S #\i #\x))
		      ((char=? char #\7) (list #\S #\e #\v #\e #\n))
		      ((char=? char #\8) (list #\E #\i #\g #\h #\t))
		      ((char=? char #\9) (list #\N #\i #\n #\e))
		      ((char=? char #\0) (list #\Z #\e #\r #\o))
		      (else (list char))))
	      (remove-numerals (cdr charlist)))))

(define (theorem-or-global-assumption-name-to-pconst-name string)
  (if (and (final-substring? "Total" string)
	   (pconst-name? (substring string 0 (- (string-length string)
						(string-length "Total")))))
      (substring string 0 (- (string-length string) (string-length "Total")))
      (string-append
       "c" (list->string (remove-numerals (string->list string))))))

(define (theorem-or-global-assumption-to-pconst thm-or-ga)
  (let* ((thm-or-ga-name (aconst-to-name thm-or-ga))
	 (pconst-name
	  (theorem-or-global-assumption-name-to-pconst-name thm-or-ga-name))
	 (pconst (pconst-name-to-pconst pconst-name))
	 (tpsubst (aconst-to-tpsubst thm-or-ga))
	 (tsubst (list-transform-positive tpsubst
		   (lambda (x) (tvar-form? (car x)))))
	 (psubst (list-transform-positive tpsubst
		  (lambda (x) (pvar-form? (car x)))))
	 (new-tsubst
	  (do ((l psubst (cdr l))
	       (res '() (let* ((pvar (caar l))
			       (cterm (cadar l))
			       (cterm-type (formula-to-et-type
					    (cterm-to-formula cterm))))
			  (if (nulltype? cterm-type)
			      res
			      (cons (list (PVAR-TO-TVAR pvar) cterm-type)
				    res)))))
	      ((null? l) (reverse res)))))
    (const-substitute pconst (compose-substitutions tsubst new-tsubst) #f)))

(define (add-theorem string . opt-proof)
  (if (and (null? opt-proof)
	   (null? PPROOF-STATE))
      (myerror
       "add-theorem" "proof argument or proof under construction expected"))
  (let ((proof (if (null? opt-proof)
		   (pproof-state-to-proof)
		   (car opt-proof))))
    (if (not (null? (proof-to-free-avars proof)))
	(apply myerror "unexpected free assumptions"
	       (proof-to-free-avars proof)))
    (if
     (is-used? string '() 'theorem)
     *the-non-printing-object*
     (let ((formula (unfold-formula (proof-to-formula proof)))
	   (nc-viols (nc-violations proof))
	   (h-deg-viols (h-deg-violations proof)))
       (if (pair? nc-viols)
	   (apply myerror "allnc-intro with cvars" nc-viols))
       (if (pair? h-deg-viols)
	   (apply myerror "h-deg violations at aconsts" h-deg-viols))
       (if (final-substring? "Sound" string)
	   (let* ((name (substring string 0 (- (string-length string)
					       (string-length "Sound"))))
		  (orig-proof (theorem-name-to-proof name)))
	     (if (not (classical-formula=?
		       formula
		       (real-and-formula-to-mr-formula
			(proof-to-extracted-term orig-proof)
			(proof-to-formula orig-proof))))
		 (myerror "add-theorem" "formula of theorem" formula
			  "should be equal to the soundness formula"
			  (proof-to-soundness-formula
			   (theorem-name-to-proof name))
			  "of theorem" name))))
       (let ((aconst (make-aconst string 'theorem formula empty-subst)))
	 (set! THEOREMS (cons (list string aconst proof) THEOREMS))
	 (if (not (member string (list "Id" "If")))
	     (comment "ok, " string " has been added as a new theorem."))
	 (if
	  (and (final-substring? "Total" string)
	       (pconst-name?
		(substring string 0 (- (string-length string)
				       (string-length "Total")))))
	  (let* ((name (substring string 0 (- (string-length string)
					      (string-length "Total"))))
		 (pconst (pconst-name-to-pconst name))
		 (term (make-term-in-const-form pconst))
		 (totality-formula (term-to-totality-formula term))
		 (t-deg (const-to-t-deg pconst)))
	    (if (not (formula=? formula totality-formula))
		(begin (remove-theorem string)
		       (myerror "add-theorem" "formula of theorem" formula
				"should be equal to the totality formula"
				totality-formula)))
	    (if (t-deg-one? t-deg)
		(comment "Warning: pconst " name
			 " should have been added with t-deg zero"))
	    (change-t-deg-to-one name))
					;string not pconstname+Total
	  (if (not (formula-of-nulltype? formula))
	      (let* ((pconst-name
		      (theorem-or-global-assumption-name-to-pconst-name
		       string))
		     (type (formula-to-et-type formula))
		     (t-deg (term-to-t-deg (proof-to-extracted-term proof)))
		     (totality-flag #t)) ;no proof of totality needed
		(add-program-constant
		 pconst-name type t-deg 'const 0 totality-flag)))))))))

(define save add-theorem)

(define (nc-violations proof)
  (if (formula-of-nulltype? (proof-to-formula proof))
      '()
      (nc-violations-aux proof)))

;; In nc-violations-aux we can assume that the proved formula has
;; computational content.

(define (nc-violations-aux proof)
  (case (tag proof)
    ((proof-in-avar-form proof-in-aconst-form) '())
    ((proof-in-imp-intro-form)
     (nc-violations-aux (proof-in-imp-intro-form-to-kernel proof)))
    ((proof-in-imp-elim-form)
     (let* ((op (proof-in-imp-elim-form-to-op proof))
	    (arg (proof-in-imp-elim-form-to-arg proof))
	    (prevop (nc-violations-aux op))
	    (prevarg (nc-violations arg)))
       (union prevop prevarg)))
    ((proof-in-impnc-intro-form)
     (let* ((avar (proof-in-impnc-intro-form-to-avar proof))
	    (kernel (proof-in-impnc-intro-form-to-kernel proof))
	    (prev (nc-violations-aux kernel)))
       (if (member-wrt avar=? avar (proof-to-cvars proof))
	   (adjoin avar prev)
	   prev)))
    ((proof-in-impnc-elim-form)
     (nc-violations-aux (proof-in-impnc-elim-form-to-op proof)))
    ((proof-in-and-intro-form)
     (let* ((left (proof-in-and-intro-form-to-left proof))
	    (right (proof-in-and-intro-form-to-right proof)))
       (if (formula-of-nulltype? (proof-to-formula left))
	   (nc-violations-aux right)
	   (union (nc-violations-aux left)
		  (nc-violations right)))))
    ((proof-in-and-elim-left-form)
     (nc-violations-aux
      (proof-in-and-elim-left-form-to-kernel proof)))
    ((proof-in-and-elim-right-form)
     (nc-violations-aux
      (proof-in-and-elim-right-form-to-kernel proof)))
    ((proof-in-all-intro-form)
     (nc-violations-aux
      (proof-in-all-intro-form-to-kernel proof)))
    ((proof-in-all-elim-form)
     (nc-violations-aux (proof-in-all-elim-form-to-op proof)))
    ((proof-in-allnc-intro-form)
     (let* ((var (proof-in-allnc-intro-form-to-var proof))
	    (kernel (proof-in-allnc-intro-form-to-kernel proof))
	    (prev (nc-violations-aux kernel)))
       (if (member var (proof-to-cvars proof))
	   (adjoin var prev)
	   prev)))
    ((proof-in-allnc-elim-form)
     (nc-violations-aux
      (proof-in-allnc-elim-form-to-op proof)))
    (else (myerror "nc-violations-aux" "proof expected" proof))))

;; h-deg-violations (parallel to nc-violation) gives the list of names
;; of aconsts where a pvar whose tvar shows up in the eterm is
;; substituted by a cterm without computational content.  Reason: this
;; situation generally produces an error when proof-to-extracted-term
;; is applied.  Exceptions: the aconsts receiving a special treatment
;; in proof-to-extracted-term .

(define (h-deg-violations proof)
  (if (formula-of-nulltype? (proof-to-formula proof))
      '()
      (h-deg-violations-aux proof)))

;; In h-deg-violations-aux we can assume that the proven formula has
;; computational content.

(define (h-deg-violations-aux proof)
  (case (tag proof)
    ((proof-in-avar-form) '())
    ((proof-in-aconst-form)
     (let* ((aconst (proof-in-aconst-form-to-aconst proof))
	    (name (aconst-to-name aconst)))
       (if (or (member
		name
		'("Ind" "Cases" "GInd" "Intro" "Elim" "Closure" "Gfp"
		  "Ex-Intro" "Ex-Elim"
		  "Exnc-Intro" "Exnc-Elim" ;obsolete
		  "Eq-Compat"))

	       (apply
		or-op
		(map
		 (lambda (string)
		   (and (<= (string-length string) (string-length name))
			(string=? (substring name 0 (string-length string))
				  string)))
		 '("All-AllPartial" "ExPartial-Ex"
		   "ExclIntro" "ExclElim" "MinPr"))))
	   '()
	   (let* ((uninst-formula (aconst-to-uninst-formula aconst))
		  (tpsubst (aconst-to-tpsubst aconst))
		  (pvars (formula-to-pvars uninst-formula))
		  (et-type (formula-to-et-type uninst-formula))
		  (et-tvars (type-to-tvars et-type))
		  (violating-pvars
		   (list-transform-positive pvars
		     (lambda (pvar)
		       (let ((info (assoc pvar tpsubst)))
			 (and info
			      (member (PVAR-TO-TVAR pvar) et-tvars)
			      (nulltype?
			       (formula-to-et-type
				(cterm-to-formula (cadr info))))))))))
	     (if (pair? violating-pvars)
		 (list (aconst-to-name aconst))
		 '())))))
    ((proof-in-imp-intro-form)
     (h-deg-violations-aux (proof-in-imp-intro-form-to-kernel proof)))
    ((proof-in-imp-elim-form)
     (let* ((op (proof-in-imp-elim-form-to-op proof))
	    (arg (proof-in-imp-elim-form-to-arg proof))
	    (prevop (h-deg-violations-aux op))
	    (prevarg (h-deg-violations arg)))
       (union prevop prevarg)))
    ((proof-in-impnc-intro-form)
     (h-deg-violations-aux (proof-in-impnc-intro-form-to-kernel proof)))
    ((proof-in-impnc-elim-form)
     (h-deg-violations-aux (proof-in-impnc-elim-form-to-op proof)))
    ((proof-in-and-intro-form)
     (let* ((left (proof-in-and-intro-form-to-left proof))
	    (right (proof-in-and-intro-form-to-right proof)))
       (if (formula-of-nulltype? (proof-to-formula left))
	   (h-deg-violations-aux right)
	   (union (h-deg-violations-aux left)
		  (h-deg-violations right)))))
    ((proof-in-and-elim-left-form)
     (h-deg-violations-aux
      (proof-in-and-elim-left-form-to-kernel proof)))
    ((proof-in-and-elim-right-form)
     (h-deg-violations-aux
      (proof-in-and-elim-right-form-to-kernel proof)))
    ((proof-in-all-intro-form)
     (h-deg-violations-aux
      (proof-in-all-intro-form-to-kernel proof)))
    ((proof-in-all-elim-form)
     (h-deg-violations-aux (proof-in-all-elim-form-to-op proof)))
    ((proof-in-allnc-intro-form)
     (h-deg-violations-aux
      (proof-in-allnc-intro-form-to-kernel proof)))
    ((proof-in-allnc-elim-form)
     (h-deg-violations-aux
      (proof-in-allnc-elim-form-to-op proof)))
    (else (myerror "h-deg-violations-aux" "proof expected" proof))))

(define (remove-theorem . strings)
  (define (rthm1 string)
    (if (assoc string THEOREMS)
	(let* ((proof (theorem-name-to-proof string))
	       (formula (unfold-formula (proof-to-formula proof))))
	  (do ((l THEOREMS (cdr l))
	       (res '() (if (string=? string (caar l))
			    res
			    (cons (car l) res))))
	      ((null? l) (set! THEOREMS (reverse res))))
	  (comment "ok, theorem " string " is removed")
	  (if (not (formula-of-nulltype? formula))
	      (remove-program-constant
	       (theorem-or-global-assumption-name-to-pconst-name
		string))))
	(multiline-comment "remove-theorem" "theorem name expected" string)))
  (for-each rthm1 strings))

(define (display-theorems . x)
  (if
   COMMENT-FLAG
   (let ((reduced-thms (if (null? x)
			   THEOREMS
			   (do ((l THEOREMS (cdr l))
				(res '() (if (member (caar l) x)
					     (cons (car l) res)
					     res)))
			       ((null? l) res)))))
     (for-each (lambda (thm)
		 (display (car thm))
		 (display tab)
		 (display-formula
		  (fold-formula (aconst-to-formula (cadr thm))))
		 (newline))
	       reduced-thms))))

;; Global assumptions

;; A global assumption is a special assumption constant.  It provides a
;; proposition whose proof does not concern us presently.  We maintain an
;; association list GLOBAL-ASSUMPTIONS assigning to every name of a
;; global assumption the assumption constant.

;; Format of GLOBAL-ASSUMPTIONS 
;; ((name aconst) ...)

(define (global-assumption-name-to-aconst name)
  (let* ((info (assoc name GLOBAL-ASSUMPTIONS)))
    (if info
	(cadr info)
	(myerror "global-assumption-name-to-aconst"
		 "global assumption name expected" name))))

(define (add-global-assumption string string-or-formula . optional-arity)
  (let ((formula (if (string? string-or-formula)
		     (pf string-or-formula)
		     string-or-formula)))
    (if (not (formula-form? formula))
	(myerror "add-global-assumption" "formula expected" formula))
    (if (pair? (formula-to-free formula))
	(apply myerror "add-global-assumption" "unexpected free variables"
	       (formula-to-free formula)))
    (let* ((fla (unfold-formula formula))
	   (aconst (make-aconst string 'global-assumption fla empty-subst)))
      (if
       (is-used? string formula 'global-assumption)
       *the-non-printing-object*
       (begin
	 (set! GLOBAL-ASSUMPTIONS
	       (cons (list string aconst) GLOBAL-ASSUMPTIONS))
	 (if
	  (not (member string (list "EfqLog" "StabLog" "Efq" "Stab")))
	  (comment
	   "ok, " string " has been added as a new global assumption."))
	 (if (not (formula-of-nulltype? fla))
	     (let* ((pconst-name
		     (theorem-or-global-assumption-name-to-pconst-name string))
		    (type (formula-to-et-type fla))
		    (arity (if (pair? optional-arity)
			       (car optional-arity)
			       0)))
	       (add-program-constant pconst-name type 1 'const arity))))))))

(define aga add-global-assumption)

(define (remove-global-assumption . strings)
  (define (rga1 ga-name)
    (let ((info (assoc ga-name GLOBAL-ASSUMPTIONS)))
      (if info
	  (let* ((aconst (global-assumption-name-to-aconst ga-name))
		 (formula (aconst-to-uninst-formula aconst)))
	    (do ((l GLOBAL-ASSUMPTIONS (cdr l))
		 (res '() (if (string=? ga-name (caar l))
			      res
			      (cons (car l) res))))
		((null? l) (set! GLOBAL-ASSUMPTIONS (reverse res))))
	    (comment "ok, global assumption " ga-name " is removed")
	    (if (not (formula-of-nulltype? formula))
		(remove-program-constant
		 (theorem-or-global-assumption-name-to-pconst-name
		  ga-name))))
	  (myerror "remove-global-assumption" "global assumption expected"
		   ga-name))))
  (for-each rga1 strings))

(define rga remove-global-assumption)

(define (display-global-assumptions . x)
  (if
   COMMENT-FLAG
   (let ((reduced-gas (if (null? x)
			  GLOBAL-ASSUMPTIONS
			  (do ((l GLOBAL-ASSUMPTIONS (cdr l))
			       (res '() (if (member (caar l) x)
					    (cons (car l) res)
					    res)))
			      ((null? l) res)))))
     (for-each (lambda (ga)
		 (display (car ga))
		 (display tab)
		 (display-formula (fold-formula (aconst-to-formula (cadr ga))))
		 (newline))
	       reduced-gas))))

(define (new-global-assumption-name string)
  (new-global-assumption-name-aux string 0))

(define (new-global-assumption-name-aux string n)
  (if (assoc (string-append string (number-to-string n))
	     GLOBAL-ASSUMPTIONS)
      (new-global-assumption-name-aux string (+ n 1))
      (string-append string (number-to-string n))))

;; (search-about string) searches in THEOREMS and GLOBAL-ASSUMPTIONS
;; for all items whose name contains string.

(define (search-about string)
  (let ((thms (list-transform-positive THEOREMS
		(lambda (x) (substring? string (car x)))))
	(gas (list-transform-positive GLOBAL-ASSUMPTIONS
		(lambda (x) (substring? string (car x))))))
    (if (null? thms)
	(comment "No theorems with name containing " string)
	(begin
	  (comment "Theorems with name containing " string)
	  (for-each (lambda (x)
		      (comment (car x))
		      (display-comment
		       (pretty-print-string
			(string-length COMMENT-STRING)
			(- PP-WIDTH (string-length COMMENT-STRING))
			(aconst-to-formula (cadr x))))
		      (newline))
		    thms)))
    (if (null? gas)
	(comment "No global assumptions with name containing " string)
	(begin
	  (comment "Global assumptions with name containing " string)
	  (for-each (lambda (x)
		      (comment (car x))
		      (display-comment
		       (pretty-print-string
			(string-length COMMENT-STRING)
			(- PP-WIDTH (string-length COMMENT-STRING))
			(aconst-to-formula (cadr x))))
		      (newline))
		    gas)))))

