/* ****************************** RmodF ******************************
 * 
 * RmodF(grpring,RqF,R,{flag=0}): Given a group ring grpring, the 
 * group (Rq/F)* in RqmodF and the order R, this computes the 
 * group (R/F)* as a 3-component vector like
 * [ cardinality , vector of cyclic components , generators ]. 
 *
 * If flag is nonzero, the result is a 4-component vector where the 
 * structure is as above and the 4-th component is the HNF H such that
 * (R/F)* = (lambda_1,...,lambda_t)H, where (Rq/F)* is generated by
 * the lambda_i.
 *
 * This is a very slow enumeration and only used for testing. Use Oumf! 
 *
 * ***************************************************************** */

RmodF(grpring , RqF , R , flag=0) = 
{

 local(m,A,H,S,U1,t,RF,ordr,mst,ords);

 m = RqF[2]; RqF = RqF[3];  \\ RqF is a 3-component vector [ cardinality , vector of cyclic components , generators ]

 A = matdiagonal(m);                          \\ Create the diagonal matrix A whose entries are the m_i


 A = concat(A , FindModuleElt(grpring , R , RqF , m)[1] ); \\ Find all Elements mu = lambda_1^x_1 x...x lambda_t^x_t
							   \\ such that mu is in R, 0 <= x_k < m_k, 
							   \\ RqF=[lambda_1,...,lambda_t].

 H = mathnf(A);          \\ Compute the Hermite Normal Form HNF

 S = matsnf(H,1);        \\ Compute the Smith Normal Form [U,V,S]
			 \\ such that S = UHV
 U1 = S[1]^(-1);         \\ U1 = U^(-1) 

 t = length(RqF);        \\ number of generators in (Rq/F)*

 \\ (R/F)* = (lambda_1,...,lambda_t)H, where H is the HNF of A as above
 RF = Vec();

 for(k=1,t,
\\ RF = concat(RF , GrPow(grpring,RqF,H[,k])~ );
   RF = vecput(RF , GrPow(grpring,RqF,H[,k])~ );
 );


 \\ compute the orders of the generators of (R/F)*
 ordr = vector(t);

 for(i=1,t,
   \\ ord(lambda_i^{v_i} = m_i/gcd(v_i,m_i) := m', where m_i = ord(lambda_i)
   mst = vector(t);

   for(j=1,t, mst[j] = m[j] / gcd(U1[j,i],m[j]) );

   ordr[i] = lcmvec(mst);                     \\ the lcm of all elements in the vector mst  

 );

 ords = vector( t , k , ordr[k]/S[3][k,k] );

\\ RF = vector(length(RF),i,RF[,i]~);
 
 if(flag , return([ prod(i=1,length(ords),ords[i]) , ords , RF , H])); \\ if flag != 0, return also the HNF H, such that
								       \\ (R/F)* = (lambda_1,...,lambda_t)H, where
								       \\ (Rq/F)* is generated by the lambda_i, i=1,...t.



 return([ prod(i=1,length(ords),ords[i]) , ords , RF ]);

}

addhelp(RmodF , "RmodF(grpring,RqF,R,{flag=0}): Given a group ring grpring, the group (Rq/F)* in RqmodF and the order R, this computes the group (R/F)* as a 3-component vector like [ cardinality , vector of cyclic components , generators ]. If flag is nonzero, the result is a 4-component vector where the structure is as above and the 4-th component is the HNF H such that (R/F)* = (lambda_1,...,lambda_t)H, where (Rq/F)* is generated by the lambda_i.");




/* ****************************** RmodFGreither ******************************
 *
 * RmodFGreither(grpring,F): This function is the analogon of RqmodF and
 * RmodF, but only for the routine Greither. Here we have a special case
 * of (Rq/F)* and (R/F)*, so the computation can be done a little bit faster.
 *
 * The result are the groups (R/F)* and (Rq/F)* in the following form:
 *
 * [ [ cardinality , vector of cyclic components , generators of (R/F)* , H] ,
 *   [ cardinality , vector of cyclic components , generators of (Rq/F)* ] ]
 *
 * ************************************************************************* */

RmodFGreither(grpring , F) =
{

 local(r,RqF,bid,gen,M,aux,v1,v2,N,A,H,m,S,U1,t,RF,ordr,mst,ords);

 r = length(grpring.Kch);          \\ number of components in the Wedderburn decomposition

 RqF = Vec();                      \\ will contain the generators of (Rq/F)*
 bid = vector(r);

 \\ bid = idealstar(grpring.Kch[1].Krhoabs , F[2][1] , 0);  \\ (O_Krho / f_rho)*

 for(i=1,r,
   bid[i] = idealstar(grpring.Kch[i].Krhoabs , F[2][i] , 2);  \\ (O_Krho / f_rho)*
   if(bid[i][2][2] == [], bid[i][2][2] = [1]; bid[i][2][3] = [1]);    
 );


 \\ no generator found                            \\ flag = 0, computes the structure of (O_Krho/frho)* 
 \\ if(bid[2] == [], bid[2] = [1]; bid[3] = [1]);    \\ as a 3-component vector

 gen = bid[1][2][3];

 M = matrix(2*length(gen) , length(gen));

 for(i=1,length(gen),  \\ bid[1][2][3] = generators

   aux = nfbasistoalg(grpring.Kch[1].Krhoabs , gen[i]); 

   v1 =  [rnfeltabstorel(grpring.Kch[1].Krho , aux) , 1];
   v2 =  [ 1 , rnfeltabstorel(grpring.Kch[2].Krho , aux)];    
   v1 = PhiInv(grpring , v1); v2 = PhiInv(grpring , v2);

\\ RqF = concat(RqF , v1~); RqF = concat(RqF , v2~);
   RqF = vecput(RqF , v1~); RqF = vecput(RqF , v2~);

   M[2*i-1,i] = 1;  M[2*i,i] = 1;

 );


 N = matrix(2*length(gen) , 2*length(gen));

 for(i=1,length(bid[1][2][2]),
   N[2*i-1,2*i-1] = bid[1][2][2][i]; N[2*i,2*i] = bid[1][2][2][i];
 );

 A = concat(M , N);

 H = mathnf(A);                                     \\ HNF
 m = vector(2*length(gen) , i , N[i,i]);

 S = matsnf(H,1);                                   \\ Compute the Smith Normal Form [U,V,S]
						    \\ such that S = UHV
 U1 = S[1]^(-1);                                    \\ U1 = U^(-1) 

 t = length(RqF);                                   \\ number of generators in (Rq/F)*

 \\ (R/F)* = (lambda_1,...,lambda_t)H, where H is the HNF of A as above
 RF = Vec();

 for(k=1,t,
\\ RF = concat(RF , GrPow(grpring,RqF,H[,k])~ );
   RF = vecput(RF , GrPow(grpring,RqF,H[,k])~ );
 );

 \\ compute the orders
 ordr=vector(t);

 for(i=1,t,
   \\ ord(lambda_i^{v_i} = m_i/gcd(v_i,m_i) := m', where m_i = ord(lambda_i)
   mst=vector(t);

   for(j=1,t, mst[j]=m[j]/gcd(U1[j,i],m[j]));

   ordr[i] = lcmvec(mst);
 );

 ords = vector(t,k,ordr[k]/S[3][k,k]);

\\ RF = vector(length(RF),i,RF[,i]~);

 return([ [ prod(i=1,length(ords),ords[i]) , ords , RF , H ] , [ prod(i=1,length(m),m[i]) , m , RqF , bid ] ]);


}

addhelp(RmodFGreither , "RmodFGreither(grpring,F): This function is used by Greither (see Examples.gp) for a faster computation of (R/F)* and (Rq/F)*.");




/* ****************************** RFQuot ******************************
 *
 * RFQuot(grpring,RqF,RF,{flag=0}: Computes (Rq/F)* / (R/F)*
 *
 * ****************************************************************** */
RFQuot(grpring , RqF , RF , flag=0) =
{

 local(n,H,S,U1,d,s);

\\ n = length(RqF[2]);        \\ RqF and RF are 3-component vectors like
			    \\ [ cardinality , vector of cyclic components , generators , (H) ]

 n = length(RF[2]);

 H = RF[4];                 \\ remember, H is the HNF such that
			    \\ (R/F)* = (lambda_1,...,lambda_t)H, 
			    \\ where (Rq/F)* is generated by the lambda_i.

 S = matsnf(H , 1);         \\ compute the smith normal form [U,V,S], such that S = UHV

 U1 = S[1]^(-1);            \\ U1 = U^(-1)

 d = Vec();

 for(i=1,n,
\\   d = concat(d, GrPow(grpring , RqF[3] , U1[,i])~); \\ the generators of (Rq/F)*/(R/F)*
\\   u1 = vector(length(RqF[2]));
\\   for(j=1,n,  u1[j] = U1[j,i]);
   d = vecput(d, GrPow(grpring , RqF[3] , U1[,i])~);
 );

 s = vector(length(S[3]) , k , S[3][k,k]);           \\ the orders of the generators


 if(flag, return( [ prod(i=1,length(s),s[i]) , s , d , U1 ] ));

 return( [ prod(i=1,length(s),s[i]) , s , d ] );

}



addhelp(RFQuot , "RFQuot(grpring,RqF,RF,{flag=0}): Given a group ring grpring in GrInit, the group (Rq/F)* in RqmodF and (R/F)* in RmodF, this computes the quotient (Rq/F)* / (R/F)* as 3-component vector [ cardinality , vector of cyclic components , generators ]. If flag is non-zero the result is more complex and only used by PicIsPrincipal.");










/* ****************************** Auxiliary functions ******************************
 * 
 *                              - NOT FOR PERSONAL USE -
 * 
 * ******************************************************************************* */

/* ****************************** RayClassGroup ******************************
 *
 * RayClassGroup(grpring,F,{flag=1}): This function computes all ray class
 * groups cl_frho(Krho), such that Cl_F(Rq) = (+)_rho cl_frho(Krho). 
 *
 * This is necessary for the computation of Pic(R) in Picard. The result is
 * 2-component vector [ vector of orders in rcgp.clgp , all ray class groups ], 
 * if flag is zero.
 *
 * Is flag is non-zero, the result is
 *
 * [ vector of orders in rcgp.clgp , generators of rcgp.clgp , all ray class groups ]
 *
 *
 * For the computation of Pic(R) we only need the cl_frho(Krho), so we don't
 * need to compute a representation ov Cl_F(Rq). 
 *
 * ************************************************************************* */

\\ not for personal use
RayClassGroup(grpring , F , flag=1) =
{

 local(r,gen,ord,rcgp,clgp);

 r = length(grpring.Kch);                     \\ number of components in the Wedderburn decomposition

 gen = vector(r); ord = vector(r); rcgp = vector(r);

 \\ compute the ray class groups to the conductors frho in F[2]
 if(flag, 
   for(i=1,r,                                
     rcgp[i] = bnrinit(grpring.Kch[i].Krhoabs, F[2][i], 1);  \\ flag 1: for bnrisprincipal in some further functions
     clgp = rcgp[i].clgp;
     if(clgp[2] == [], clgp[2] = [1]; clgp[3] = [1]);   \\ if there's no generator (group is trivial), take 1
     ord[i] = clgp[2]; gen[i] = clgp[3];                \\ the orders and the generators of the ray class group
   );

 , \\ else

   for(i=1,r,                                   
     rcgp[i] = bnrinit(grpring.Kch[i].Krhoabs, F[2][i], 0);  \\ flag 0: we don't need the generators
     clgp = rcgp[i].clgp;
     if(clgp[2] == [], clgp[2] = [1]);    \\ if there's no generator (group is trivial), take 1
     ord[i] = clgp[2];
   );

 );

 if(flag, return([  ord , gen , rcgp ]));

 return([ ord , rcgp ]);

}

addhelp(RayClassGroup , "RayClassGroup(grpring,F,{flag=1}): This function is used by Picard to compute the orders and the generators of cl_frho(Krho) in Cl_F(Rq) = (+)_rho cl_frho(Krho). The result is a 3-component vector [ vector of orders in rcgp.clgp , vector of generators in rcgp.clgp , all ray class group ]. Is flag=0, only the orders and the ray class groups are computed.");





/* ****************************** zz_rcgpIsPrincipal ******************************
 *
 * zz_rcgpIsPrincipal(grpring,rcgp,cRq,{flag=1}): This routine solves the
 * discrete logarithm of cRq in the components cl_frho(Krho) in 
 * Cl_F(Rq) = (+) cl_frho(Krho). 
 *
 * rcgp must be given by RayClassGroup(,,1)[3]. If flag=1, solve the refined
 * discrete logarithm, otherwise (flag=0) compute only the exponents on the 
 * ray class group generators.
 *
 * ****************************************************************************** */

\\ not for personal use
zz_rcgpIsPrincipal(grpring , rcgp , cRq , flag=1) =
{

 local(r,n,cRqrho,v,zeta1,phirho,b);

 r = length(grpring.Kch);

 n = length(cRq);

 \\ to do a principal ideal test for cRq we work in the Wedderburn decomposition
 cRqrho = vector(r);  v = vector(r);  zeta1 = vector(r);  \\ v will be the vector of exponents on group generators 
							  \\ and zeta1 is the generator of the resulting principal ideal

 \\ we will compute cRq = zeta1 * a_1^{v_1} * ... * a_m^{v_m}
 for(i=1,r,

   for(j=1,n,          

     phirho = Phirho(grpring , i , cRq[,j]);    \\ to the Wedderburn decomposition
     phirho = rnfeltreltoabs(grpring.Kch[i].Krho , phirho);
     cRqrho[i]  = idealadd( grpring.Kch[i].Krhoabs , cRqrho[i] , phirho );

   );


   b = bnrisprincipal(rcgp[i] , cRqrho[i] , flag);  \\ now we can do a principal ideal test in Cl_F(Rq), component wise

   if(!flag, b = [b]);

   if(b[1] == []~, b[1] = [0]~);    \\ if there's no exponent, take 0

   v[i] = b[1]; 

   if(flag,
     zeta1[i] = nfbasistoalg(grpring.Kch[i].Krhoabs , b[2]);
     zeta1[i] = rnfeltabstorel(grpring.Kch[i].Krho , zeta1[i]);
   );

 );

 v = concat(v);                      \\ we have all exponents

 if(flag, 
   zeta1 = PhiInv(grpring,zeta1);    \\ back to the group ring
   return([ v , zeta1 ]);
 );

 return(v);


}

addhelp(zz_rcgpIsPrincipal , "zz_rcgpIsPrincipal(grpring,rcgp,cRq,{flag=1}): Not for personal use! This function is used by Picard to solve the discrete logarithm of cRq in Cl_F(Rq) = (+)_rho cl_frho(Krho). If flag=1, solve the refined discrete logarithm, otherwise (flag=0) compute only the exponents on the ray class group generators. rcgp must be given by RayClassGroup(,,1)[3].");





/* ****************************** zz_Pic_MatrixD ******************************
 *
 * zz_Pic_MatriD(ord,V1): Not for personal use! This function is used by 
 * Picard to compute a matrix D such that 
 *
 * D = 0 ( mod [ord(a_1),...,ord(a_m)]~ ) and C = V^-1 + D > 0. 
 * V comes from the SNF S = VHU, see Picard(). 
 *
 * We need this matrix D to compute with integral ideals.
 *
 * V1 = V^-1, ord = [ord(a_1),...,ord(a_m)], Cl_F(Rq) = <a_1> x ... x <a_m>.
 *
 * ************************************************************************** */

\\ not for personal use
zz_Pic_MatrixD(ord , V1) =
{

 local(m,D,u,k);

 ord = concat(ord); m = length(ord); D = matrix(m,m);

 for(i=1,m,

   for(j=1,m,

     u = V1[i,j]; k = 0;

     while( u < 0, k++; u += ord[i]); \\ find a d, such that V1[i,j] + d >= 0, where d = 0 (mod ord[i])
     D[i,j] = k*ord[i]

   ); 

 );

 return(D);

}

addhelp(zz_Pic_MatrixD , "zz_Pic_MatrixD(ord,V1): Not for personal use! This function is used by Picard to compute a matrix D such that D = 0 ( mod [ord(a_1),...,ord(a_m)]~ ).");





/* ****************************** zz_Pic_bq_GetMatrixCol ******************************
 *
 * zz_Pic_bq_GetMatrixCol(M,col,beg,end): Given a matrix M. This extracts in the col-th
 * column the rows beg until end. 
 *
 * ********************************************************************************** */

\\ not for personal use
zz_Pic_bq_GetMatrixCol(M , col , beg , end) =
{

 local(l,v);

 l = end - beg + 1; v = vector(l,k,beg++ - 1);

 return(vecextract(M[,col],v));

}

addhelp(zz_Pic_bq_GetMatrixCol , "zz_Pic_bq_GetMatrixCol(M,col,beg,end): Given a matrix M. This extracts in the col-th column the rows beg until end.");





/* ****************************** zz_Pic_epsilon ******************************
 *
 * zz_Pic_epsilon(grpring,rcgp,RF): 
 *
 * ************************************************************************** */

\\ not for personal use
zz_Pic_epsilon(grpring , rcgp , RF , flag=0) =
{

 local(r,t,eps,A,phi,mu,v,aux,muinv,phiinv);

 r = length(grpring.Kch);                     \\ number of K_rho components
 t = length(RF[3]);                           \\ RF = [ cardinality , vector of cyclic components , generators ]

 eps = Mat(); A = Mat();

 for(i=1,t,

\\phi = Phi(grpring,RF[3][,i]);              \\ Phi(lambda) is in (+)_rho K_rho, lambda in (R/F)*
   phi = Phi(grpring,RF[3][i]);

   mu = vector(r);
   v = []~;

   for(j=1,r,                                  

     aux = rnfeltreltoabs(grpring.Kch[j].Krho , phi[j]); \\ for bnrisprincipal we need absolute components
     aux = bnrisprincipal(rcgp[j] , aux, flag);          \\ solve the discrete logarithm

     if(flag == 0, aux = [aux]);              \\ technical reasons from bnrisprincipal(,,0) !

     if(aux[1] == []~, aux[1] = [0]~);        \\ if there is no exponent of aux on the ray class group 
					      \\ generatros, set them to 0

     v = concat( v, aux[1] ) ;                \\ aux[1] are the exponents on the ray class group generators

     if(flag == 1, 
       muinv = nfbasistoalg(rcgp[j] , aux[2])^(-1); \\ aux[2] is given on the integral basis
       mu[j] = rnfeltabstorel(grpring.Kch[j].Krho , muinv);
     );

   );

   A = concat(A,v);

   if(flag == 1,                              \\ if flag=1 we want to solve the refined DL
     phiinv = PhiInv(grpring,mu);             \\ back to the group ring
\\     eps = concat(eps, GrMult(grpring,RF[3][,i],phiinv)~);
     eps = concat(eps, GrMult(grpring,RF[3][i],phiinv)~);
   );

 ); 

 if(flag == 1, return([ A, eps ]));

 return(A);

} 

addhelp(zz_Pic_epsilon , "zz_Pic_epsilon(grpring,rcgp,RF,{flag=0}):");





/* ****************************** zz_Pic_alpha ******************************
 *
 * zz_Pic_alpha(grpring,rcgp,gen,ord): Given a group ring grpring and a 
 * vector rcgp with the ray class groups of the Wedderburn-decomposition as
 * entries, cl_frho(Krho) = <gen_1> x ... x <gen_m>, this computes the 
 * alphas in a_i^{ord(a_i)} = alpha_iRq in the group ring. 
 * C_F(Rq) = <a_1> x ... x <a_m>.
 * 
 * ************************************************************************ */

\\ not for personal use
zz_Pic_alpha(grpring , rcgp , gen , ord) =
{

 local(r,alpha,v,apow,aux);

 r = length(grpring.Kch);                     \\ number of components in the Wedderburn-decomposition

 alpha = Mat();                               \\ will contain the alpha in a_i^{ord(a_i)) = alpha_iRq

 for(i=1,r,

   v = vector(r,k,1);                         \\ v only for PhiInv

   for(j=1,length(gen[i]),                    \\ take each generator in the i-th component

     apow = idealpow(rcgp[i] , gen[i][j] , ord[i][j]);  \\ rcgp is a vector with the r-'rcgp'-components

     aux = bnrisprincipal(rcgp[i] , apow);    \\ solve the discrete logarithm

     aux = nfbasistoalg(rcgp[i] , aux[2]);     

     v[i] = rnfeltabstorel(grpring.Kch[i].Krho , aux); \\ PhiInv need relative components

     alpha = concat( alpha, PhiInv(grpring,v)~ );      \\ back to the group ring

   );
 );

 return(alpha);

}

addhelp(zz_Pic_alpha , "zz_Pic_alpha(grpring,rcgp,gen,ord): Not for personal use! Computes the alphas in a_i^{ord(a_i)} = alpha_iRq, where C_F(Rq) = <a_1> x ... x <a_m>. ");





/* ****************************** zz_Pic_beta ******************************
 *
 * zz_Pic_beta(grpring , alep , W , U , g1 , s): Computes the betas such 
 * that bq_i^{s_i} = beta_iRq, i=1,...,m in the group ring grpring.
 *
 * We know 
 *
 * (bq_1^{s_1},...,bq_m^{s_m}) = (a_1,...,a_m)CS =
 * = (alpha_1Rq,...,alpha_mRq,epsilon_1Rq,...,epsilon_tRq)W*[0,0;0,U] + 
 *   + (0,...,0,gamma_1^{s_1}Rq,...,gamma_m^{s_m}Rq).
 *
 * alep = [alpha,epsilon], g1 = [gamma_1,...,gamma_m], s = [s_1,...,s_m].
 *
 * *********************************************************************** */

\\ not for personal use
zz_Pic_beta(grpring , alep , W , U , g1 , s) = 
{

 local(l,m,beta,M,rc,row,col,U1,v,WU,aux);

 \\ W is a m x m matrix
 l = length(W); m = length(s); beta = Mat(); M = Mat();

 rc = matsize(U); row = rc[1]; col = rc[2]; 

 g1 = GrPow(grpring , g1 , s , 1);            \\ compute the gamma_i^{s_i} in 
					      \\ g1 = [ gamma_1^{s_1},...,gamma_m^{s_m} ]
					      \\ flag 1 returns the vector of the gamma_i^{s_i}

 U1 = matrix(l , l-col);                      \\ fill the matrix U up with nulls, we 
					      \\ need U = [0,0;0,U]
 for(i=1,col,
   v = concat(vectorv(l-row) , U[,i]);        \\ concat the columns
   U1 = concat(U1,v);                         \\ than U1 and the column
 );

 WU = W*U1;                                   \\ now we can multiply W by U

 for(i=l-col+1,l,                             \\ we only need the last m entries
   aux = GrPow(grpring,alep,WU[,i]~)~;        
   M = concat(M,aux);
 );

 for(i=1,m,                                   \\ add the 'aleps' and the 'gammas' and we've done
   beta = concat( beta, GrMult(grpring ,M[,i] , g1[i])~ );
 );

 return(beta);

}

addhelp(zz_Pic_beta , "zz_Pic_beta(grpring,alep,W,U,g1,s): Not for personal use! Computes the betas such that bq_i^{s_i} = beta_iRq, i=1,...,m in the group ring grpring. Only necessary for the routine Picard.");





/* ****************************** zz_Pic_bq ******************************
 *
 * zz_Pic_bq(grpring,rcgp,gen,C): Not for personal use! This functions is
 * used by the routine Picard to compute some technical data to compute 
 * the Picard group. 
 *
 * grpring is the group ring, rcgp is a vector with all ray class groups 
 * such that Cl_F(Rq) = cl_f1(K_1) (+) ... (+) cl_fr(K_r). 
 *
 * Than we compute (bq_1,...,bq_m) = (a_1,...,a_m)C where all informations, 
 * the a_i,i=1,...,m and the matrix C is given by the function Picard. 
 * The a_i are the generators of the ray class group Cl_F(Rq).
 *
 * ******************************************************************************************* */

\\ not for personal use
zz_Pic_bq(grpring , rcgp , gen , C) = 
{

 local(r,m,n,yrho,t,y1,d,bq,M,v);

 r = length(grpring.Kch); m = length(concat(gen)); n = length(C); yrho = vector(r);

 t = vector(r+1); t[1] = 1;

 for(i=1,r,
   t[i+1] = t[i] + length(gen[i]);
 );

 y1 = vector(m);

 for(i=1,n,
   for(j=1,r,

     d = zz_Pic_bq_GetMatrixCol(C , i , t[j] , t[j+1]-1)~; \\ get special rows out of the i-th column in C

     yrho[j] = IdealPow(rcgp[j] , gen[j] , d);   \\  

     yrho[j] = IdealAbsToRel(grpring.nfd , grpring.Kch[j].Krhoabs , grpring.Kch[j].Krho , yrho[j]); 
   );

   y1[i] = yrho;

 );

 bq = vector(m); 

 for(i=1,m,

   M = Mat();

   for(j=1,r,

     v = vector(r);

     for(k=1,length(y1[i][j]),

       v[j] = y1[i][j][k];

       M = concat( M, PhiInv(grpring , v)~ );

     );

   );

   bq[i] = M;

 );

 return(bq);                                  \\ now (bq_1,...,bq_m) = (a_1,...,a_m)C

}

addhelp(zz_Pic_bq , "zz_Pic_bq(grpring,rcgp,gen,C): Not for personal use! This functions is used by the routine Picard to compute some technical data to compute the Picard group.");





\\ not for personal use
zz_rcgpIdealToGrElt(grpring , ideal) = 
{

local(r,GE,ID,w,v);

r = length(grpring.Kch);

GE = Vec();

for(i=1,r,

  ID = Mat();
  for(k=1,length(ideal[i]),
     w =  IdealAbsToRel(grpring.nfd , grpring.Kch[i].Krhoabs , grpring.Kch[i].Krho , ideal[i][k]);

     v = vector(r);

     for(j=1,length(w),
       v[i] = w[j];
       phiinv = PhiInv(grpring , v)~;
       ID = concat(ID , phiinv);
     );

     for(l=1,r,

       if(l!=i, 
         w = IdealAbsToRel(grpring.nfd , grpring.Kch[l].Krhoabs , grpring.Kch[l].Krho , 1);
         v = vector(r);
 
         for(j=1,length(w),
           v[l] = w[j];
           phiinv = PhiInv(grpring , v)~;
           ID = concat(ID , phiinv);
         );

       );
  
      );

  );

  GE = concat(GE , [ID]);

);

return(GE);

}



















