\\  -*-gp-script-*-

/* ****************************** Picard.gp ******************************
 * 
 * Picard.gp: This file contains routines to compute the class group 
 * Pic(R) of a order R in a group ring and to solve the discrete logarithm
 * problem in Pic(R). These functions are
 *
 *      - Picard(grpring,R,{flag=0},{greither=0})
 *      - PicIsPrincipal(grpring,CR,a,{flag=1})
 *      - RqmodF(grpring,F)
 *      - RmodF(grpring,RqF,R,{flag=0}) 
 *      - RmodFGreither(grpring,F)
 *      - GroupQuot(grpring,RqF,RF,F,flag=0)
 *      - RFQuot(grpring,RqF,RF,{flag=0})
 *
 *
 * Auxiliary functions: NOT FOR PERSONAL USE
 * 
 *      - RayClassGroup(grpring,F,{flag=1}) =
 *      - zz_rcgpIsPrincipal(grpring,rcgp,cRq,{flag=1})
 *      - zz_Pic_MatrixD(ord,V1) 
 *      - zz_Pic_bq_GetMatrixCol(M,col,beg,end) 
 *      - zz_Pic_epsilon(grpring,rcgp,RF,{flag=0}) 
 *      - zz_Pic_alpha(grpring,rcgp,gen,ord)
 *      - zz_Pic_beta(grpring,alep,W,U,g1,s) 
 *      - zz_Pic_bq(grpring,rcgp,gen,C) 
 *      - zz_rcgpIdealToGrElt(grpring,ideal)
 *
 * time-stamp: <Tue, Nov 02, 2004 - 15:44>
 *
 * author:     markus endres (mail: parigpo@mendres.org)
 *             werner bley (mail: bley@math.uni-augsburg.de)
 *
 * ********************************************************************* */


/* ****************************** Class group structure ******************************
 *
 * The structure of the class group computed by Picard(...) is a 2-component vector 
 * like [ prod_{i=1}^n s_i , [s_1,..,s_n] , B ]. If flag is nonzero there are a lot 
 * of technical informations for solving the discrete logartihm in Pic(R). 
 *
 * ********************************************************************************* */

\\ two possible structures of the class group
PicardType1 = [ "t_INT" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_MAT" , "t_VEC" , "t_MAT" , "t_MAT" , "t_MAT" ];

PicardType2 = [ "t_INT" , "t_VEC" , "t_INT" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_VEC" , "t_MAT" , "t_VEC" , "t_INT" , "t_INT" , "t_INT" ];


\\ The structure of the class group computed by Picard(,,1)
x.grporder  = x[1];                          \\ the group order
x.ord       = x[2];                          \\ the orders s
x.B         = x[3];                          \\ the generators of Pic(R)
x.R         = x[4];                          \\ the module R from Picard(grpring,R)
x.F         = x[5];                          \\ the concdutor F
x.RqF       = x[6];                          \\ the group (Rq/F)*
x.RF        = x[7];                          \\ the group (R/F)*
x.H         = x[8];                          \\ the Hermite Normalform
x.S         = x[9];                          \\ the Smith Normalform of H
x.D         = x[10];                         \\ the matrix D
x.rcgp      = x[11];                         \\ the ray class groups
x.beta      = x[12];                         \\ the betas
x.alep      = x[13];                         \\ alpha and epsilons
x.g1        = x[14];                         \\ the gammas



print_level = 0;


/* ****************************** Picard ******************************
 * 
 * Picard(grpring,R,{flag=0},{greither=0}): Given a group ring grpring
 * in GrInit and an order R in the group ring. This functions computes
 * the class group of R. The result is a 3-component vector as all 
 * other finite abelian groups:
 *
 * [ cardinality , vector of cyclic components , generators ].
 *
 * This will computed by default, flag=0. If flag=1, compute some 
 * additional technical data to solve the refined discrete logarithm
 * in Pic(R) (PicIsPrincipal(,,1)). If flag=2, this computes enough
 * data to compute PicIsPrincipal(,,0), only the exponents on the
 * picard group generators. 
 *
 * If greither is nonzero, a special function is used to compute
 * (Rq/F)* and (R/F)*. This should only be used for computing
 * special Greither examples. See the function Greither(...) in 
 * Examples.gp. 
 * 
 * ****************************************************************** */


Picard(grpring , R , flag=0 , greither=0) =
{

 \\ local(RmodFGrp,r,F,RqF,RF,rcgp,ord,gen,m,A,eps,H,S,U1,s,D,alpha,cord,g1,aux,alep,beta,bq,B,sp);
 local(RmodFGrp,r,F,RqF,RF,rcgp,ord,gen,m,A,eps,H,S,U1,s,D,alpha,cord,g1,aux,alep,beta,bq,B,sp);


 if( StructType(R) != ModuleInitType, R = ModuleInit(grpring,R)); \\ R in ModuleInitType 

 r = length(grpring.Kch);                     \\ number of components


 F = ModuleConductor(grpring , grpring.MaxOrd , R , 1);\\ compute the conductor F of the module R
						       \\ and the ideal f_rho in O_{K_rho}
 if(greither,   \\ for faster computations, only for Greithers examples
   
   RmodFGrp = RmodFGreither(grpring,F); 
   RF = RmodFGrp[1]; RqF = RmodFGrp[2]; RF=vecput(RF,1); 

  , \\ else

   \\ all groups are given as a 3-component vector 
   \\ [ cardinality , vector of cyclic components , generatos ]  

   RqF = RqmodF(grpring , F);             \\ compute the group (Rq/F)* and
                                          \\ (R/F)*, flag = 1:
					  \\ return also the HNF H, such that
					  \\ (R/F)* = (lambda_1,...,lambda_t)H, where
					  \\ (Rq/F)* is generated by the lambda_i, i=1,...t.
   if (print_level > 0, print1("Computation of (R/F)^*"));
   RF = Oumf(grpring, R, F, 1);
   if (print_level > 0, print(" *** done *** \n"));
   RF=vecput(RF,0); \\ 0 is a greither indicater, i.e. don't use the greither special implementation
 );

 
  if(flag != 2,                                \\ if flag != 2, we need the generators of cl_frho(K_rho)

   rcgp = RayClassGroup(grpring , F , 1);     \\ compute the ray class group Cl_F(Rq).
					      \\ flag 1: return also all the ray 
   ord = rcgp[1];                             \\ class groups cl_frho(Krho) in a vector.
   gen = rcgp[2];                             \\ we only need the components cl_frho(Krho)
   rcgp = rcgp[3];                            \\ so the result is a vector [ ord , gen , rcgp ]
					      \\ from bnrinit(Krhoabs,frho);
  , \\ else we don't need the generators
   
   if (print_level > 0, print1("Coputation of rcgp"));
   rcgp = RayClassGroup(grpring , F , 0);     \\ compute the ray class group Cl_F(Rq).
   if (print_level > 0, print(" *** done *** \n"));
   ord = rcgp[1]; rcgp = rcgp[2]

 );

 m = length( concat(ord) );                   \\ number of generators


 \\ epsilons

 if(flag != 2,   \\ flag == 1, we want to solve the refined discrete logarithm problem in Pic(R)

   A = zz_Pic_epsilon(grpring , rcgp , RF , 1);   \\ compute the epsilon's in
						  \\ (eps_1Rq,...,eps_tRq) = (a_1,...,a_m)A
   eps = A[2]; A = A[1];                          \\ result was [ A , eps ] 

  , \\ else flag == 0 || flag == 2, than we only need the matrix A
   
   if (print_level > 0, print1("Computation of epsilons"));
   A = zz_Pic_epsilon(grpring , rcgp , RF , 0);   \\ compute the epsilon's in
   if (print_level > 0, print(" *** done *** \n"));

 );



 A = concat( matdiagonal(concat(ord)) , A );

 \\ compute the hermite normal form HNF
 H = mathnf(A , 1);                           \\ mathnf(x,1) outputs a two-component row  vector  [H,W], 
					      \\ where H is the upper triangular  hermite  normal form  
					      \\ of  A and W is the unimodular transformation matrix 
					      \\ such that AW = [0|H].

 \\ compute the smith normal form SNF
 S = matsnf(H[1], 1);                         \\ compute the smith normal form [U,V,S], such that S = UHV.
 U1 = S[1]^(-1);                              \\ U1 = U^(-1)

 s = vector(length(S[3]) , k , S[3][k,k]);    \\ the orders of the generators of Pic(R)

 D = zz_Pic_MatrixD(ord , U1);                \\ we want to use only integral ideals, so compute 
 C = U1 + D;                                  \\ C = U1 + D with  D = 0 (mod [ord(a_1), ..., ord(a_m)]~);


 \\ alphas
 if(flag == 1,   \\ if flag == 1, we have to compute some more data
		 \\ to solve the refined discrete logarithm problem in Pic(R)

   alpha = zz_Pic_alpha(grpring , rcgp , gen , ord); \\ compute the alphas sucht that a_i^(ord(a_i) = alpha_iRq)

   cord = concat(ord); g1 = Mat();            \\ g1 are the gammas as below

   for(i=1,m,                                 \\ compute gammas such that
					      \\ (a_1,...,a_m)D = (g_1Rq,...,g_mRq), g_i in K[G]
     aux = vector(m, k, D[k,i]/cord[k]);

     g1 = concat( g1 , GrPow(grpring , alpha , aux)~ );

   );

   alep = concat(alpha , eps);                 \\ contains the alphas and the epsilons

   beta = zz_Pic_beta(grpring , alep , H[2] , S[2] , g1 , s); \\ compute the betas such that
                                                              \\  bq_i^{s_i} = beta_iRq, i=1,...,m.

 );


 \\ generators B
 if(flag != 2,  \\ compute the generators of Pic(R)  

  bq = zz_Pic_bq(grpring , rcgp , gen , C);  \\ compute (bq_1,...,bq_m) = (a_1,...,a_m)C, 
					     \\ where C = U1 + D as above

  B = vector(m);      \\ will contain the generators of Pic(R)

  for(i=1,m,
    B[i] = ModuleSection(grpring , ModuleInit(grpring , bq[i]) , R);
  );

 );

\\ sp = prod(i=1,m, s[i]);  \\ the order of the Picard group


\\ if S the identity matrix
 if( s == vector(length(s),k,1), 
   B = [R]; sp = 1; s = [1];
  ,  \\ else

   g = searchIndex(s);

   s = vector(g,k,s[k]);
   sp = prod(i=1,g, s[i]);  \\ the order of the Picard group   
   if(flag != 2, B = vector(g,k,B[k]));
);

 \\ flag == 0, give only the structure of Pic(R)
 if(flag == 0 , return([ sp , s , B , R , F])); 

 \\ flag == 1, solve the refined discrete logarithm in Pic(R)
 if(flag == 1 , return([ sp , s , B , R , F , RqF , RF , H , S , D , rcgp , beta , alep , g1 ]));   

 \\ flag == 2, solve the discrete logarithm, we don't need the generators in B
 if(flag == 2 , return([ sp , s , 0 , R , F , RqF , RF , H , S , D , rcgp , 0 , 0 , 0 ]));

}

addhelp(Picard , "Picard(grpring,R,{flag=0},{greither=0}): Given a group ring grpring in GrInit and an order R in the group ring. This functions computes the class group of R. The result is a 3-component vector as all other finite abelian groups: [ cardinality , vector of cyclic components , generators ]. If flag=1, compute some additional technical data to solve the refined discrete logarithm in Pic(R) (PicIsPrincipal(,,1)). If flag=2, this computes enough data to compute PicIsPrincipal(,,0), only the exponents on the picard group generators. If greither is nonzero, a special function is used to compute (Rq/F)* and (R/F)*. This should only be used for computing Greither examples. See the function Greither(...) in Examples.gp.");



\\ search largest index in s not equalt to 1
searchIndex(s) =
{

local(g);

for(i=1,length(s),
  if(s[i] == 1, return(i-1));
);

return(length(s));

}


/* ****************************** Findz ******************************
 *
 * Findz(grpring,M,d,s, F, {flag=0}): Given a group ring grpring in GrInit,
 * modules M and N, a n-component vector d of elements in the group ring, a 
 * n-component vector s of integers and an ideal F. This routine finds (an) element(s)
 * z = d_1^{c_1}...d_n^{c_n}, 0 <= c_i < s_i, such that zM < N. It is assumed that
 * z is defined modulo F.
 * 
 * If flag is omitted or zero the result are all elements z with the above
 * feature. If flag is nonzero search only one element which is included in M.
 * 
 * The result is a 2-component vector where the first component is a matrix A
 * which contains the exponents c_i (column of A) as above. The second one is
 * the vector of the elements z. 
 *
 * ************************************************************************* */

Findz(grpring , M , N, d , s , F, flag=1) = 
{
 local(n,c,z,A,mu,X, count);

 n = length(d);

 if(n != length(s), error("***   incomaptible vector length"));

 c = InitMultiCounter( vector(n) , s-vector(n,k,1) );    \\ initialize multi counter

 z = Vec();                             \\ z will contain the elements which are in M
 A = Mat();                             \\ A will contain the exponents c such that 
					\\ z = d_1^{c_1}...d_n^{c_n} is an element of M
 

\\ while(c,                               \\ c[1] contains the integers c_i, sucht that 0 <= c_i < s_i
  count = 0;
  if (print_level > 0, print1("count = ", count));
  until(!c,
   count = count + 1;
   \\ mu = GrPowModIdeal(grpring , d , c[1],F);      \\ compute z = d_1^{c_1}...d_n^{c_n}
   mu = GrPow(grpring , d , c[1]);
   mu = RedModIdeal(grpring, mu~, F);
   if (count % 50 == 0 && print_level > 0, print1("  ", count));
   X = ModuleScalarProd(grpring, mu, M);

   if( ModuleIncl(grpring , X , N),     \\ if zM < N then

     A = concat(A,c[1]~);               \\ write the c_i in a column of A

     z = concat(z , mu);                \\ z contains all elements such that
					\\ z = d_1^{c_1}...d_n^{c_n}, 0 <= c_i < s_i, z in M     

     if(flag, return( [ A , z ]));      \\ we search only one element with the above feature.
					\\ so, if we have one return this element
   );

   c = IncMultiCounter(c);              \\ increase MultiCounter
 );

 return( [ A , z ]);

}






/* ****************************** PicIsPrincipal ******************************
 *
 * PicIsPrincipal(grpring,CR,a,{flag=1}): Given a group ring grpring, the
 * class group CR in Picard and a locally free R-module a. This solves the 
 * discrete logarithm problem in the class group Pic(R). If flag=1, solve the
 * refined discrete logarithm problem, else (flag=0) compute only the exponents 
 * on the generators of Pic(R).
 *
 * ************************************************************************** */

PicIsPrincipal(grpring , CR , a , flag=1) =
{

 local(i,t,G,r,n,arho,arhoint,d,phirho,hnf,yrho,aux,ya,cq,as,z1,y1,zs,c,cRq,v,zeta1,m,r1,v1,betav1,g1y1inv,mu,yzs,nu,var, id); 

 var = variable(grpring.Kch[1].Krhoabs.pol);

 if(flag == 0 && ( (t=StructType(CR)) != PicardType1 && t != PicardType2 ),
   error("please apply Picard(,,2) or Picard(,,1) and not Picard(,,0)!"));

 if(flag == 1 && StructType(CR) != PicardType1,
   error("please apply Picard(,,1) and not Picard(,,0) or Picard(,,2)!"));

 if( (t=StructType(a)) == ModuleInitType, a = a.hnf);

 
  \\ if we use the faster computer for the Greither examples
  if(CR.RF[5],  \\ a flag to use the special situation of the Greither examples 

      G = RFQuot(grpring , CR.RqF , CR.RF , 1); \\ the group (Rq/F)*/(R/F)* and some other essential data
  , \\ else
      if (print_level > 0, print1("Computation of RqF / RF  "));
      G = GroupQuot(grpring , CR.RqF , CR.RF , CR.F , 1); \\ the group (Rq/F)*/(R/F)* and some other essential data
      if (print_level > 0, print(" G =  ", [G[1], G[2]]));
  );

 r = length(grpring.Kch);                \\ number of components
 n = length(a[1]);                          \\ order of the group

 arho = vector(r); arhoint = vector(r);  \\ integral ideals a_rho
 arhorel = vector(r);                    \\ a_rho in relative representation 
 cqrho = vector(r);                      \\ components of cq
 d = vector(r);                          \\ will contain the denominators of the arho


 \\ step 1: compute the a_rho
 if (print_level > 0, print1("Computation of the arho  "));
 for(i=1,r,   
   for(j=1,n,          
     phirho = Phirho(grpring, i, a[1][,j]);
     phirho = rnfeltreltoabs(grpring.Kch[i].Krho, phirho);
     id = rnfidealup(grpring.Kch[i].Krho, a[2][j]);
     id = idealgentohnf(grpring.Kch[i].Krhoabs, id);
     id = idealmul(grpring.Kch[i].Krhoabs, id, phirho);
     arho[i]  = idealadd( grpring.Kch[i].Krhoabs , arho[i] , id ); 
   );
   hnf = idealhnf(grpring.Kch[i].Krhoabs , arho[i]);   
   d[i] = denominator(hnf);                            
   arhoint[i] = idealmul(grpring.Kch[i].Krhoabs , d[i] , arho[i]);   

 );
 if (print_level > 0, print(" *** done *** "));

 \\ we need ideals a_rho in O_{K_rho} coprime to f_rho
 yrho = vector(r);        \\ compute the y_rho
 if (print_level > 0, print1("Computation of the yrho  "));
 for(i=1,r,

   aux = idealcoprime(grpring.Kch[i].Krhoabs , arhoint[i] , CR.F[2][i]);
   cqrho[i] = idealmul(grpring.Kch[i].Krhoabs, aux, arhoint[i]);
   \\ cqrho[i] = rnfidealabstorel(grpring.Kch[i].Krho , grpring.Kch[i].Krhoabs.zk*cqrho[i]);
   cqrho[i] = IdealAbsToRel(grpring.nfd, grpring.Kch[i].Krhoabs ,grpring.Kch[i].Krho, cqrho[i]);
   cqrho[i] = StandardPseudoBasis(grpring.nfd , cqrho[i]);
   aux = d[i] * nfbasistoalg(grpring.Kch[i].Krhoabs , aux);
   yrho[i] = rnfeltabstorel(grpring.Kch[i].Krho,aux);

 );
 if (print_level > 0, print(" *** done *** "));

 yrho = PhiInv(grpring,yrho)~;                     \\ yrho back to the group ring with PhiInv

 \\ step 2: cq = yaRq

 ya = ModuleScalarProd(grpring , yrho , a);        \\ at first compute ya
 \\ cq1 = ModuleProd(grpring , a , grpring.MaxOrd); \\ we need cq = yaRq, so compute at first aRq
 \\ cq1 = ModuleScalarProd(grpring , yrho , cq1);  \\ and than cq = yaRq
 
 cq = [Mat(), Vec()];
 for (i=1,r,
   v = vector(r);
   for (j=1, length(cqrho[i][1]),
     v[i] = cqrho[i][1][j];
     cq[1] = concat(cq[1], PhiInv(grpring, v)~);
     cq[2] = concat(cq[2], [ cqrho[i][2][j] ] );
   );
 );
 cq = ModuleHNF(grpring, cq, 1);
 \\ step 3: compute as = ya(cq cut R)^{-1}
 as = ModuleSection(grpring , ModuleInit(grpring,cq) , CR.R);  \\ cq cut R

 if (print_level > 0, print1("Search z  "));
 z1 = Findz(grpring , as , ya, G[3] , G[2] , CR[5][1], 1)[1];
 if (print_level > 0, print(" *** done *** "));

 if (length(z1) != 1, error("***** too many z's found\n"));

 \\ step 4: Find an element z = d_1^{x_1}...d_n^{x_n}, 0 <= x_i < s_i, in G = (Rq/F)*/(R/F)* 
 \\ such that z is an element of as. The d's are represented by G[3], the s_i by G[2]. 

 y1 = G[4]*z1; y1 = Vec(y1)[1];                          \\ technical reasons



 for(i=1,length(y1),         \\ reduce modulo the orders of the generatos of (Rq/F)*
   y1[i] = (-y1[i]) % CR.RqF[2][i];      
 );



 \\ step 5: Find a zs such that zs*z = 1 (mod F), zs in MaxOrd, 
 \\ and compute a c = zs*ya. 
 \\ G[4] is the V^(-1) from the SNF of the group; G = (Rq/F)*/(R/F)* 
 zs = GrPow(grpring , CR.RqF[3] , y1~);       \\ zs = (u_1,...,u_m)*y1; (Rq/F)} = <u_1,...,u_m>
 c = ModuleScalarProd(grpring , zs , ya);     \\ c = zs*ya. Now we have a c with c lhd R and c + F = R 

 \\ step 6: solve the discrete logarithm in Cl_F(Rq) = (+) cl_frho(K_rho)
 \\ and compute cRq = zeta1 * a_1^{v_1} * ... * a_m^{v_m}

 if (print_level > 0, print1("Computation of zz_rcgpIsPrincipal "));
 v = zz_rcgpIsPrincipal(grpring , CR.rcgp , c , flag);
 if (print_level > 0, print(" *** done *** "));

 if(flag, 
   zeta1 = v[2]; v = v[1]; 
 );



 \\ step 7: cRq = zeta1 * ( a_1 ,..., a_m)[ v_1 ,..., v_m]~
 \\             = zeta1 * ( a_1 ,..., a_m)* V * V^{-1} * [ v_1 ,..., v_m]~, V from the SNF S = VHU
 \\             = zeta1 * ( a_1 ,..., a_m)* V * [ y1_1 ,..., y1_m]~, see below

 y1 = CR.S[1]*v;     \\ CR.S is the SNF S = VHU



 \\ step 8: compute r1 and v1 such that y1_i = v1_i * s_i + r1_i
 \\ and solve the discrete logarithm in Pic(R)

 m = length(CR.ord); r1 = vector(m); v1 = vector(m);

 for(i=1,m,

   r1[i] = y1[i] % CR.ord[i];          \\ CR.ord = [s_1,...,s_m]

   v1[i] = (y1[i] - r1[i]) / CR.ord[i];

 );



 \\ step 9: ready to solve the refined discrete logartihm in Pic(R)
 \\ we need a mu such that c = mu * B_1^{r1_1} * ... * B_m^{r1_m}
 if(flag,  

  betav1 = GrPow(grpring,CR.beta , v1);      \\ compute beta_1^{v1_1} * ... * beta_m^{v1_m}, 
					     \\ because bq_i^{s_i} = beta_iRq

  g1y1inv = GrEltInv(grpring , GrPow(grpring,CR.g1,y1)~ );  \\ ( g1_1^{y1_1} * ... * g1_m^{y1_m} )^{-1}

  mu = GrMult(grpring , [ zeta1,betav1,g1y1inv ] );         \\ mu = zeta1 * betav1 * g1y1inv

  \\ now we have c = mu * B_1^{r1_1} * ... * B_m^{r1_m}, Pic(R) = <B_1> x ... <B_m>


  \\ step 10: we want a = mu * (yzs)^{-1} * B_1^{r1_1} * ... * B_m^{r1_m}

  yzs = GrMult(grpring , yrho , zs);   \\ compute y * zs

  nu = GrMult(grpring , mu , GrEltInv(grpring,yzs));     \\ nu := mu * (yzs)^{-1}

  return( [ r1 , nu ] );             \\ r1 is the vector of exponents on the Picard group generators 
				     \\ and nu is the generator of the resulting principal ideal
 );

return( r1 );
}

addhelp(PicIsPrincipal , "PicIsPrincipal(grpring,CR,a,{flag=1}): Given a group ring grpring, the class group CR in Picard and a locally free R-module a. This solves the discrete logarithm problem in the class group Pic(R). If flag=1, solve the refined discrete logarithm problem, else (flag=0) compute only the exponents on the generators of Pic(R).");



/* ****************************** RmodF ******************************
 * 
 * RmodF(grpring,RqF,R,{flag=0}): Given a group ring grpring, the 
 * group (Rq/F)* in RqmodF and the order R, this computes the 
 * group (R/F)* as a 3-component vector like
 * [ cardinality , vector of cyclic components , generators ]. 
 *
 * If flag is nonzero, the result is a 4-component vector where the 
 * structure is as above and the 4-th component is the HNF H such that
 * (R/F)* = (lambda_1,...,lambda_t)H, where (Rq/F)* is generated by
 * the lambda_i. 
 *
 * ***************************************************************** */

RmodF(grpring , RqF , R) = 
{

 local(m,A,H,S,U1,M, t,RF,ords,gen);

 m = RqF[2]; gen = RqF[3];  \\ RqF is a 3-component vector [ cardinality , vector of cyclic components , generators ]

 A = matdiagonal(m);                          \\ Create the diagonal matrix A whose entries are the m_i


 A = concat(A , FindModuleElt(grpring , R , gen , m)[1] ); \\ Find all Elements mu = lambda_1^x_1 x...x lambda_t^x_t
							   \\ such that mu is in R, 0 <= x_k < m_k, 
							   \\ RqF=[lambda_1,...,lambda_t].

 H = mathnf(A);          \\ Compute the Hermite Normal Form HNF
                         \\ columns of H represent the generators of RmodF
 RF = [];
 t = length(gen); 
 for(k=1,t,
   RF = vecput(RF , GrPowModF(grpring,RqF,gen, H[,k])~ );
 );

 M = H^(-1) * matdiagonal(m); \\ the new relations
 S = matsnf(M, 1);
 gen = [];
 U1 = S[1]^(-1); 
 for(k=1,t,
   gen = vecput(gen , GrPowModF(grpring,RqF, RF,U1[,k])~ );
 );
 \\ gen contains the generators with respect to the SNF

 ords = vector(length(S[3]), k, S[3][k,k]);

 return([ prod(i=1,length(ords),ords[i]) , ords , gen ]);

}


addhelp(RmodF , "RmodF(grpring,RqF,R,{flag=0}): Given a group ring grpring, the group (Rq/F)* in RqmodF and the order R, this computes the group (R/F)* as a 3-component vector like [ cardinality , vector of cyclic components , generators ]. If flag is nonzero, the result is a 4-component vector where the structure is as above and the 4-th component is the HNF H such that (R/F)* = (lambda_1,...,lambda_t)H, where (Rq/F)* is generated by the lambda_i.");




/* ****************************** GroupQuot ******************************
 *
 * GroupQuot(grpring,RqF,RF): Given a group ring grpring in GrInit, the
 * group (Rq/F)* in RqmodF and (R/F)* in Oumf, this computes the 
 * quotient (Rq/F)* / (R/F)* as 3-component vector 
 * [ cardinality , vector of cyclic components , generators ].  
 *
 * ****************************************************************** */

GroupQuot(grpring , RqF , RF , F , flag=0) =
{

local(r,m,bid,A,i,j,k,phi,c,v,phij,H,S,U1,ord,d);

r = length(grpring.Kch); 
m = length(RF[3]);

A = matdiagonal(RqF[2]);



bid = RqF[4];

for(i=1,m,

  phi = Phi(grpring,RF[3][i]);

  v = []~;
  for(j=1,r,
    phij = rnfeltreltoabs(grpring.Kch[j].Krho,phi[j]);
    
    \\ solve the discrete logarithm in (Rq/F)* ~ (+)_rho (O_Krho/frho)*
    c = ideallog(grpring.Kch[j].Krhoabs , phij , bid[j]);
    if ( c == vector(length(c))~, c = vector(length(bid[j][2][3]))~ );
    v = concat(v,c);
  );
  A = concat(A,v);
 
);

H = mathnf(A);

S = matsnf(H,1);

U1 = S[1]^-1;

ord = vector(length(S[3]), i, S[3][i,i]);

d = [];
for(k=1,length(U1),
  d = vecput( d , GrPowModF(grpring , RqF, RqF[3] , U1[,k])~);
);

if(flag, return( [ prod(i=1,length(ord),ord[i]) , ord , d , U1 ] ));



return([ prod(i=1,length(ord),ord[i]) , ord , d]);


}


addhelp(GroupQuot , "GroupQuot(grpring,RqF,RF,F,{flag=1}): Given a group ring grpring in GrInit, the group (Rq/F)* in RqmodF and (R/F)* in Oumf, this computes the quotient (Rq/F)* / (R/F)* as 3-component vector [ cardinality , vector of cyclic components , generators ].");





RqFLog(grpring , RqF , lambda) =
{

    local(r,bid,j,phi,c,v,phij);

    r = length(grpring.Kch); 
    bid = RqF[4];

    phi = Phi(grpring,lambda);
    v = []~;
    for(j=1,r,
        phij = rnfeltreltoabs(grpring.Kch[j].Krho,phi[j]);
    
        \\ solve the discrete logarithm in (Rq/F)* ~ (+)_rho (O_Krho/frho)*
        c = ideallog(grpring.Kch[j].Krhoabs , phij , bid[j]);
        if ( c == vector(length(c))~, c = vector(length(bid[j][2][3]))~ );
        v = concat(v,c);
    );

    return(v);
}   

addhelp(RqFLog , "RqFLog(grpring , RqF , lambda): Computes the discrete logarithm in RqF.");


/* ****************************** RFQuot ******************************
 *
 * RFQuot(grpring,RqF,RF,{flag=0}: Computes (Rq/F)* / (R/F)*
 *
 * ****************************************************************** */
RFQuot(grpring , RqF , RF , flag=0) =
{

 local(n,H,S,U1,d,s);

\\ n = length(RqF[2]);        \\ RqF and RF are 3-component vectors like
			    \\ [ cardinality , vector of cyclic components , generators , (H) ]

 n = length(RF[2]);

 H = RF[4];                 \\ remember, H is the HNF such that
			    \\ (R/F)* = (lambda_1,...,lambda_t)H, 
			    \\ where (Rq/F)* is generated by the lambda_i.

 S = matsnf(H , 1);         \\ compute the smith normal form [U,V,S], such that S = UHV

 U1 = S[1]^(-1);            \\ U1 = U^(-1)

 d = Vec();

 for(i=1,n,
\\   d = concat(d, GrPow(grpring , RqF[3] , U1[,i])~); \\ the generators of (Rq/F)*/(R/F)*
\\   u1 = vector(length(RqF[2]));
\\   for(j=1,n,  u1[j] = U1[j,i]);
   d = vecput(d, GrPow(grpring , RqF[3] , U1[,i])~);
 );

 s = vector(length(S[3]) , k , S[3][k,k]);           \\ the orders of the generators


 if(flag, return( [ prod(i=1,length(s),s[i]) , s , d , U1 ] ));

 return( [ prod(i=1,length(s),s[i]) , s , d ] );

}



addhelp(RFQuot , "RFQuot(grpring,RqF,RF,{flag=0}): Given a group ring grpring in GrInit, the group (Rq/F)* in RqmodF and (R/F)* in RmodF, this computes the quotient (Rq/F)* / (R/F)* as 3-component vector [ cardinality , vector of cyclic components , generators ]. If flag is non-zero the result is more complex and only used by PicIsPrincipal.");










/* ****************************** Auxiliary functions ******************************
 * 
 *                              - NOT FOR PERSONAL USE -
 * 
 * ******************************************************************************* */

/* ****************************** RayClassGroup ******************************
 *
 * RayClassGroup(grpring,F,{flag=1}): This function computes all ray class
 * groups cl_frho(Krho), such that Cl_F(Rq) = (+)_rho cl_frho(Krho). 
 *
 * This is necessary for the computation of Pic(R) in Picard. The result is
 * 2-component vector [ vector of orders in rcgp.clgp , all ray class groups ], 
 * if flag is zero.
 *
 * Is flag is non-zero, the result is
 *
 * [ vector of orders in rcgp.clgp , generators of rcgp.clgp , all ray class groups ]
 *
 *
 * For the computation of Pic(R) we only need the cl_frho(Krho), so we don't
 * need to compute a representation ov Cl_F(Rq). 
 *
 * ************************************************************************* */

\\ not for personal use
RayClassGroup(grpring , F , flag=1) =
{

 local(r,gen,ord,rcgp,clgp);

 r = length(grpring.Kch);                     \\ number of components in the Wedderburn decomposition

 gen = vector(r); ord = vector(r); rcgp = vector(r);

 \\ compute the ray class groups to the conductors frho in F[2]
 if(flag, 
   for(i=1,r,                                
     rcgp[i] = bnrinit(grpring.Kch[i].Krhoabs, F[2][i], 1);  \\ flag 1: for bnrisprincipal in some further functions
     clgp = rcgp[i].clgp;
     if(clgp[2] == [], clgp[2] = [1]; clgp[3] = [1]);   \\ if there's no generator (group is trivial), take 1
     ord[i] = clgp[2]; gen[i] = clgp[3];                \\ the orders and the generators of the ray class group
   );

 , \\ else

   for(i=1,r,                                   
     rcgp[i] = bnrinit(grpring.Kch[i].Krhoabs, F[2][i], 0);  \\ flag 0: we don't need the generators
     clgp = rcgp[i].clgp;
     if(clgp[2] == [], clgp[2] = [1]);    \\ if there's no generator (group is trivial), take 1
     ord[i] = clgp[2];
   );

 );

 if(flag, return([  ord , gen , rcgp ]));

 return([ ord , rcgp ]);

}

addhelp(RayClassGroup , "RayClassGroup(grpring,F,{flag=1}): This function is used by Picard to compute the orders and the generators of cl_frho(Krho) in Cl_F(Rq) = (+)_rho cl_frho(Krho). The result is a 3-component vector [ vector of orders in rcgp.clgp , vector of generators in rcgp.clgp , all ray class group ]. Is flag=0, only the orders and the ray class groups are computed.");





/* ****************************** zz_rcgpIsPrincipal ******************************
 *
 * zz_rcgpIsPrincipal(grpring,rcgp,cRq,{flag=1}): This routine solves the
 * discrete logarithm of cRq in the components cl_frho(Krho) in 
 * Cl_F(Rq) = (+) cl_frho(Krho). 
 *
 * rcgp must be given by RayClassGroup(,,1)[3]. If flag=1, solve the refined
 * discrete logarithm, otherwise (flag=0) compute only the exponents on the 
 * ray class group generators.
 *
 * ****************************************************************************** */

\\ not for personal use
zz_rcgpIsPrincipal(grpring , rcgp , cRq , flag=1) =
{

 local(r,n,cRqrho,v,zeta1,phirho,b, id);

 r = length(grpring.Kch);

 n = length(cRq[1]);

 \\ to do a principal ideal test for cRq we work in the Wedderburn decomposition
 cRqrho = vector(r);  v = vector(r);  zeta1 = vector(r);  \\ v will be the vector of exponents on group generators 
							  \\ and zeta1 is the generator of the resulting principal ideal

 \\ we will compute cRq = zeta1 * a_1^{v_1} * ... * a_m^{v_m}
 for(i=1,r,

   for(j=1,n,          

     phirho = Phirho(grpring , i , cRq[1][,j]);    \\ to the Wedderburn decomposition
     phirho = rnfeltreltoabs(grpring.Kch[i].Krho , phirho);
     id = rnfidealup(grpring.Kch[i].Krho, cRq[2][j]);
     id = idealgentohnf(grpring.Kch[i].Krhoabs, id);
     id = idealmul(grpring.Kch[i].Krhoabs, id, phirho);
     cRqrho[i]  = idealadd( grpring.Kch[i].Krhoabs , cRqrho[i] , id );

   );


   b = bnrisprincipal(rcgp[i] , cRqrho[i] , flag);  \\ now we can do a principal ideal test in Cl_F(Rq), component wise

   if(!flag, b = [b]);

   if(b[1] == []~, b[1] = [0]~);    \\ if there's no exponent, take 0

   v[i] = b[1]; 

   if(flag,
     zeta1[i] = nfbasistoalg(grpring.Kch[i].Krhoabs , b[2]);
     zeta1[i] = rnfeltabstorel(grpring.Kch[i].Krho , zeta1[i]);
   );

 );

 v = concat(v);                      \\ we have all exponents

 if(flag, 
   zeta1 = PhiInv(grpring,zeta1);    \\ back to the group ring
   return([ v , zeta1 ]);
 );

 return(v);


}

addhelp(zz_rcgpIsPrincipal , "zz_rcgpIsPrincipal(grpring,rcgp,cRq,{flag=1}): Not for personal use! This function is used by Picard to solve the discrete logarithm of cRq in Cl_F(Rq) = (+)_rho cl_frho(Krho). If flag=1, solve the refined discrete logarithm, otherwise (flag=0) compute only the exponents on the ray class group generators. rcgp must be given by RayClassGroup(,,1)[3].");





/* ****************************** zz_Pic_MatrixD ******************************
 *
 * zz_Pic_MatriD(ord,V1): Not for personal use! This function is used by 
 * Picard to compute a matrix D such that 
 *
 * D = 0 ( mod [ord(a_1),...,ord(a_m)]~ ) and C = V^-1 + D > 0. 
 * V comes from the SNF S = VHU, see Picard(). 
 *
 * We need this matrix D to compute with integral ideals.
 *
 * V1 = V^-1, ord = [ord(a_1),...,ord(a_m)], Cl_F(Rq) = <a_1> x ... x <a_m>.
 *
 * ************************************************************************** */

\\ not for personal use
zz_Pic_MatrixD(ord , V1) =
{

 local(m,D,u,k);

 ord = concat(ord); m = length(ord); D = matrix(m,m);

 for(i=1,m,

   for(j=1,m,

     u = V1[i,j]; k = 0;

     while( u < 0, k++; u += ord[i]); \\ find a d, such that V1[i,j] + d >= 0, where d = 0 (mod ord[i])
     D[i,j] = k*ord[i]

   ); 

 );

 return(D);

}

addhelp(zz_Pic_MatrixD , "zz_Pic_MatrixD(ord,V1): Not for personal use! This function is used by Picard to compute a matrix D such that D = 0 ( mod [ord(a_1),...,ord(a_m)]~ ).");





/* ****************************** zz_Pic_bq_GetMatrixCol ******************************
 *
 * zz_Pic_bq_GetMatrixCol(M,col,beg,end): Given a matrix M. This extracts in the col-th
 * column the rows beg until end. 
 *
 * ********************************************************************************** */

\\ not for personal use
zz_Pic_bq_GetMatrixCol(M , col , beg , end) =
{

 local(l,v);

 l = end - beg + 1; v = vector(l,k,beg++ - 1);

 return(vecextract(M[,col],v));

}

addhelp(zz_Pic_bq_GetMatrixCol , "zz_Pic_bq_GetMatrixCol(M,col,beg,end): Given a matrix M. This extracts in the col-th column the rows beg until end.");





/* ****************************** zz_Pic_epsilon ******************************
 *
 * zz_Pic_epsilon(grpring,rcgp,RF): 
 *
 * ************************************************************************** */

\\ not for personal use
zz_Pic_epsilon(grpring , rcgp , RF , flag=0) =
{

 local(r,t,eps,A,phi,mu,v,aux,muinv,phiinv);

 r = length(grpring.Kch);                     \\ number of K_rho components
 t = length(RF[3]);                           \\ RF = [ cardinality , vector of cyclic components , generators ]

 eps = Mat(); A = Mat();

 for(i=1,t,

\\phi = Phi(grpring,RF[3][,i]);              \\ Phi(lambda) is in (+)_rho K_rho, lambda in (R/F)*
   phi = Phi(grpring,RF[3][i]);

   mu = vector(r);
   v = []~;

   for(j=1,r,                                  

     aux = rnfeltreltoabs(grpring.Kch[j].Krho , phi[j]); \\ for bnrisprincipal we need absolute components
     aux = bnrisprincipal(rcgp[j] , aux, flag);          \\ solve the discrete logarithm

     if(flag == 0, aux = [aux]);              \\ technical reasons from bnrisprincipal(,,0) !

     if(aux[1] == []~, aux[1] = [0]~);        \\ if there is no exponent of aux on the ray class group 
					      \\ generatros, set them to 0

     v = concat( v, aux[1] ) ;                \\ aux[1] are the exponents on the ray class group generators

     if(flag == 1, 
       muinv = nfbasistoalg(rcgp[j] , aux[2])^(-1); \\ aux[2] is given on the integral basis
       mu[j] = rnfeltabstorel(grpring.Kch[j].Krho , muinv);
     );

   );

   A = concat(A,v);

   if(flag == 1,                              \\ if flag=1 we want to solve the refined DL
     phiinv = PhiInv(grpring,mu);             \\ back to the group ring
\\     eps = concat(eps, GrMult(grpring,RF[3][,i],phiinv)~);
     eps = concat(eps, GrMult(grpring,RF[3][i],phiinv)~);
   );

 ); 

 if(flag == 1, return([ A, eps ]));

 return(A);

} 

addhelp(zz_Pic_epsilon , "zz_Pic_epsilon(grpring,rcgp,RF,{flag=0}):");





/* ****************************** zz_Pic_alpha ******************************
 *
 * zz_Pic_alpha(grpring,rcgp,gen,ord): Given a group ring grpring and a 
 * vector rcgp with the ray class groups of the Wedderburn-decomposition as
 * entries, cl_frho(Krho) = <gen_1> x ... x <gen_m>, this computes the 
 * alphas in a_i^{ord(a_i)} = alpha_iRq in the group ring. 
 * C_F(Rq) = <a_1> x ... x <a_m>.
 * 
 * ************************************************************************ */

\\ not for personal use
zz_Pic_alpha(grpring , rcgp , gen , ord) =
{

 local(r,alpha,v,apow,aux);

 r = length(grpring.Kch);                     \\ number of components in the Wedderburn-decomposition

 alpha = Mat();                               \\ will contain the alpha in a_i^{ord(a_i)) = alpha_iRq

 for(i=1,r,

   v = vector(r,k,1);                         \\ v only for PhiInv

   for(j=1,length(gen[i]),                    \\ take each generator in the i-th component

     apow = idealpow(rcgp[i] , gen[i][j] , ord[i][j]);  \\ rcgp is a vector with the r-'rcgp'-components

     aux = bnrisprincipal(rcgp[i] , apow);    \\ solve the discrete logarithm

     aux = nfbasistoalg(rcgp[i] , aux[2]);     

     v[i] = rnfeltabstorel(grpring.Kch[i].Krho , aux); \\ PhiInv need relative components

     alpha = concat( alpha, PhiInv(grpring,v)~ );      \\ back to the group ring

   );
 );

 return(alpha);

}

addhelp(zz_Pic_alpha , "zz_Pic_alpha(grpring,rcgp,gen,ord): Not for personal use! Computes the alphas in a_i^{ord(a_i)} = alpha_iRq, where C_F(Rq) = <a_1> x ... x <a_m>. ");





/* ****************************** zz_Pic_beta ******************************
 *
 * zz_Pic_beta(grpring , alep , W , U , g1 , s): Computes the betas such 
 * that bq_i^{s_i} = beta_iRq, i=1,...,m in the group ring grpring.
 *
 * We know 
 *
 * (bq_1^{s_1},...,bq_m^{s_m}) = (a_1,...,a_m)CS =
 * = (alpha_1Rq,...,alpha_mRq,epsilon_1Rq,...,epsilon_tRq)W*[0,0;0,U] + 
 *   + (0,...,0,gamma_1^{s_1}Rq,...,gamma_m^{s_m}Rq).
 *
 * alep = [alpha,epsilon], g1 = [gamma_1,...,gamma_m], s = [s_1,...,s_m].
 *
 * *********************************************************************** */

\\ not for personal use
zz_Pic_beta(grpring , alep , W , U , g1 , s) = 
{

 local(l,m,beta,M,rc,row,col,U1,v,WU,aux);

 \\ W is a m x m matrix
 l = length(W); m = length(s); beta = Mat(); M = Mat();

 rc = matsize(U); row = rc[1]; col = rc[2]; 

 g1 = GrPow(grpring , g1 , s , 1);            \\ compute the gamma_i^{s_i} in 
					      \\ g1 = [ gamma_1^{s_1},...,gamma_m^{s_m} ]
					      \\ flag 1 returns the vector of the gamma_i^{s_i}

 U1 = matrix(l , l-col);                      \\ fill the matrix U up with nulls, we 
					      \\ need U = [0,0;0,U]
 for(i=1,col,
   v = concat(vectorv(l-row) , U[,i]);        \\ concat the columns
   U1 = concat(U1,v);                         \\ than U1 and the column
 );

 WU = W*U1;                                   \\ now we can multiply W by U

 for(i=l-col+1,l,                             \\ we only need the last m entries
   aux = GrPow(grpring,alep,WU[,i]~)~;        
   M = concat(M,aux);
 );

 for(i=1,m,                                   \\ add the 'aleps' and the 'gammas' and we've done
   beta = concat( beta, GrMult(grpring ,M[,i] , g1[i])~ );
 );

 return(beta);

}

addhelp(zz_Pic_beta , "zz_Pic_beta(grpring,alep,W,U,g1,s): Not for personal use! Computes the betas such that bq_i^{s_i} = beta_iRq, i=1,...,m in the group ring grpring. Only necessary for the routine Picard.");





/* ****************************** zz_Pic_bq ******************************
 *
 * zz_Pic_bq(grpring,rcgp,gen,C): Not for personal use! This functions is
 * used by the routine Picard to compute some technical data to compute 
 * the Picard group. 
 *
 * grpring is the group ring, rcgp is a vector with all ray class groups 
 * such that Cl_F(Rq) = cl_f1(K_1) (+) ... (+) cl_fr(K_r). 
 *
 * Than we compute (bq_1,...,bq_m) = (a_1,...,a_m)C where all informations, 
 * the a_i,i=1,...,m and the matrix C is given by the function Picard. 
 * The a_i are the generators of the ray class group Cl_F(Rq).
 *
 * ******************************************************************************************* */

\\ not for personal use
zz_Pic_bq(grpring , rcgp , gen , C) = 
{

 local(r,m,n,yrho,t,y1,d,bq,M,v);

 r = length(grpring.Kch); m = length(concat(gen)); n = length(C); yrho = vector(r);

 t = vector(r+1); t[1] = 1;

 for(i=1,r,
   t[i+1] = t[i] + length(gen[i]);
 );

 y1 = vector(m);

 for(i=1,n,
   for(j=1,r,

     d = zz_Pic_bq_GetMatrixCol(C , i , t[j] , t[j+1]-1)~; \\ get special rows out of the i-th column in C

     yrho[j] = IdealPow(rcgp[j] , gen[j] , d);   \\  

     yrho[j] = IdealAbsToRel(grpring.nfd , grpring.Kch[j].Krhoabs , grpring.Kch[j].Krho , yrho[j]); 
   );

   y1[i] = yrho;

 );

 bq = vector(m); 

 for(i=1,m,

   M = Mat();

   for(j=1,r,

     v = vector(r);

     for(k=1,length(y1[i][j]),

       v[j] = y1[i][j][k];

       M = concat( M, PhiInv(grpring , v)~ );

     );

   );

   bq[i] = M;

 );

 return(bq);                                  \\ now (bq_1,...,bq_m) = (a_1,...,a_m)C

}

addhelp(zz_Pic_bq , "zz_Pic_bq(grpring,rcgp,gen,C): Not for personal use! This functions is used by the routine Picard to compute some technical data to compute the Picard group.");





\\ not for personal use
zz_rcgpIdealToGrElt(grpring , ideal) = 
{

local(r,GE,ID,w,v);

r = length(grpring.Kch);

GE = Vec();

for(i=1,r,

  ID = Mat();
  for(k=1,length(ideal[i]),
     w =  IdealAbsToRel(grpring.nfd , grpring.Kch[i].Krhoabs , grpring.Kch[i].Krho , ideal[i][k]);

     v = vector(r);

     for(j=1,length(w),
       v[i] = w[j];
       phiinv = PhiInv(grpring , v)~;
       ID = concat(ID , phiinv);
     );

     for(l=1,r,

       if(l!=i, 
         w = IdealAbsToRel(grpring.nfd , grpring.Kch[l].Krhoabs , grpring.Kch[l].Krho , 1);
         v = vector(r);
 
         for(j=1,length(w),
           v[l] = w[j];
           phiinv = PhiInv(grpring , v)~;
           ID = concat(ID , phiinv);
         );

       );
  
      );

  );

  GE = concat(GE , [ID]);

);

return(GE);

}



/* ****************************** RqmodF ******************************
 * 
 * RqmodF(grpring,F): Given a group ring grpring in GrInit(K,G) and the
 * conductor F of R, this computes the group (Rq/F)*, where Rq 
 * identifies the maximal order in K[G]. 
 *
 * We know, (Rq/F)* ~ (+)_rho (O_Krho/frho)* from the
 * Wedderburn decomposition. So we only have to compute the
 * (O_Krho/frho)* with idealstar and put them together. 
 *
 * The f_rho are given by F = [ F , [frho_1,...,frho_r ]. 
 *
 * ****************************************************************** */

RqmodF(grpring , F) =
{

 local(r,RqF,m,v,bid,gen,aux);

 r = length(grpring.Kch);                     \\ number of the rho components

 RqF = Vec();                                 \\ RqF will contain the generators of (Rq/F)*
 m = Vec();                                   \\ m will contain the orders of the generators in (Rq/F)*

 v = vector(r,k,1);                           \\ only for the function PhiInv, see below
 bid = vector(r);

 \\ compute (Rq/F)* ~ (+)_rho (O_Krho/frho)*  
 for(i=1,r,
   bid[i] = idealstar(grpring.Kch[i].Krhoabs , F[2][i] , 2);   \\ F = [ F , [frho_1,...,frho_r]

   \\ no generator found                            \\ flag = 0, computes the structure of (O_Krho/frho)* 
   if(bid[i][2][2] == [], bid[i][2][2] = [1]; bid[i][2][3] = [1]);    \\ as a 3-component vector
						    \\ bid[1] the order, 
   m = concat(m,bid[i][2][2]);                            \\ bid[2] is the vector of SNF cyclic components and    
   gen = bid[i][2][3];                                    \\ bid[3] the corresponding generators of (O_K_rho/f_rho)*

   for(j=1,length(gen),

     aux = nfbasistoalg(grpring.Kch[i].Krhoabs , gen[j]); \\ PhiInv needs relative components
     v[i] = rnfeltabstorel(grpring.Kch[i].Krho , aux);    

					       \\ use PhiInv to compute an element in the group ring  
  \\ RqF = concat(RqF , PhiInv(grpring,v)~);   \\ Now RqF contains a system of generators of (RqF/F)*
     RqF = vecput(RqF , PhiInv(grpring,v)~); 
   );

   v[i] = 1;

 );


 return([ prod(i=1,length(m),m[i]) , m , RqF , bid]);   \\ return [ cardinality , vector of cyclic components , generators , the (OK_rho/f_rho)* ]

}

addhelp(RqmodF , "RqmodF(grpring,F): Given a group ring grpring in GrInit and the conductor F of R, this computes the group (Rq/F)*, where Rq is the integral closure of R. In our case this means Rq is the maximal order in the group ring.");



















