/* ****************************** Oumf.gp **********************************
*
*
*       Oumf.gp contains all necessary functions to compute the following finite group:
*       Given a group ring KG initialized by GrInit (q.v. GrInit.gp) and an order A in
*       ModuleInitType (q.v. Module.gp) then Oumf(KG,A) is the group of invertible 
*       elements  of A/f, where f denotes the conductor of A. This is essentially
*       an adapted version of the algorithm in Cohen's 2nd book (see the section in
*       Picard groups and refined discrete logarithms)
*
*       -Oumf(KG, A, {flag=0})
*       -MmodN(KG, M, N)
*       -PseudoToZ(ideals, gen)
*       -OkToZ(KG, M)
*       -AmodPx(KG, A, p, eins)
*       -GrEltinMOrd(KG, M, elt)
*       -GrEltinMaxOrd(KG, elt)
*       -some auxiliary functions
*
* author:     Andreas Nickel (mail: andreas.nickel@freenet.de)
*             markus endres (parigp@mendres.org)
*             werner bley (bley@math.uni-augsburg.de)
*
* Last modification: Tue, Nov 02, 2004
*
* created:    DEZ 2003 and JAN 2004
*
* ********************************************************************* */



/* *********************PseudoToZ******************************************
*
*       PseudoToZ(ideals, gen): given a vector of KG-ideals ideals and a vector 
*       of generators gen
*       this computes a HNF-Z-basis of the module M, created by the pseudo-basis,
*       which belongs to the given ideals and generators. The result will be a matrix
*       (in general not a square matrix) which columns represent group ring elements
*
************************************************************************* */

PseudoToZ(ideals, gen) =
\\Z-Basis fr M = A1 v1 + A2 v2 +..., falls ideals = [...A_i...] und gen = [...v_i...]
{
        local(M, i, n, A, v, a);
        n = length(gen);
        M = Mat();

        for(i=1, n,
                A = ideals[i];
                v = gen[i]~;
                a = length(A);


                for(j=1, a,
                        M = concat(M, A[j]*v);
                );
        );


\\        M = mathnf(M);



        return(M);
}

addhelp(PseudoToZ, "PseudoToZ(ideals, gen):  given a vector of KG-ideals ideals and a vector of generators gen this computes a HNF-Z-basis of the module M, created by the pseudo-basis, which belongs to the given ideals and generators. The result will be a matrix (in general not a square matrix) which columns represent group ring elements.");





/* *************************OkToZ******************************************
*
*       OkToZ(KG, M): given a group ring KG and a KG-module M
*       this computes a HNF-Z-basis of the module M. The result will be a matrix
*       (in general not a square matrix) which columns represent group ring elements.
*
************************************************************************* */

OkToZ(KG, M) =
{
        local(ok, m, ideals, gen, a, v, i);

        if (StructType(M) == ModuleInitType, M = M.hnf);
        
        m = length(M[1]);                  \\Laenge der ok-Basis von M
        if (length(ok) == 1, return(M););        \\ok = Z
                

        v = vector(m, a, idealhnf(KG.nfd, M[2][a]));
        ideals = vector(m,a,[]);
        for (a=1, m,
            for (i=1, length(v[a]),  
                ideals[a] = concat(ideals[a], [nfbasistoalg(KG.nfd, v[a][,i])]);
            );
        );
        gen = vector(m, a, M[1][,a]~);

        return(PseudoToZ(ideals, gen));
}

addhelp(OkToZ, "OkToZ(KG, M): given a group ring KG and a KG-module M this computes a HNF-Z-basis of the module M. The result will be a matrix (in general not a square matrix) which columns represent group ring elements.");






/* ****************************AmodPx**********************************
*
*       AmodPx(KG, A, p, eins): given a group ring KG, an KG-order A and some maximal ideal p
*       of A this computes the cyclic group of invertible elements of A/p. eins has to be the
*       vector representing the 1-element in KG. The result will be in group type.
*
********************************************************************* */


AmodPx(KG, A, p, eins) =
\\ berechnet Erzeugendes von (A/p)-mal in GroupType
{
        local(X, X_pot, q, a, i);
        
        A = MmodN(KG,A,p);

        q = A.order;
        A = A.gens;

        i=0; a = length(A);

        while(i != q-1,

                X = vector(length(eins));

                for (i=1, a,
                       X = X + (random(6)-3)*A[i];
                );
                i=1; X_pot = X;

                if(!IsModuleElt(KG, X_pot, p),

                   while(!IsModuleElt(KG, X_pot-eins, p),

                      X_pot = GrMult(KG, X_pot, X); 
                      i++;
                   );
                , \\ else

                   i=0;  
                );
        );

        return([i, [i], [X], Mat(1)]);
}

addhelp(AmodPx, "AmodPx(KG, A, p, eins): given a group ring KG, an KG-order A and some maximal ideal p of A this computes the cyclic group of invertible elements of A/p. eins has to be the vector representing the 1-element in KG. The result will be in group type.");


\\the following function computes some data often used in Oumf
K_and_p_pots(KG, p, q, A) =
{
        local(k, p_pot);

        k = 0;
        p_pot = Vec([p]);      \\p^1 = p;

        while(!ModuleIncl(KG, p_pot[k+1], q),
                p_pot = concat(p_pot, [ModulePow(KG, p_pot[k+1], 2,A)]);
                k++;
        );

        p_pot = concat(p_pot,[q]);      \\for computational reasons
        return([k, p_pot]);
}

/*the following function computes the group of invertible elements of R/q
  where q is the ideal defined on page 7 */
GroupExt(KG, R, k, p_pot) =
{
       local(p, q, a, b, G, cs, gammas, U_as, ns, h, Amod, g0, H, v, i, j, elt, pos, eins);

\\       print("GroupExt k = ",k, "  length(p_pot) = ", length(p_pot));
       p = p_pot[1];
       q = p_pot[(length(p_pot))];

       v = KG.grp.ord;
       eins = vector(prod(i=1, length(v),v[i])); eins[1] = 1;

       Amod = AmodPx(KG,R, p, eins);            \\in GroupType
       g0 = Amod.gens[1];                 \\Amod is a cyclic group, so g0 will be a generator

       if(k==0,
                return(Amod);
       );

       U_as = Vec();
       cs = Vec();
       gammas = Vec();
       ns = Vec();

       a = 1; k++;

       while(a < k,
                b = min(a+1, k);
                G      = MmodN(KG, ModuleSum(KG,p_pot[a],q), ModuleSum(KG,p_pot[b],q));
                U_as   = concat(U_as, [G.ua]);
                cs     = concat(cs, [G.invfac]);
                gammas = concat(gammas, [G.gens]);
                ns     = concat(ns, length(G.invfac));

                a++;

       );


       a--;
       k--;
       h = sum(i=1, length(ns), ns[i]) + 1;

       H = matrix(h,h);

       elt = GrPow(KG, [g0], [Amod.order]);
       v = DlogExt(KG, elt, k, U_as, cs, gammas, eins);
       H[,1]  = concat(Vec(Amod.order), -v)~;
       pos = 1;
       G = Vec([g0]);

       for(j=1, a,
                for(i=1, ns[j],
                        pos++;
                        v = eins + gammas[j][i];
                        elt = GrPow(KG, [v], [cs[j][i]]);
                        G = concat(G,[v]);
                        v = DlogExt(KG, elt, k, U_as, cs, gammas, eins);
                        v = concat(0,-v)~;
                        v[pos] = cs[j][i];
                        H[,pos] = v;
                );
       );
       return(GroupSNF(KG, R, q, G,H));
}

\\this function solves the discrete logarithm problem for the groups needed in the function GroupExt above
DlogExt(KG, elt, k, U_as, cs, gammas, eins_Ok) =
{
        local(B, a, U_a, cs_a, gammas_a, Z, z, i, eins_Z,yps);

\\        print("DlogExt k = ", k, " cs = ", cs);
        B = GrEltinMaxOrd(KG, elt);
        eins_Z = vector(length(B))~; eins_Z[1] = 1;
        a = 1; yps = Vec();

        while(a <= k,
                U_a      = U_as[a];
                cs_a     = cs[a];
                gammas_a = gammas[a];

                Z = U_a * (B-eins_Z);
                z = length(Z);
                for(i=1, z,
                        Z[i] = -((-Z[i]) % cs_a[i]);
                        elt = GrMult(KG, elt, GrPow(KG, [eins_Ok + gammas_a[i]], [-Z[i]]));
                );
                yps = concat(yps, Vec(Z));
                B = GrEltinMaxOrd(KG, elt);
                a++;
        );

        return(yps);
}



/* this function makes the inverse of the isomorphism phi (page 8) explicit.
  phi is given by the Chinese Remainder theorem */
ChinRev(KG, R, A, B, v) =
{
        local(n, m,  i, z, U, a, b);

        n = length(v);
        m = length(R.zbase);
        \\w = KG.grp.ord;
        \\m2 = prod(i=1, length(w),w[i]);
        z = vector(m); z[1] = 1;
        
        A = R.zhnf^(-1) * A.zhnf;
        B = R.zhnf^(-1) * B.zhnf;

        U = mathnf(concat(A,B), 1)[2];
        z = U * FillVec(z, 2*m)~;
        a = R.zbase * A * VecCut(z,m)~;
        b = R.zbase * B * VecCutLast(z,m)~;

        for(i=1,n,

                v[i] = GrMult(KG,b,v[i]) + a~;
        );

        return(v);
}




/* ****************************Oumf*************************************
*
*       Oumf(KG, R, {flag=0}): Given a group ring KG initialized by GrInit (q.v. GrInit.gp) and
*       an order R in ModuleInitType (q.v. Module.gp) then Oumf(KG,R) computes the group of
*       invertible elements of R/f, where f denotes the conductor of R. If flag is non-zero
*       the result will be in GroupType and contains a matrix for discrete logarithm problems
*       as his last component. Otherwise this matrix will be omitted.
*
********************************************************************** */

Oumf(KG, R, F, flag=0) =
{

\\ F
local(i, j, k, m, n, primefact, ei, P, P_i,Ps,erz,  ps, qs, pj, prodaqj, Amod, p_pot, neu, AmodF, bj, zz, Fcon);

	Fcon = F[1];
        F = F[2];
        
        n = length(F);
        Ps = Vec(); eis = Vec();

        erz = Vec();  ei = vector(n);
        for(j=1, n,
                P_i = [ Mat(), Vec() ];
                \\ neu = IdealAbsToRel(KG.nfd, KG.Kch[j].Krhoabs, KG.Kch[j].Krho,
                \\                    matid(poldegree(KG.Kch[j].Krhoabs.nf.pol)));
                neu = KG.Kch[j].Krho[7];
                neu = StandardPseudoBasis(KG.nfd, neu);
                for (k=1, length(neu[1]),
                        ei[j] = neu[1][k];
                        P_i[1] = concat(P_i[1], PhiInv(KG, ei)~);
                        P_i[2] = concat(P_i[2], [ neu[2][k] ]);
                );
                ei[j] = 0;
                erz = concat(erz, [P_i]);
        );

        erz_db = erz;         

        for (i=1, n,
                primefact = idealfactor(KG.Kch[i].Krhoabs, F[i]);
                P = Mat(); ei = vector(n); P_i = [Mat(), Vec()];
                for(j=1, n,
                        if(j != i,
                                P_i[1] = concat(P_i[1], erz[j][1]);
                                P_i[2] = concat(P_i[2], erz[j][2]);
                        );
                );
                for(j=1, matsize(primefact)[1],
                        P = P_i;
                        zz = IdealAbsToRel(KG.nfd, KG.Kch[i].Krhoabs, KG.Kch[i].Krho, primefact[j,1]);
                        
                        zz = StandardPseudoBasis(KG.nfd, zz);
                        for (k=1, length(zz[1]),
                                ei[i] = zz[1][k];
                                P[1] = concat(P[1], PhiInv(KG, ei)~);
                                P[2] = concat(P[2], [ zz[2][k] ])
                        );
                        P = ModuleInit(KG,P);
                        Ps = concat(Ps, [P]);
                        eis = concat(eis, primefact[j,2]);
                );
        );
        Ps_db = Ps;

        ps = Vec(); qs = Vec();
        n = length(eis);        \\Anzahl der Primideale = |I|

        pj = ModuleSection(KG, Ps[1], R);
        pj = ModuleInit(KG, pj);
        ps = concat(ps, [pj]);

        qs = concat(qs, [ModuleInit(KG,
                         ModuleSection(KG,R,
                         ModuleInit(KG,ModulePow(KG,Ps[1][1],eis[1],KG.MaxOrd))))]);

        for(i=2, n,
                pj = ModuleSection(KG, Ps[i], R);
                pj = ModuleInit(KG, pj);
                j = 1; k = length(ps);
                while (j <= k,
                        if (ModuleCompare(KG,ps[j],pj)[1],
                                qs[j] = ModuleSection(KG, qs[j],
                                                ModuleInit(KG,
                                                ModuleSection(KG,R,
                                                ModuleInit(KG,ModulePow(KG,Ps[i][1],eis[i],KG.MaxOrd)))));
                                qs[j] = ModuleInit(KG,qs[j]);
				if(!ModuleIncl(KG, qs[j], ps[j]), error("q nicht in p"););
                                j = k+2;
\\                                print("----> ps[",j,"] == Ps[",i,"] <-----");

                        );
                        j++;
                );
                if (j == k+1,
                        ps = concat(ps, [pj]);
                        qs = concat(qs, [ModuleInit(KG,
                                         ModuleSection(KG,R,
                                         ModuleInit(KG,ModulePow(KG,Ps[i][1],eis[i],KG.MaxOrd))))]);

                );

        );

      \\ return(Ps);

        /**********************************************************************/
        /* ps enthaelt die Primideale in R, qs die zugehoerigen Primaerideale */
        /**********************************************************************/

        m = length(ps); \\ = length(qs) = Anzahl der Primideale unten = |J|
        AmodF = [[], []];    \\trivial group

\\        print("Anzahl der Primideal in R = ", m);
	Hs = Vec();
        for (j=1, m,
        \\Berechnung von (A/qj)-mal
                p_pot = K_and_p_pots(KG, ps[j], qs[j], R);
                k = p_pot[1]; p_pot = p_pot[2];

                prodaqj =  GroupExt(KG, R, k, p_pot);
        Hs = concat(Hs, [[prodaqj, ps[j], qs[j]]]);
                AmodF[2] = concat(AmodF[2], prodaqj.invfac);
                bj = R;

                for(i=1, m,
                        if(i != j,
                                bj = ModuleSection(KG,bj, qs[i]);
                                bj = ModuleInit(KG,bj);
                        );
                );

                AmodF[1] = concat(AmodF[1], ChinRev(KG, R, qs[j], bj, prodaqj.gens));
        );
	AmodF = GroupSNF(KG, R, Fcon, AmodF[1], matdiagonal(AmodF[2]));

        if(!flag, AmodF = VecCut(AmodF, 3););

        return(AmodF);
}

addhelp(Oumf, "Oumf(KG, R, F, {flag=0}): Given a group ring KG initialized by GrInit (q.v. GrInit.gp) and an order R in ModuleInitType (q.v. Module.gp) then Oumf(KG,R) computes the group of invertible elements of R/f, where f denotes the conductor of R. If flag is non-sero the result will be in GroupType and contains a matrix for discrete logarithm problems as his last component. Otherwise this matrix will be omitted.");




FillVec(v, h) =
\\fills v with leading seros
{
        local(n, i);

        n = h - length(v);
        if(n <= 0, return(v));

        for(i=1, n,
                v = concat(0,v);
        );
        return(v);
}




/* **********************GrEltinMOrd*****************************************
*
*       GrEltinMOrd(KG, M, elt): given a group ring KG, a KG-module M and a
*       group ring element elt lying in M this computes the vector representing
*       the element elt in the basis of M.
*
************************************************************************** */

GrEltinMOrd(KG, M, elt) =
{
        local(v, i, ideals);

        if (StructType(M) != ModuleInitType, M = ModuleInit(KG,M););

        elt = elt * M.inv~;
        ideals = vector(length(M[2]), i, idealhnf(KG.nfd, M[1][2][i]));
        v = Vec();
        for (i=1, length(elt),
                v = concat(v, ideals[i]^(-1) * nfalgtobasis(KG.nfd, elt[i]));
        );
        return(M.Uinv * v);
        return(v);
}

addhelp(GrEltinMOrd, "GrEltinMOrd(KG, M, elt): given a group ring KG, a KG-module M and a group ring element elt lying in M this computes the vector representing the element elt in the basis of M.");


/* **********************GrEltinMOrd*****************************************
*
*       GrEltinMOrd(KG, M, elt): given a group ring KG, a KG-module M and a
*       group ring element elt lying in M this computes the vector representing
*       the element elt in the basis of M.
*
************************************************************************** */

OldGrEltinMOrd(KG, M, elt) =
{
        local(v, i);

        if (StructType(M) != ModuleInitType, M = ModuleInit(KG,M););

        elt = elt * M.inv~;
        v = Vec();
        for (i=1, length(elt),
                v = concat(v, nfalgtobasis(KG.nfd, elt[i]));
        );
        return(v);
}

addhelp(GrEltinMOrd, "GrEltinMOrd(KG, M, elt): given a group ring KG, a KG-module M and a group ring element elt lying in M this computes the vector representing the element elt in the basis of M.");





/* **********************GrEltinMaxOrd*****************************************
*
*       GrEltinMaxOrd(KG, elt): given a group ring KG and a
*       group ring element elt lying in the maximal order MAX of KG this computes the vector representing
*       the element elt in the basis of MAX.
*
************************************************************************** */


GrEltinMaxOrd(KG, elt) =
{
        return(GrEltinMOrd(KG, KG.MaxOrd, elt));
}

addhelp(GrEltinMaxOrd, "GrEltinMaxOrd(KG, elt): given a group ring KG and a group ring element elt lying in the maximal order MAX of KG this computes the vector representing the element elt in the basis of MAX.");




NewXToZ(KG, M, N) = 
{

    local (n,l,k,j,Mideals, Nideals, i, s, S, T,X );

    if (StructType(M) == ModuleInitType, M = M.hnf);
    if (StructType(N) == ModuleInitType, N = N.hnf);

    n = length(M[1]);   s = length(KG.nfd.zk);
    Mideals = vector(n, i, idealhnf(KG.nfd, M[2][i]));
    Nideals = vector(n, i, idealhnf(KG.nfd, N[2][i]));

    S = matrix(n*s, n*s);  T = matrix(n*s, n*s);

    for (l=1,n, for (k=1,s,
        for (i=1,n, for (j=1, s,
            v = nfalgtobasis(KG.nfd, nfbasistoalg(KG.nfd, Mideals[i][,j]) * M[1][l,i]);
            S[(l-1)*s+k, (i-1)*s+j] = v[k];
            v = nfalgtobasis(KG.nfd, nfbasistoalg(KG.nfd, Nideals[i][,j]) * N[1][l,i]);
            T[(l-1)*s+k, (i-1)*s+j] = v[k];
        ););
    ););
    \\ return(T^(-1) * S);
    return(S^(-1) * T);
}



MatPolToInt(A) =
{
    local(lc, i, j, B);

    lc = matsize(A);
    B = matrix(lc[1], lc[2]);
    for (i=1, lc[1],
        for (j=1, lc[2],
            if (poldegree(lift(A[i, j])) > 0, 
                error("*** not an integer in MatPolToInt");
            );
            B[i,j] = polcoeff(lift(A[i,j]), 0);
        );
    );
    return(B);
}



/* ********************MmodN*********************************************
*
*       MmodN(KG, M, N): given a group ring KG, and two KG-module M and N,
*       N a submodule of M this computes the factor module M/N belonging to a
*       Z-basis. This result will be in GroupType.
*
********************************************************************** */

MmodN (KG,M,N) =
\\(M)X = (N), returns a groupSNF of M/N (Z-basis) in GroupType

{
        local(X, G, m, i, orders, gammas);

        if (StructType(M) != ModuleInitType, M = ModuleInit(KG,M));
        if (StructType(N) != ModuleInitType, N = ModuleInit(KG,N));

        \\ A = M.zhnf; B = N.zhnf in Cohen's notation
        X = M.zhnf^(-1) * N.zhnf;           \\ X = A^(-1) * B, as in Cohen

        X = matsnf(X, 1);                   \\ X = [U, V, S]

        G = Vec();
        m = length(M.zbase);

        for(i=1, m,
                G = concat(G, [M.zbase[,i]~]);
        );
        gammas = G * X[1]^(-1);
        orders = vector(length(X[3]), i, X[3][i,i]);
        return([matdet(X[3]), orders, gammas, X[1] * M.zhnf^(-1)]);


}

addhelp(MmodN, "MmodN(KG, M, N): given a group ring KG, and two KG-module M and N, N a submodule of M this computes the factor module M/N belonging to a Z-basis. This result will be in GroupType.");


