\\  -*-gp-script-*-

/* ****************************** NormalBasis.gp ******************************
 *
 * NormalBasis.gp contains some function to compute a normal basis and a 
 * weak normal basis element.
 * 
 * time-stamp: <Mon, Oct 04, 2004 - 14:18>
 *
 * author: markus endres (mail: parigp@mendres.org)
 *
 * ************************************************************************** */



/* ****************************** NormalBasisElt ******************************
 *
 * NormalBasisElt(grpring,relpol,{aut=0}): Computes a normal basis element. 
 * grpring is the group ring, aut the galois group automorphisms gal(L|K) and 
 * relpol the relative defining polynomial for L|K. if aut=0, then the 
 * generators of gal(L|K) must be given in grpring.grp.ggen.
 *
 * Lemma: theta ia a NB-element <=> e_i*theta != 0 for all i=1,...,r
 * e_i are the idempotents
 *
 * ************************************************************************** */

NormalBasisElt(grpring , relpol , aut=0) = 
{

 local(var,OL,n,r,omj,idem,i,found,eom,th);

 if(!aut, aut = AllGroupElt(grpring.grp,relpol));

 var = variable(relpol);

 OL = rnfpseudobasis(grpring.nfd , relpol);         \\ we need a basis of O_L, L|K, but rnfpseudobasis
 OL = StandardPseudoBasis(grpring.nfd , OL);   \\ gives only a pseudobasis, so we have to
					            \\ generate a 'real' basis 
 n = length(OL[1]); r = length(grpring.Kch);

 omj = vector(r); 

 for(k=1,r,
  idem = grpring.Kch[k].idem;                            \\ the idempotents
 
  i=1; found=0;
  while(i <= n && !found,
    eom = GrAction(aut, relpol, idem, OL[1][i]);       

    

    if(eom != 0, 
      omj[k] = OL[1][i]; found=1;

      if(k==1, omj[1] = OL[1][i] / (eom * length(grpring.grp.eltlist)));
 
     , \\ else

      i++;

    );

  );
 );

 th = sum(i=1,r, GrAction(aut , relpol , grpring.Kch[i].idem , omj[i])); 

 return(th);

}

addhelp(NormalBasisElt , "NormalBasisElt(grpring,relpol,{aut=0}): Computes a normal basis element. grpring is the group ring, aut the galois group automorphisms gal(L|K) and relpol the relative defining polynomial for L|K. if aut=0, then the generators of gal(L|K) must be given in grpring.grp.ggen.");






/* ****************************** WeakNormalBasisElt ******************************
 * WeakNormalBasisElt(grpring,relpol,Ath,th): Computes a weak normal basis element.
 * Given a group ring grpring with the galois group generators in grpring.grp.ggen, 
 * the relative defining polynomial relpol for L|K, A_theta and th, th a normal
 * basis element, this compute a weak normal basis element 
 *
 * ******************************************************************************* */


WeakNormalBasisElt(grpring , relpol , Ath , th) = 
{

local(CR,M,FDL,a,nth,aut);

CR = Picard(grpring , grpring.MaxOrd , 1 , 1);   \\ flag 0: don't use RmodFGreither (see GrInit.gp)

M = ModuleProd(grpring , Ath , grpring.MaxOrd);  \\ Ath*MaxOrd

FDL = PicIsPrincipal(grpring , CR , M);               \\ solve the refined discrete logarithm of M=Ath*MaxOrd

aut = AllGroupElt(grpring.grp , relpol);         \\ compute all galois automorphisms

nth = GrAction(aut,relpol,FDL[2],th);            \\ now we have a weak normal basis element

return(nth);

}

addhelp(WeakNormalBasisElt , "WeakNormalBasisElt(grpring,relpol,Ath,th): Given a group ring grpring with the galois group generators in grpring.grp.ggen, the relative defining polynomial relpol for L|K, A_theta and th, this compute a weak normal basis element.");





/* ****************************** ThetaOnOLBasis ******************************
 * ThetaOnOLBasis(OL,th): Gives the element th on the O_L basis.
 *
 *
 * ************************************************************************** */


ThetaOnOLBasis(OL , th) = 
{

 local(var,n,A,B);

 var = variable(OL[1]); n = length(OL); A = matrix(n,n);

   for(i=1,n,
   for(j=0,poldegree(OL[i]),
     A[j+1,i] = polcoeff(OL[i] , j , var);
   );
 );

 B = vectorv(n,i,polcoeff(th,i-1));

 \\ AX = B
 matsolve(A,B);

}

addhelp(ThetaOnOLBasis , "ThetaOnOLBasis(OL,th): Gives the element th on the O_L basis.");







\\ Tests, if th is a normal basis element
\\ Lemma: theta ia a NB-element <=> e_i*theta != 0 for all i=1,...,r
\\ e_i are the idempotents
TestNormalBasisElt(grpring , aut , th) =
{

local(r,i,eth);

r = length(KG.Kch);

for(i=1,r,
  eth = GrAction(aut , relpol , grpring.Kch[i].idem , th);
  print(" eth = " eth);
);

}
addhelp(TestNormalBasisElt , "TestNormalBasisElt(grpring,aut,th): th is a normal basis element if e_i*theta != 0 for all i, e_i the idempotents, aut the galois authomorphisms and grpring a group ring.");
