/* ****************************** group structure ******************************
 *
 * The structure of 'GroupType' is a 4-component vector like
 * [group order , invariant factors , generators , matrix for discrete logarithms].
 * 
 * The functions are:
 *
 * 	- GroupSNF(G,M)
 * 
 * Auxiliary functions: NOT FOR PERSONAL USE
 *
 * 	- DLog(X,U)
 * 	- VecCut(v,s)
 * 	- VecCutLast(v,s)
 *	- MatCut(M,s,t)
 * 	
 * time-stamp: <Mon, Oct 04, 2004 - 10:38>
 *
 * author:     Andreas Nickel (mail: andreas.nickel@freenet.de)
 *             markus endres (parigp@mendres.org)
 *
 * ***************************************************************************** */

GroupType = ["t_INT", "t_VEC", "t_VEC", "t_MAT"];

x.order         = x[1];         \\group order
x.invfac        = x[2];         \\invariant factors
x.gens          = x[3];         \\generators of the group
x.ua            = x[4];         \\matrix U_alpha for discrete logarithms


/* *******************************GroupSNF*****************************************
*
*       GroupSNF(G, M): given a vector of generators and a matrix of relations between them
*       this computes a Smith normal form (SNF) of the group defined by the given elements.
*       The result will be in GroupType.
*
********************************************************************************* */

GroupSNF(KG, A, f, G, M) =
\\G generators in A mod f, M relations
{
        local(H, X, Xme, N, n, s, orders, i);


        M = Mat(M);     \\falls M ein Vektor ist
        H = mathnf(M);
	s = matsize(H);
        if(s[1] != s[2], error("****not a finite group or not a complete system of relations***"););
        s = s[1];
        X = matsnf(H,1);        \\X = [U,V,D]
	Xme = X[1]^(-1);
	n = length(Xme);
	N = vector(n);

	for(i=1, n,
		N[i] = VectorMod(KG,A,f,G, Xme[,i]);
	);

   \\     N = G * X[1]^(-1);
   \\     while(s>1 && X[3][s,s] == 1, s--;);
   \\     X[3] = MatCut(X[3],s, s);
   \\     N = VecCut(N,s);
   \\     X[1] = MatCut(X[1],s, matsize(X[1])[2]);
        orders = vector(length(X[3]), i, X[3][i,i]);
        return([matdet(X[3]), orders, N, X[1]]);
        \\N generators, X[1] = U_alpha for discrete logarithms, matdet = order of the group
}

addhelp(GroupSNF, "GroupSNF(G, M): given a vector of generators and a matrix of relations between them this computes a Smith normal form (SNF) of the group defined by the given elements. The result will be in GroupType.");

VectorMod(KG,A,f, v1, w) =
{
	local(i, erg);
\\        print("\n ***********************************\nVectorMod");

	erg = vector(length(A.hnf[1]));
	erg[1] = 1;
	for (i=1, length(v1),
		erg = GrMult(KG, erg, ModPot(KG,A,f,v1[i],w[i]));
	);

	return(erg);
}

ModPot(KG, A, f, x, z) =
{
\\        print("ModPot:", "  z = ", z);
        
	if(z >= 0, return(GrPow(KG,x,z)););
	\\z < 0:
	x = Inverse(KG,A,f,x);

	return(GrPow(KG,x,-z));
}

Inverse(KG, A, f, x) =
{
	local(B, F, n, m,  i, z, U, a, b);

\\        print("Inverse: f = ", f);

	B = ModuleScalarProd(KG,x,A);

	B = ModuleInit(KG, B);
        f = ModuleInit(KG, f);

	if (!ModuleIncl(KG, A, ModuleSum(KG, B, f)), error("not an invertible Element."););

        n = length(x);
        m = length(A.zbase);
        z = vector(m); z[1] = 1;

        F = A.zhnf^(-1) * f.zhnf;
        B = A.zhnf^(-1) * B.zhnf;
	U = mathnf(concat(B,F), 1)[2];
        z = U * FillVec(z, 2*m)~;
        b = A.zbase * B * VecCut(z,m)~;
        a = A.zbase * F * VecCutLast(z,m)~;

        x = GrMult(KG, b, GrEltInv(KG,x));

        return(x);
}


DLog(X,U) =
/* if an element a is known on the generators G as a = GX and
   U is given by the above algorithm, this computes a in terms of the new generatos */
{
        X = X*U~;
        return(X);
}

addhelp(DLog,"DLog(X,U): Discrete Logarithm");


\\ some additional auxiliary funcions:

\\ not for personal use
VecCut(v, s) =
{
        local(w, i);
        w = vector(s);
        s = min(s, length(v));
        for (i=1, s,
                w[i] = v[i];
        );
        return(w);
}

addhelp(VecCut,"VecCut(v,s): Not for personal use!");


\\ not for personal use
VecCutLast(v, s) =
{
        local(w, i, n);

        w = vector(s);
        n = length(v);
        s = min(s, n);
        for(i=1, s,
                w[i] = v[n-s+i];
        );
        return(w);
}

addhelp(VecCutLast,"VecCutLast(v,s): Not for personal use!");


\\ not for personal use
MatCut(M, s, t) =
{
        local(N, n, i, j);
        N = matrix(s,t);
        n = matsize(M);
        s = min(s, n[1]); t = min(t, n[2]);
        for(i=1,s,
                for(j=1,t,
                        N[i,j] = M[i,j];
                );
        );
        return(N);
}

addhelp(MatCut,"MatCut(M,s,t): Not for personal use!");
