\\  -*-gp-script-*-

/* ****************************** AAO.gp ******************************
 * 
 * AAO.gp Contains some functions to compute the arithmetic associated
 * order A(O_L) and a locally free module Ath, see below.
 * 
 * This file contains:
 *
 *      - AAO(grpring,relpol,{flag=0})
 *      - Atheta(grpring,relpol)
 *
 * Auxiliary functions: NOT FOR PERSONAL USE
 *
 *      - zz_Ath_repmat(relpol,aut,om) 
 *      - zz_Ath_nbgen(relpol,aut,OL) 
 *      - yy_Ath_nbgen(grpring,relpol,aut,OL)
 *      - zz_Ath_basis(A,t) 
 *
 *
 * time-stamp: <Tue,  Nov 02, 2004 - 14:52>
 *
 * author:     markus endres (mail: parigp@mendres.org)
 *
 * ****************************************************************** */





/* ****************************** AAO ******************************
 *
 * AAO(grpring,relpol): Given a group ring grpring in GrInit(K,G) 
 * and a relative defining polyomial relpol for the field extension
 * L|K, this computes the arithmetic associated order A(O_L).
 * A(O_L) := {lambda in K[G] | lambda(O_L) subseteq OL}.
 *
 * For this we compute an A_theta defined as 
 * A_thetea := {lambda in K[G] | lambda(theta) in O_L} where theta
 * is a nomral basis generator of O_L.
 *
 * Than A(O_L) = (A_theta A_theta*)*.
 *
 * See: W. Bley and D. Burns: Ueber Arithmetische Assoziierte 
 * Ordnungen (JNT, Vol. 58, No.2, 1996)
 *
 * *************************************************************** */

AAO(grpring , relpol , aut, flag=0) =
{

 local(Ath,A);

 Ath = Atheta(grpring , relpol, 0, aut);              \\ compute the A_theta := {lambda in K[G] | lambda(theta) in O_L}

 Ath = ModuleInit(grpring,Ath);               \\ and initalize A_theta with ModuleInit

 A = ModuleConductor(grpring , Ath , Ath);    \\ now compute A = (A_theta A_theta*)*

 if(flag, return( [ A , Ath.hnf]));             

 return(A);

}

addhelp(AAO , "AAO(grpring,relpol,{flag=0}):Given a group ring grpring in GrInit(K,G) and a relative defining polyomial relpol for the field extension L|K, this computes the arithmetic associated order A(O_L). If flag is nonzero, return a two-component vector [ A(O_L) , Ath.hnf ].");





/* ****************************** Atheta ******************************
 *
 * Atheta(grpring,relpol): Given a group ring grpring in GrInit(K,G) 
 * and a relative defining polynomial relpo for the field extension
 * L|K, this computes the 
 * A_theta := {lambda in K[G] | lambda(theta) in O_L}
 * such that the associated order A(O_L) = (A_theta A_theta*)*.
 * 
 * If you already have a normal basis element th you can pass it
 * through the th argument.  
 * If the relative galois automorphisms are already computed,
 * put them in the relaut argument.  
 *
 * See: W. Bley and D. Burns: Ueber Arithmetische Assoziierte 
 * Ordnungen (JNT, Vol. 58, No.2, 1996)
 *
 * ****************************************************************** */

Atheta(grpring , relpol , th=0 , relaut=0) = 
{


 local(n,aut,OL,A,t,k,Ath,var);

 n = poldegree(relpol);
 var = variable(relpol);

 if (!relaut,
   if(type(grpring.grp.ggen[1]) != "t_POL", 
     aut = GaloisAut(grpring , relpol)[1];           \\ compute the galois automorphisms
     , \\ else
     aut = AllGroupElt(grpring.grp , relpol);
   );
 ,
   aut = relaut;
 );
 
 OL = rnfpseudobasis(grpring.nfd , relpol);   \\ we need a basis of O_L, L|K, but rnfpseudobasis
 OL = StandardPseudoBasis(grpring.nfd, OL, var);

 A = zz_Ath_repmat(relpol , aut , OL[1]);        \\ now we compute the representation matrices

 if(!th,          \\ no normal basis element th is given

   t = vector(n);

   \\search a normal basis generator theta
   if( (k = yy_Ath_nbgen(grpring, relpol , aut , OL[1])) != 0,  \\ zz_Ath_nbgen() returns the index of the normal basis generator
   t[k] = 1;                                  \\ in the vector OL[1]
   th = OL[1][k];                                \\ now, theta = OL[1][k]
    ,  \\ else
     error("no normal basis generator\n");    \\ no generator found
   );
   th_db = th;
 , \\ else

   t = ThetaOnOLBasis(OL[1],th)~;                \\ we need th on the OL-basis

 );

 \\ compute a basis for A_theta
 Ath = zz_Ath_basis(A,t);                     \\ now, A_{theta} = <ld_1,...,ld_n> where
					      \\ ld_i = sum(j=1,n, xs[i][j]*aut[j]) 
					      \\ resp. ld_i = xs[i]*aut


return([Ath, OL[2]]);

}

addhelp(Atheta , "Atheta(grpring,relpol,{th=0},{relaut=0}):  Given a group ring grpring in GrInit(K,G) and a relative defining polynomial relpol for the field extension L|K, this computes the A_theta such that the associated order A(O_L) = (A_theta A_theta*)*. th can be a normal basis element and relaut are the relative galois group automorphisms gal(L|K). If no th and no relaut is passed, the Atheta functions computes them for internal use only.");



 

/* ****************************** Auxiliary functions ******************************
 * 
 *                              - NOT FOR PERSONAL USE -
 * 
 * ******************************************************************************* */

/* ****************************** zz_Ath_repmat ******************************
 *
 * zz_Ath_repmat(relpol,aut,om): Let relpol be a relative defining polynomial
 * for the number field L|K and O_L = om = (w_1,...,w_n). This computes the
 * representation matrices for the galois automorphisms aut in G, G = Gal(L|K), 
 * such that 
 *
 * [w_1,..,w_n]~^aut = A(aut)[w_1,...,w_n]~
 *
 * ************************************************************************* */

\\ not for personal use
zz_Ath_repmat(relpol , aut , om) = 
{

 local(var,n,X,A,T,Tinv,B,colB);

 var = variable(relpol);
 n = poldegree(relpol);

 X = vector(n , i , var^(i-1));

 A = vector(n);  \\ will contain the representation matrices

 \\ compute T
 T = matrix(n,n);

 for(i=1,n,
   for(j=0,poldegree(om[i]),
     T[i,j+1] = polcoeff(om[i] , j , var);
   );
 );

 Tinv = T^(-1);

 \\ compute B
 for(i=1,n,

   B = matrix(n,n);
   for(k=0,n-1,

     colB = subst(var^k , var , aut[i]) % relpol;

     for(j=0,poldegree(colB),
       B[k+1,j+1] = polcoeff(colB , j , var);
     );
   );

   A[i] = T*B*Tinv;

 );

 return(A);

}

addhelp(zz_Ath_repmat , "zz_Ath_repmat(relpol,aut,om): Not for personal use! This function is used by Atheta to compute the representation matrices for the galoisautomorphisms.");





/* ****************************** zz_Ath_nbgen ******************************
 *
 * zz_Ath_nbgen(relpol,aut,OL): This function is used by 
 * Atheta(grpring,relpol) (see above) and is not for personal use. 
 *
 * zz_Ath_nbgen() computes a generator for a normal basis of L|K. relpol 
 * defines the field extension L|K, aut are the galois automorphisms
 * of the galois group Gal(L|K) and OL is O_L computed in Atheta.
 *
 * Almost all elements of OL generate a normal basis of L|K. So, take
 * one in OL, say theta, and test if theta generates a normal basis. 
 * The test is as follows:
 *
 * theta generates normal basis  <=>  
 *                            det(theta^{st})_{s,t in Gal(L|K)} != 0
 *
 * ***************************************************************** */

\\ not for personal use
zz_Ath_nbgen(relpol , aut , OL) = 
{

 local(n,var,k,th,sth,tsth);

 n = length(OL);                              \\ number of elements in the basis for OL

 var = variable(relpol);                      \\ in which variable is the polyomial relpol given

 for(k=1,n,                                   \\ test, if theta (th) is a generator for a normal basis.

   th = OL[k];

   \\ vector sth with s(th), s in Gal(L|K), as entries
   sth = vector(n,k, subst(th , var , aut[k]) % relpol );    \\ th^s, s in gal(L|K)

   \\ generate the matrix with (th^{st})_{s,t in Gal(L|K)} as entries
   tsth = matrix(n,n,i,j, subst(sth[j] , var , aut[i]) % relpol);    \\ th^{st}, t in gal(L|K)

   \\ compute determinant of tsth, det( th^{st})

   if( (matdet(tsth) % relpol) != 0,      \\ then th is a normal basis generator
     return(k);                           \\ return the index of the normal basis generator in OL
    , \\else test another element in OL
     next();                              \\ next th
    );
 );

 return(0);                               \\ no generator for a normal basis found
					  \\ this should not happen

}

addhelp(zz_Ath_nbgen , "zz_Ath_nbgen(relpol,aut,OL): Not for personal use! This function is used by Atheta and computes a normal basis generator of L|K. relpol defines the field extension L|K, aut are the galois automorphisms of Gal(L|K) and OL represents O_L.");





/* ****************************** yy_Ath_nbgen ******************************
 *
 * yy_Ath_nbgen(grpring, relpol,aut,OL): This function is used by 
 * Atheta(grpring,relpol) (see above) and is not for personal use. 
 *
 * yy_Ath_nbgen() computes a generator for a normal basis of L|K. relpol 
 * defines the field extension L|K, aut are the galois automorphisms
 * of the galois group Gal(L|K) and OL is O_L computed in Atheta.
 *
 * This algrithm is dirty; it is not sure that one of the basis elements 
 * generates a normal basis.
 *
 * One should use Girstmaier's method!
 * 
 * ***************************************************************** */

\\ not for personal use
yy_Ath_nbgen(grpring, relpol , aut , OL) = 
{

 local(n,r,var,i,k,th,echith,found);


 n = length(OL);                              \\ number of elements in the basis for OL
 r = length(grpring.Kch);


 var = variable(relpol);                      \\ in which variable is the polyomial relpol given

 k=1; found=0;
 while (k<=n & !found,

   th = OL[k];
   found=1; i=1;
   while (found & i<=r,

     echith = GrAction(aut, relpol, grpring.Kch[i].idem, th);

     if (echith == 0, 
        found=0; k=k+1; 
     ,
        i = i+1;
     );
   );
 );

 if (!found,
      print("No NB element found!!!!!");
      return(0);
 );
 return(k); 

}

addhelp(yy_Ath_nbgen , "yy_Ath_nbgen(grpring, relpol,aut,OL): Not for personal use! This function is used by Atheta and computes a normal basis generator of L|K. grpring is the group ring, relpol defines the field extension L|K, aut are the galois automorphisms of Gal(L|K) and OL represents O_L.");





/* ****************************** zz_Ath_basis ******************************
 *
 * zz_Ath_basis(A,t): Not for personal use! Compute A_theta = <ld_1,...,ld_2>, 
 * theta a generator for a normal basis, ld_i in K[G], G=gal(L|K).  
 *
 * Take ld_i = sum(s in G, x_s*s) than we have to solve a system of 
 * linear equations which is formed by the coefficients of
 *
 * ld_i(th) = sum(k=1,n, sum(s in G, sum(j=1,n, t_jA(s)_{jk})x_s )w_k) = w_i
 *
 *  where OL=<w_1,...,w_n>. 
 *
 * ************************************************************************ */

\\ not for personal use
zz_Ath_basis(A,t) = 
{

 local(n,v,xs,M);                                  \\ user defined variables

 n = length(t); v = vectorv(n); xs = matrix(n,n);  \\ save result in xs

 M = matrix(n,n,i,j, t*A[j][,i]);                  \\ we have to solve Mx = v 

 \\ solve the system of linear equations
 for(i=1,n,
    v[i]=1;                                        \\ compute lambda_i
    xs[,i] = matsolve(M,v);                        \\ solve Mx = v
    v[i]=0;
 );

 return(xs);                                       \\ now, 
						   \\ ld_i = sum(j=1,n, xs[i][j]*aut[j])
						   \\ resp. ld_i = xs[i]*aut
}

addhelp(zz_Ath_basis , "zz_Ath_basis(A,t): Not for personal use! This function is used by Atheta. Compute A_theta = <ld_1,...,ld_2>, theta a generator for a normal basis, ld_i in K[G].");












