\\  -*-gp-script-*-

/* ****************************** Module.gp ******************************
 * 
 * Module.gp: Contains all necessery functions to compute with modules in 
 * a group ring initialized by GrInit (q.v. GrInit.gp): 
 *
 * Modules:
 *
 *      - ModuleInit(grpring,M)
 *      - ModuleHNF(grpring,M,{flag=0})
 *      - ModuleProd(grpring,M,N,{flag=0})
 *      - ModulePow(grpring,M,n,ORD,{flag=0})
 *      - ModuleSum(grpring,M,N,{flag=0})
 *      - ModuleScalarProd(grpring,c,M,{flag=0})
 *      - ModuleSection(grpring,M,N,{flag=0})
 *      - ModuleCompare(grpring,M,N)
 *      - ModuleIncl(grpring,M,N)
 *      - ModuleIndex(grpring,M,N) 
 *      - ModuleConductor(grpring,U,V,{flag=0})
 *      - IsModuleElt(grpring,M,N)
 *      - FindModuleElt(grpring,M,d,s,{flag=0})
 *      - DualModule(grpring,M)
 *      - IsLocallyFree(grpring,a)
 *      - IsFreeOverMax(grpring,a,R)
 *      - CalcMaxOrd(grpring,{flag=0})
 *      - IdealAbsToRel(K,nf,rnf,ideal)
 *      - IdealInv(grpring,a,R)
 *      - IdealPow(nf,ideal,pow)
 *      - IdealIsPrimitive(nf,ideal) =
 *
 * time-stamp: <Tue, Nov 02, 2004 - 15:24>
 *
 * author:     markus endres (mail: parigp@mendres.org)
 *             werner bley   (mail: bley@math.uni-augsburg.de)
 *
 * created:    Tue Apr 15 2003
 *
 * ********************************************************************* */





/* ****************************** Module structure ******************************
 *
 * The structure of 'ModuleInitType' is a 5-component vector like
 * [ module in HNF , inverse module , dual module , module in Z-base , matrix A as below].
 *
 * The structure of 'ModuleHNFType' is a 2-component vector like
 * [ module in HNF , module denominator ]. 
 *
 * **************************************************************************** */

\\ ModuleInitType = [ "t_VEC" ,  "t_MAT" , "t_VEC", "t_MAT", "t_MAT"];
ModuleInitType = [ "t_VEC" ,  "t_MAT" , "t_VEC", "t_MAT", "t_MAT", "t_MAT"];

\\ Module structure initialized by ModuleInit resp. ModuleHNF
x.hnf     = x[1];         \\ pseudobasis in  Hermite Normal Form of the module
x.inv     = x[2];         \\ the inverse module
x.dual    = x[3];         \\ the dual module
x.zbase   = x[4];         \\ HNF of the module in a Z-base
x.zhnf    = x[5];         \\ matrix A in HNF, which satisfies x.zbase = KG.MaxOrd.zbase*A, 
x.Uinv    = x[6]          \\ inverse of the matrix which transforms the canonical zbase into HNF                       




/* ****************************** ModuleInit ******************************
 *
 * ModuleInit(grpring,M): Given a group ring grpring in GrInit and a module
 * M, this computes all the necessary data to work with the module M. The
 * result is a 6-component vector as follows:
 * [ [matrix of generators, list of ideals] , inverse of HNF[1] , dual module , 
 * Z-base , a matrix zhnf as below, Uinv].
 * M[1] contains the module in Hermite Normal Form (M.hnf), M[2] contains the inverse 
 * of M[1][1] (M.inv),  M[3] is the dual module of M (M.dual), M[4] contains the
 * module in HNF with respect to a Z-base, M[5] is a square matrix zhnf, 
 * which satisfies KG.MaxOrd.zbase * zhnf = x.zbase (x.zhnf), and finally M[6] is
 * the inverse of the matrix U of some HNF computation (see source code).
 *
 * I call this representation 'ModuleInitType'.
 *
 * ********************************************************************** */
ModuleInit(grpring , M, flag = 0) =
\\ flag=1 is used only in GrInit
{
 
 local(inv,dual,zbase,maxbase, HU);


 if( StructType(M) == ModuleInitType, return(M) ); \\ M is already in ModuleInitType

 M = ModuleHNF(grpring, M);
 inv = M[1]^(-1); 
 dual = DualModule(grpring , M);          \\ compute the dual module
 zbase = OkToZ(grpring, M);               \\ compute the canonical Z-base  
 
 if(flag,
   return([ M , inv , dual, zbase]);    \\ return [ HNF, denominator, inverse, dual, zbase ]
 );

 maxbase = NewXToZ(grpring, grpring.MaxOrd, M);
 HU = mathnf(maxbase, 1);
 zbase = zbase * HU[2];
 return([ M, inv , dual, zbase , HU[1], HU[2]^(-1) ]);
 \\ return([ M, inv , dual, zbase , maxbase]);

}

addhelp(ModuleInit , "ModuleInit(grpring,M): Given a group ring grpring in GrInit and a module M, this computes all the necessary data to work with the module M. The result is a 6-component vector as follows: M[1] contains the pseudobasis of the module in Hermite Normal Form (M.hnf), M[2] contains the inverse of M.hnf[1] (M.inv), M[3] is the dual module of M (M.dual), M[4] contains the module in HNF with respect to a Z-base (M.zbase), M[5] is a square matrix zhnf, which satisfies KG.MaxOrd.zbase * zhnf = x.zbase (x.zhnf), and finally M[6] isthe inverse of the matrix U of some HNF computation (see source code).");





/* ****************************** ModuleHNF ******************************
 *
 * ModuleHNF(grpring,M,{flag=0}: Computes the Hermite Normal Form (HNF)
 *
 * ********************************************************************* */

ModuleHNF(grpring , M , flag=0) =
{

 local(t,n,d,il,H,q,alg, detM);

 if( (t=StructType(M)) == ModuleInitType,
    return( M.hnf);
 );

 n = length(M[2]);

 if( type(M[1][1,1]) == "t_POLMOD" ||  type(M[1][1,1]) == "t_POL",             \\ If the module M is in polmod representation
   M[1] = matalgtobasis(grpring.nfd , M[1]);        \\ change to a representation on the integral
 );                                           \\ basis of the group ring number field
 H = nfhnf(grpring.nfd , M);
 H[1] = matbasistoalg(grpring.nfd , H[1]);
 return(H);
}

addhelp(ModuleHNF , "ModuleHNF(grpring,M): Given a group ring grpring in GrInit and a module M, this returns the Hermite Normal Form of M.");





/* ****************************** ModuleProd ******************************
 *
 * ModuleProd(grpring,M,N,{flag=0}: Computes the product of two modules M 
 * and N in the group ring grpring initialized by GrInit (q.v. GrInit.gp).
 *
 * ********************************************************************** */

ModuleProd(grpring , M , N , flag=0) =
{

 local(t,m,n,MN, MN1, MN2,i,j);

 if( (t=StructType(M)) == ModuleInitType, M = M.hnf );

 if( type(M[1][1,1]) != "t_POLMOD",            \\ change to polmod representation, if necessary
   M[1] = matbasistoalg(grpring.nfd , M[1]);
 );

 if( (t=StructType(N)) == ModuleInitType, N = N.hnf );

 if( type(N[1][1,1]) != "t_POLMOD",            \\ change to polmod representation, if necessary
   N[1] = matbasistoalg(grpring.nfd , N));


 \\ now compute the product of the modules
 m = length(M[1]); n = length(N[1]); MN1 = Mat(); MN2 = Vec();
 for(i=1,m,
   for(j=1,n,
     MN1 = concat(MN1 , GrMult(grpring , M[1][,i] , N[1][,j])~);
     MN2 = concat(MN2,  [ idealmul(grpring.nfd, M[2][i], N[2][j]) ] );
   );
 );

 MN = ModuleHNF(grpring , [MN1, MN2]);                \\ compute HNF 

 return(MN);

}

addhelp(ModuleProd , "ModuleProd(grpring,M,N): Given a group ring grpring in GrInit and two modules M and N in the group ring, this returns the product of M and N");



/* ****************************** ModulePow ******************************
 *
 * ModulePow(grpring,M,n,ORD,{flag=0}): Given a group ring grpring in GrInit 
 * and a module M as well as an integer n, this computes the n-th power of 
 * the module M in the group ring. 
 *
 * M can be also a vector of modules and n a vector of integers. Then this
 * routine computes each module raised to the corresponding power in n. 
 * If flag is omitted or zero the result is the product of these 
 * exponentiated modules. If flag is nonzero the result is a vector which 
 * these exponentiated modules as entries.
 *
 * The flag is only allowed in the case that M and n are vectors of the 
 * same length. 
 *
 * ORD represents the corresponding order such that M^0 = ORD.
 *
 * ********************************************************************* */

ModulePow(grpring , M , n , ORD , flag=0) =
{

 local(t,M1,i);

 if(n==0, return(ORD)); \\ M^0 = ORD
 if(n==1, return(M));

 t = length(n);

 if(type(n) == "t_VEC",

   for(i=1,t,
     M[i] = ModulePow(grpring,M[i] , n[i] , ORD);
   );

   \\ compute the product of all modules
   for(i=1,t-1,
     M[i+1] = ModuleProd(grpring , M[i] , M[i+1]);
   );

   return(M[t]); 
 );

 M1 = M;

 for(i=2,n,
   M = ModuleProd(grpring , M , M1);
 );

 return(M);

}

addhelp(ModulePow , "ModulePow(grpring,M,n,ORD,{flag=0}): Given a group ring grpring in GrInit and a module M as well as an integer n, this computes the n-th power of the module M in the group ring. M can be also a vector of modules and n a vector of integers. Than this routine computes each module raised to the corresponding power in n. If flag is omitted or zero the result is the product of these exponentiated modules. If flag is nonzero the result is a vector which these exponentiated modules as entries. The flag is only allowed in the case that M and n are vectors of the same length. ORD represents the corresponding order such that M^0 = ORD.");





/* ****************************** ModuleSum ******************************
 *
 * ModuleSum(grpring,M,N): Compute the sum of two modules M and N 
 * in the group ring grpring initialized by GrInit (q.v. GrInit.gp).
 *
 * ********************************************************************* */

ModuleSum(grpring , M , N) = 
{

 local(t,S, S1, S2);

 M = ModuleHNF(grpring , M); 
 N = ModuleHNF(grpring , N);

 S1 = concat(M[1], N[1]);
 S2 = concat(M[2], N[2]);
 S = ModuleHNF(grpring , [S1, S2]);                  \\ compute HNF of the sum

 return(S);
}

addhelp(ModuleSum , "ModuleSum(grpring,M,N,{flag=0}): Compute the sum of two modules M and N in the group ring grpring initialized by GrInit.");





/* ****************************** ModuleScalarProd ******************************
 *
 * ModuleScalarProd(grpring,c,M): Given a group ring grpring in GrInit, an 
 * element c and a module M in this group ring, this functions computes the 
 * product of the group ring element with the module.
 *
 * **************************************************************************** */

ModuleScalarProd(grpring , c , M) = 
{

 local(i, t,n,cM);

 if( (t=StructType(M)) == ModuleInitType,
   M = M.hnf
 );

 n = length(M[2]);
 cM = [Mat(), M[2]];  \\ will contain the product cM

 for(i=1,n,
   cM[1] = concat(cM[1] , GrMult(grpring , c , M[1][,i])~);
 );


 cM = ModuleHNF(grpring , cM);                \\ compute HNF of cM
 return(cM);
}

addhelp(ModuleScalarProd , "ModuleScalarProd(grpring,c,M}): Given a group ring grpring in GrInit, an element c and a module M in this group ring, this functions computes the product of the group ring element with the module, i.e. cM.");





/* ****************************** ModuleSection ******************************
 * 
 * ModuleSection(grpring,M,N): Given a group ring grpring in GrInit and two 
 * modules M and N, this computes the section of the two modules. 
 *
 * The section of two modules will be computed as: M cap N = (M* + N*)*, 
 * where * means the dual module. 
 *
 * M and N must be given in 'ModuleInitType'!
 *
 * ************************************************************************* */

ModuleSection(grpring , M , N ) =
{

 local(S);

 \\ M and N must be given in 'ModuleInitType'
 if( StructType(M) != ModuleInitType || StructType(N) != ModuleInitType, 
   error("***   please apply ModuleInit first "));   
 S = ModuleSum(grpring , M.dual , N.dual);    \\ compute the sum (M* + N*)
 S = DualModule(grpring , S);                   \\ and than the dual module
 S = ModuleHNF(grpring , S);                    \\ Hermite Normal Form

 return(S);
}

addhelp(ModuleSection , "ModuleSection(grpring,M,N): Given a group ring grpring in GrInit and two modules M and N, this computes the section of the two modules. M and N must be given in 'ModuleInitType'!");





/* ****************************** ModuleCompare ******************************
 *
 * ModuleCompare(grpring,M,N): Compares two modules M and N in the group
 * ring grpring. 
 * 
 * The result is more than a simple equality test. It's a 3 component-vector 
 * where the first component is 1 if both modules are equal (or 0 if not) 
 * in the sense of HNF equality (see below). The second component checks 
 * if M < N, i.e. if M is included in N and the third one checks if N < M,
 * i.e. if N is included in M.
 *
 * The HNF representation of a free module is unique. So, we can trivially 
 * test equality of modules: their HNF and their denominator must be the 
 * same. 
 *
 * ************************************************************************* */

ModuleCompare(grpring , M , N) = 
{ 

 M = ModuleHNF(grpring , M);
 N = ModuleHNF(grpring , N);

 return( [ M == N ,                           \\ are the modules equal ?
	   ModuleIncl(grpring , M , N) ,      \\ is M < N
	   ModuleIncl(grpring , N , M) ] );   \\ is N < M

}

addhelp(ModuleCompare , "ModuleCompare(grpring,M,N): Compares two modules M and N in the group ring grpring. The result is more than a simple equality test. It's a 3 component-vector where the first component is 1 if both modules are equal (or 0 if not). The second component checks if M < N, i.e. if M is included in N and the third one checks if N < M, i.e. if N is included in M.");





/* ****************************** ModuleIncl ******************************
 * 
 * ModuleIncl(grpring,M,N): Given a group ring grpring in GrInit and two
 * modules M and N, this routine checks if the module M is contained in 
 * the module N. The result is only true (1) or false (0). 
 * 
 * For a detailed check use ModuleCompare.
 *
 * To test wether M is included in N compute S = M + N and test the
 * equality S == M. 
 * 
 * ********************************************************************** */

ModuleIncl(grpring , M , N) = 
{

 local(t,S);

 M = ModuleHNF(grpring , M);
 N = ModuleHNF(grpring , N); 


 S = ModuleSum(grpring , M , N);          
 if( S == N ,   
   return(1);                                 \\ if M < N
  , \\ else
   return(0);                                 \\ M is not included in N
 );

}

addhelp(ModuleIncl , "ModuleIncl(grpring,M,N): Given a group ring grpring in GrInit and two modules M and N, this routine checks if the module M is contained in the module N. The result is only true (1) or false (0). For a detailed check use ModuleCompare.");





/* ****************************** ModuleIndex ******************************
 *
 * ModuleIndex(grpring,M,N): Given a group ring grpring (K[G]) in GrInit 
 * and two modules M and N, this computes the module index [M : N]. 
 *
 * [M : N] = ( det(a_{ij}) ) x B / A, where a_{ij} is the matrix which transforms the
 * generators of M to generators of N, B is the product over the ideals of N and
 * A is the product over the ideals of M. 
 *
 * *********************************************************************** */

ModuleIndex(grpring , M , N) =
{

 local(t,n,d, D, A, B);

 M = ModuleHNF(grpring , M);
 N = ModuleHNF(grpring , N);

 n = length(M[2]);
 D = matrix(n,n);

 for(i=1,n, D[,i] = matsolve(M[1] , N[1][,i]));

 d = idealhnf(grpring.nfd, matdet(D));
 A = M[2][1];
 for (i=2, n, A = idealmul(grpring.nfd, A, M[2][i]));
 B = N[2][1];
 for (i=2, n, B = idealmul(grpring.nfd, B, N[2][i]));

 d = idealmul(grpring.nfd, d,  idealmul(grpring.nfd, B, idealinv(grpring.nfd, A)));
 
 return( idealhnf(grpring.nfd , d) );

}

addhelp(ModuleIndex , "ModuleIndex(grpring,M,N): Given a group ring grpring in GrInit(K,G) and two modules M and N, this computes the module index [M : N]_{O_K}.")





/* ****************************** ModuleConductor ******************************
 * 
 * ModuleConductor(grpring,M,N,{flag=0}): Given a group ring grpring and two
 * modules M and N. This compute the conductor 
 * F :=  {lambda in grpring | lambda(M) subseteq N}.
 *
 * If flag is nonzero, this computes also the conductors f_rho in the 
 * Wedderburn-decomposition of the group ring grpring. Then the result is a two
 * component vector like [ F , [f_rho_1,...,f_rho_r] ].
 * 
 * We compute the conductor as follows:
 *   F* = MN* ==> F = (MN*)*
 * where * denotes the dual module.
 *
 * *************************************************************************** */

ModuleConductor(grpring , M , N , flag=0) = 
{

 local(F,r,n,frho,phirho, arho);                 \\ local variables

 \\ we need all modules in 'ModuleInitType'
 if( StructType(M) != ModuleInitType || StructType(N) != ModuleInitType,
   error("***   please apply ModuleInit first"));
 F = ModuleProd(grpring , M , N.dual);
 F = DualModule(grpring , F);

 F = ModuleHNF(grpring , F);

 if(flag,            \\ compute the concutor F in the Wedderburn-decomposition

   \\ number of components
   r = length(grpring.Kch); n = length(F[2]);
   frho=vector(r);
   
   for(i=1,r,
      for(j=1,n,
       phirho = Phirho(grpring , i , F[1][,j]);     \\ Phirho(F)
       phirho = rnfeltreltoabs(grpring.Kch[i].Krho , phirho);
       arho = rnfidealup(grpring.Kch[i].Krho, F[2][j]);
       arho = idealgentohnf(grpring.Kch[i].Krhoabs, arho);
       arho = idealmul(grpring.Kch[i].Krhoabs, arho, phirho);
       frho[i]  = idealadd(grpring.Kch[i].Krhoabs , frho[i] , arho);
     );
   );
   
   return([ F , frho ]);

 );

 return(F);
}

addhelp(ModuleConductor,"ModuleConductor(grpring,M,N,{flag=0}): Given a group ring grpring and two modules M and N. This compute the conductor F :=  {lambda in grpring | lambda(M) subseteq N}. If flag is nonzero, this computes also the conductors f_rho in the Wedderburn-decomposition of the group ring grpring. Than the result is a two component vector like [ F , [f_rho_1,...,f_rho_r] ].");





/* ****************************** IsModuleElt ******************************
 * 
 * IsModuleElt(grpring,lambda,M): Given a group ring grpring in Grinit and
 * an element lambda as well as a module M, this checks if lambda is an 
 * element of M. 
 *
 * It's only a true (1) - false(0) test. 
 *
 * lambda in M <=> M^(-1)*lambda in Z_K^n
 * 
 * *********************************************************************** */

IsModuleElt(grpring, lambda, M) =
{

 local(N, S, i);

 M = ModuleHNF(grpring , M);

 lambda = vector(length(lambda), i, lambda[i])~;  \\make it a column vector
 N = [concat(M[1], lambda), concat(M[2], [ matid(length(grpring.nfd.zk)) ])];

 S = ModuleSum(grpring , M , N);          

 if( S == M ,   
   return(1);                                 \\ if M < N
  , \\ else
   return(0);                                 \\ M is not included in N
 );
}

addhelp(IsModuleElt , "IsModuleElt(grpring, lambda, M): Given a group ring grpring in Grinit and an element lambda as well as a module M, this checks if lambda is an element of M. It's only a true (1) - false (0) test.");





/* ****************************** FindModuleElt ******************************
 *
 * FindModuleElt(grpring,M,d,s,{flag=0}): Given a group ring grpring in GrInit,
 * a module M, a n-component vector d of elements in the group ring and a 
 * n-component vector s of integers. This routine finds (an) element(s)
 * z = d_1^{c_1}...d_n^{c_n}, 0 <= c_i < s_i, such that z is an element of M.
 * 
 * If flag is omitted or zero the result are all elements z with the above
 * feature. If flag is nonzero search only one element which is included in M.
 * 
 * The result is a 2-component vector where the first component is a matrix A
 * which contains the exponents c_i (column of A) as above. The second one is
 * the vector of the elements z. 
 *
 * ************************************************************************* */

FindModuleElt(grpring , M , d , s , flag=0) = 
{

 local(n,c,z,A,mu);

 n = length(d);

 if(n != length(s), error("***   incomaptible vector length"));

 c = InitMultiCounter( vector(n) , s-vector(n,k,1) );    \\ initialize multi counter

 z = Vec();                             \\ z will contain the elements which are in M
 A = Mat();                             \\ A will contain the exponents c such that 
					\\ z = d_1^{c_1}...d_n^{c_n} is an element of M

 while(c,                               \\ c[1] contains the integers c_i, sucht that 0 <= c_i < s_i

   mu = GrPow(grpring , d , c[1]);      \\ compute z = d_1^{c_1}...d_n^{c_n}

   if( IsModuleElt(grpring , mu , M),   \\ if mu is an element of M

     A = concat(A,c[1]~);               \\ write the c_i in a column of A

     z = concat(z , mu);                \\ z contains all elements such that
					\\ z = d_1^{c_1}...d_n^{c_n}, 0 <= c_i < s_i, z in M     

     if(flag, return( [ A , z ]));      \\ we search only one element with the above feature.
					\\ so, if we have one return this element
   );

   c = IncMultiCounter(c);              \\ increase MultiCounter
 );

 return( [ A , z ]);

}

addhelp(FindModuleElt , "FindModuleElt(grpring,M,d,s,{flag=0}): Given a group ring grpring in GrInit, a module M, a n-component vector d of elements in the group ring and a n-component vector s of integers, this finds (an) element(s) z = d_1^{c_1}...d_n^{c_n}, 0 <= c_i < s_i, such that z is an element of M. If flag is omitted or zero the result are all elements z with the above feature. If flag is nonzero search only one element which is included in M. The result is a 2-component vector where the first component is a matrix A which contains the exponents c_i (column of A) as above. The second one is the vector of the elements z.");





/* ****************************** DualModule ******************************
 *
 * DualModule(grpring,M): Given a group ring grpring and a module M. This
 * computes the dual module of M. 
 *
 * We consider the non degenerate, symmetric bilinear form 
 *  
 * <?,?> : K[G] x K[G] -> K,   G=gal(L|K)
 *  
 * with   
 *
 *          { 1, if st=id
 *  <s,t> = {                   s,t in G
 *          { 0, otherwise
 *
 *
 * Now, solve for every j in {1,...,n} the system of linear 
 * equations given by
 *
 * <ld_i,ld_j^*> = kronecker-delta(ij), i=1,...,n.
 *
 *  Then, [ [ld_i^*, i=1..n], [a1^-1, ..., an^-1]] is the dual module.
 *
 * ********************************************************************** */

DualModule(grpring , M) =
{

 local(n,v,db,dM,ideals);

 n = length(M[1]);                               \\ number of basis elements
 v = vectorv(n);
 db = matrix(n,n);                            \\ to save the dual basis

 \\ we have to solve the linear equation Mx=v  
 dM = matrix(n,n,i,j,  M[1][,i]~ * grpring.grp.mtbi[,j] ); 
 
 for(i=1,n,
   v[i] = 1;
   db[,i] = matsolve(dM , v);                 \\ for every ld_i solve this equation
					      \\ then db gives the dualbasis
   v[i] = 0;
 );

 ideals = vector(n,i,idealinv(grpring.nfd, M[2][i]));

 db = ModuleHNF(grpring, [db, ideals]); 

 return(db);

}

addhelp(DualModule , "DualModule(grpring,M): Given a group ring grpring and a module M. This computes the dual module of M.");





/* ****************************** IsLocallyFree ******************************
 *
 * IsLocallyFree(grpring,a,R): Given a as an R-module in the group ring
 * grpring (K[G]). This tests if a is locally free over the module R, i.e.
 * it exists theta_p in a_p with R_p*theta_p = R_p for all p-ideals in O_K, 
 * p prime.
 * 
 * a is locally free over R <=> [a*MaxOrd : MaxOrd]_{O_K} = [a : R]_{O_K}, 
 * where MaxOrd is the maximal order in ther group ring and [_:_] denotes 
 * the module index over O_K.
 *
 * The result is a 3-component vector. The first component is 0 or 1, that 
 * means a is locally free or not. The second and the third component are the
 * module indices a described above.
 *
 * ************************************************************************* */

IsLocallyFree(grpring , a , R) =
{

 local(aM, l, r);

 aM = ModuleProd(grpring , a , grpring.MaxOrd); \\ compute a*MaxOrd

 l = ModuleIndex(grpring , aM , grpring.MaxOrd);\\ the module index [a*MaxOrd : MaxOrd]
 r = ModuleIndex(grpring , a , R);              \\ the module index [a : R] 

 return([ l == r , l , r ]); 

}

addhelp(IsLocallyFree , "IsLocallyFree(grpring,a,R): Given a as an R-module in the group ring grpring. This tests if a is locally free over the module R. The result is a 3-component vector like [ true (1)/false (0) , [aM : M]_{O_K} , [a : R]_{O_K} ].");





/* ****************************** IsFreeOverMax ******************************
 *
 * IsFreeOverMax(grpring,a): Given a group ring grpring in GrInit(K,G) and a
 * module a in K[G]. This tests if aM is free over the maximal order M. 
 *
 * aM is free over M if I_rho := [ aMe_rho : Me_rho ]_{O_Krho} is principal
 * for all rho in the K-irreducible characters D(G). e_rho is the 
 * corresponding idempotent.
 *
 * The result is a 3-component vector like 
 * [ true (10/ false (0) , I_rho , alpha ], where alpha is the result 
 * from bnfisprincipal.
 *
 * (q.v. W. Bley, Computing Associated Orders and Galois Generating Elements
 * of Unit Lattices, Lemma 2.5. JNT Vol. 62, No. 2. 
 *
 * ************************************************************************* */

IsFreeOverMax(grpring , a) =
{

 local(r,aM,n,Irho,alpha,idem,aux,phirho, arho);


 r = length(grpring.Kch);                     \\ number of components in the Wedderburn-decomposition

 aM = ModuleProd(grpring , a , grpring.MaxOrd);  \\ module product a*MaxOrd
 n = length(aM[1]);                              

 Irho = vector(r); alpha = vector(r);         \\ Irho will contain the module indices 

 for(i=1,r,

   idem = grpring.Kch[i].idem;                \\ the idempotents e_rho in K_rho

   for(j=1,n, 
     aux = GrMult(grpring , aM[1][,j] , idem);   \\ a*MaxOrd*e_rho

     phirho = Phirho(grpring , i , aux);      \\ Phi_rho(aMe_rho) in the i-th component

     \\ for the principal ideal test we need an absolute representation 

     phirho = rnfeltreltoabs(grpring.Kch[i].Krho , phirho);
     arho = rnfidealup(grpring.Kch[i].Krho, aM[2][j]);
     arho = idealgentohnf(grpring.Kch[i].Krhoabs, arho);
     arho = idealmul(grpring.Kch[i].Krhoabs, arho, phirho);
     Irho[i] = idealadd(grpring.Kch[i].Krhoabs , Irho[i] , arho); 
    );

   alpha[i] = bnfisprincipal(grpring.Kch[i].Krhoabs , Irho[i] );  \\ now, do the principal ideal test

 );

 for(k=1,r,
   if( alpha[k][1] != vector(length(alpha[k][1]))~ , 
     print("alpha[", k, "] = ", alpha[k]);
     return(0));  \\ return 0, if ideal is not principal
 );

 return([ 1 , Irho , alpha ]);

}

addhelp(IsFreeOverMax,"IsFreeOverMax(grpring,a): Given a group ring grpring in GrInit(K,G) and a module a. This tests if aM is free over the maximal order M. aM is free over M if I_rho := [ aMe_rho : Me_rho ]_{O_Krho} is principal for all rho in the K-irreducible characters D(G). e_rho is the corresponding idempotent. The result is a 3-component vector like [ true (1)/ false (0) , I_rho , alpha ], where alpha is the result from bnfisprincipal.");




/* ****************************** CalcMaxOrd ******************************
 *
 * CalcMaxOrd(grpring,{flag=0}): Computes the maximal order in the group
 * ring grpring. If flag is non-zero the result is in 'ModuleInitType'.
 *
 * ********************************************************************** */

CalcMaxOrd(grpring , flag=0) =
{

 local(var,r,v,MaxOrd,OKrho);

 var = variable(grpring.Kch[1].Krhoabs.pol);

 r = length(grpring.Kch);
 v = vector(r); 
 MaxOrd = [Mat(), Vec()];

 for(i=1,r,
   OKrho = grpring.Kch[i].Krho[7];            \\ pseudo basis 
   OKrho = StandardPseudoBasis(grpring.nfd, OKrho);
   \\ OKrho = PseudoBasisToBasis(grpring.nfd , OKrho, ivar);
   MaxOrd[2] = concat(MaxOrd[2], OKrho[2]);
   for(j=1,length(OKrho[2]),
     v[i] = OKrho[1][j];
     MaxOrd[1] = concat( MaxOrd[1], PhiInv(grpring,v)~ );
     v[i] = 0;
   );  
 );

 if(flag, 
     MaxOrd = ModuleInit(grpring , MaxOrd);  \\ return MaxOrd in 'ModuleInitType'
 );

 return(MaxOrd);

}

addhelp(CalcMaxOrd, "CalcMaxOrd(grpring,{flag=0}): Computes the maximal order in the group ring. If flag is nonzero, output the maximal order in 'ModuleInitType' (see manual for details). ");



/* ****************************** ModuleInv ******************************
 *
 * ModuleInv(grpring,a,R): Let a be a locally free R-module (i.e. an 
 * invertible R-ideal) in the group ring grpring. The routine computes 
 * the inverse of this ideal. 
 *
 * a and R must be given in 'ModuleInitType'.
 *
 * The inverse is given by a^(-1) = {z in K[G] | za subseteq R}. This 
 * implies (a^(-1))* = aR*, where * denotes the dual module.
 *
 * ******************************************************************** */

IdealInv(grpring , a , R) =
{

 local(aR,ainv);

 if(StructType(a) != ModuleInitType || StructType(R) != ModuleInitType,
   error("***   please apply ModuleInit first"));

 aR = ModuleProd(grpring , a.hnf , R.dual);   \\ compute the product aR*
 ainv = DualModule(grpring , aR);             \\ the inverse is the dual module of aR

 return(ainv);

}

addhelp(IdealInv , "IdealInv(grpring,a,R): Let a be a local free R-module (i.e. an invertible R-ideal) in the group ring grpring. The routine computes the inverse of this ideal. a and R must be given in 'ModuleInitType'.");





/* ****************************** IdealAbsToRel ******************************
 *
 * IdealAbsToRel(K,nf,rnf,elt): Lifts an ideal elt of the number field nf to 
 * a relative representation in the number field rnf with coefficients in K.
 * nf and rnf defines the same number field over K, but nf is an absolute 
 * representation and rnf is a relative representation. 
 *
 * ************************************************************************* */



IdealAbsToRel(K , Eabs , E , ideal) =
{

 local(var, A, pol, M1, M2, i, j, g, v);

 var = variable(E.pol);

 ideal = idealhnf(Eabs , ideal);      
 A = Eabs[7][8]^(-1) * ideal;
 pol = subst(E[1], var, var - E[11][3]*Mod(variable(K.pol), K.pol));
 
 M1 = Mat();  M2 = Vec();
 
 for (j=1, length(A),
     g = sum(i=1,length(A[,j]), A[i,j] * var^(i-1));
     g = lift( Mod(g, pol) );
     g = subst(g, var, var + E[11][3]*Mod(variable(K.pol), K.pol));
     v = vector(poldegree(pol));
     for (i=0,poldegree(g, var), 
         v[i+1] =  polcoeff(g, i, var);
     );
     M1 = concat(M1, v~);
     M2 = concat(M2, [ matid(length(K.zk)) ] );
 );
 ideal = StandardPseudoBasis(K, nfhnf(K, [M1, M2]));
 
 return(ideal);
}
     
 


addhelp(IdealAbsToRel , "IdealAbsToRel(K,nf,rnf,ideal): Lifts an ideal of the number field nf to a relative representation in the number field rnf with coefficients in K.");



/* ****************************** IdealIsPrimitive ******************************
 *
 * IdealIsPrimitive(nf,ideal): Given a number field nf in nfinit and an ideal
 * ideal this checks if ideal is primitive in nf. An ideal is primitiv if
 * the entry at position [n,n] is equal to 1.
 *
 * **************************************************************************** */

IdealIsPrimitive(nf , ideal) =
{

 local(n);

 ideal = idealhnf(nf , ideal);     \\ ideal in hnf

 n = matsize(ideal);

 if( ideal[n[1],n[2]] != 1,        \\ ideal is not primitive
  return(0);
 );

 return(1);

}

addhelp(IdealIsPrimitive , "IdealIsPrimitive(nf,ideal): Checks if ideal is primitive in the number field nf.");


/* ****************************** IdealPow ******************************
 *
 * IdealPow(nf,ideal,pow): Computes ideal^pow in the number field nf. 
 * ideal can also be given as a n-component vector of ideals and pow as 
 * a vector of integer. Than this computes the product 
 * prod(i=1,n,ideal[i]^pow[i]).
 *
 * ******************************************************************** */

IdealPow(nf , ideal , pow) = 
{

 local(t,p1,ap);

 if(type(pow) != "t_VEC",                     \\ if ideal and pow are not vectors
   ideal = [ideal]; pow = [pow]);             \\ handle them as vectors

 t = length(ideal);                           \\ number of ideals

 if(t != length(pow), error("*** incomatible vector length"));

 p1 = 1;

 for(i=1,t,
     ap = idealpow(nf , ideal[i] , pow[i]);
     p1 = idealmul(nf , p1 , ap);
 );

 return(p1);

}

addhelp(IdealPow , "IdealPow(nf,ideal,pow): Computes ideal^pow in the number field nf. ideal can also be given as a n-component vector of ideals and pow as a vector of integer. Than this computes the product prod(i=1,n,ideal[i]^pow[i]).");


/* ****************************** RedModIdeal ******************************
 *
 * reduces the group ring element lambda modulo the module Id. lambda and Id must
 * be contained in the maximal order.
 *
 * ******************************************************************** */



RedModIdeal(KG, lambda, Id)=
{
    local(v, i, a, b);

    Id = ModuleInit(KG, Id);       
    v = GrEltinMaxOrd(KG, lambda~);
    i = length(v);
    while (i > 0,
        b = v[i] % Id.zhnf[i,i];
        a = (v[i] - b) / Id.zhnf[i,i];
        v = v - a*Id.zhnf[,i];
        i = i - 1;
    );
    return( KG.MaxOrd.zbase * v );
}


addhelp(RedModIdeal , "RedModIdeal(KG,lambda,Id): Reduces the coefficient of the group ring element lambda modulo the ideal Id. It is assumed that lambda and Id are contained in KG.MaxOrd");
