\\  -*-gp-script-*-

/* ****************************** GrInit.gp ******************************
 * 
 * GrInit.gp: Contains all necessery functions to initialize and compute
 * in a group ring initialized by GrInit. There are also functions to
 * initialize the underlying group G. The underlying number field is 
 * always called K. The group G has to be initialized by Ginit.
 *
 * Group ring:
 *
 *      - GrInit(K,G)
 *      - GrMult(grpring,a,b=0)
 *      - GrPow(grpring,lambda,pow,{flag=0})
 *      - GrEltInv(grpring,lambda)
 *      - GrOKG(grpring);
 *      - GrAction(aut,relpol,lambda,th) 
 *      - Phi(grpring,lambda)
 *      - PhiInv(grpring,alpha)
 *      - Phirho(grpring,comp,lambda)
 *      - PhirhoModule(grpring,M,comp)
 *
 * 
 * Group:
 * 
 *      - Ginit(gen,ord,ivar=x)
 *      - Gmult(G,g,h)
 *      - GeltOrder(G,elt)
 *      - GaloisAut(grpring,rpol)
 *      - GaloisGen(nf,relpol,raut=0)
 *      - AutPow(aut,pow,relpol)
 *      - AllGroupElt(G,relpol)
 *
 * Auxiliary functions: NOT FOR PERSONAL USE 
 *
 *      - zz_GaloisAut_aut
 *      - zz_GaloisAut_gen
 *      - zz_Ginit_biform(mt)
 *      - zz_mtab(aut,rpol) 
 *      - zz_GaloisAut_sort(aut,gen,rpol) 
 *      - zz_Gbasistolist(G,g)
 *      - zz_Glisttobasis(G,i)
 *      - zz_CharOrd(G,chi)
 *      - zz_GrInit_CycloPolK(K,m)
 *      - zz_MakeList(G)
 *      - zz_GrInit_CompIdemSigma(G,chi,m,Krho,s)
 *
 * 
 * time-stamp: <Tue, Nov 02, 2004 - 14:57>
 *
 * author:     werner bley (mail: werner.bley@math.uni-augsburg.de)
 *             markus endres (mail: parigp@mendres.org)
 *
 * ********************************************************************* */





/* ****************************** GrInit structure ******************************
 *
 * The structure of a group ring in GrInit is a 3-component vector like
 * [ number field , group , Kch ]
 * where Kch is
 * [ ch[i], idemsigma[2], idemsigma[1], Krho, Krhoabs ]
 *
 * The structure of a group in Ginit is a 5-component vector like
 * [ generators , orders , element list , multiplication table , multiplication 
 * table for the bilinear form ]
 *
 * **************************************************************************** */

\\ Group ring structure initialized by GrInit
x.nfd    = x[1];                             \\ number field K of the group ring
x.grp    = x[2];                             \\ the group G
x.Kch    = x[3];                             \\ the K-irreducible characters Kch
x.MaxOrd = x[4];                             \\ the maximal order in ther group ring K[G]

\\ Group structure initialized by Ginit
x.ggen    = x[1];                            \\ the generators of the group G
x.ord     = x[2];                            \\ the order of the group G
x.eltlist = x[3];                            \\ element list of the group G
x.mt      = x[4];                            \\ a multiplication table for the group
x.mtbi    = x[5];                            \\ multiplication table for the bilinear form
                               
\\ the structure of Kch
x.ch      = x[1];                            \\ chi
x.idem    = x[2];                            \\ the idempotents
x.g       = x[3];                            \\ inverse image of x = zeta_m under phirho, where m = ord(chi)
x.Krho    = x[4];                            \\ the number field K_rho in relative representation over K
x.Krhoabs = x[5];                            \\ K_rho absolute over Q


true = 1; false = 0;





/* ****************************** GrMult ******************************
 *
 * GrMult(grpring,a,b=0): Given a group ring grpring in GrInit and two
 * elements a and b in this group ring, this computes the product of 
 * a and b in the group ring. 
 *
 * a can also be a vector of group ring elements. Than b must be 
 * omitted or zero. Than this computes the product of all elements in
 * the vector a. 
 *
 * ****************************************************************** */

GrMult(grpring , a , b=0) =
{

 local (n,c,k,r,i,j);


 n = length(grpring.grp.eltlist);             \\ number of elements in the group G
 c = vector(n);

 if(b,                                        \\ if b is given as an element of the group ring
   for(i=1,n,                       
     for(j=1,n, 
       k = Gmult(grpring.grp , i , j);        \\ compute the product of a and b
       c[k] += a[i] * b[j];
     );
   );
   return(c);                                 \\ return the product ab in the grpring
 );


 \\ else a = [a_1,...,a_r] is a vector of group ring elements and b is zero
 r = length(a);

 if(r == 1, 
   if(type(a[1])=="t_COL", 
     return(a[1]~)                            \\ only for technical reasons
    , \\ else
     return(a[1]);
  );
 );

 c = GrMult(grpring , a[1] , a[2]);

 for(k=3,r,
   c = GrMult(grpring , c , a[k]);
 );

 return(c);

}

addhelp(GrMult , "GrMult(grpring,a,b=0): Given a group ring grpring in GrInit and two elements a and b in this group ring, this computes the product of a and b in the group ring. a can also be a vector of group ring elements. Than b must be omitted or zero. Than this computes the product of all elements in the vector a.");




/* ****************************** GrMultModIdeal ******************************
 *
 * GrMultModIdeal(grpring,a,b=0, F): Given a group ring grpring in GrInit and two
 * elements a and b in this group ring, this computes the product of 
 * a and b in the group ring modulo the ideal F. 
 * a,b,F must be contained in the maximal order.
 *
 * a can also be a vector of group ring elements. Than b must be 
 * omitted or zero. Than this computes the product of all elements in
 * the vector a. 
 *
 * ****************************************************************** */

GrMultModIdeal(grpring , a , b, F) =
{

 local (n,c,k,r,i,j);


 n = length(grpring.grp.eltlist);             \\ number of elements in the group G
 c = vector(n);

 if(b,                                        \\ if b is given as an element of the group ring
   for(i=1,n,                       
     for(j=1,n, 
       k = Gmult(grpring.grp , i , j);        \\ compute the product of a and b
       c[k] += a[i] * b[j];
     );
   );
   c = RedModIdeal(grpring, c~, F);
   return(c~);                                 \\ return the product ab in the grpring
 );


 \\ else a = [a_1,...,a_r] is a vector of group ring elements and b is zero
 r = length(a);

 if(r == 1, 
   if(type(a[1])=="t_COL", 
     return(a[1]~)                            \\ only for technical reasons
    , \\ else
     return(a[1]);
  );
 );

 c = GrMultModIdeal(grpring , a[1] , a[2], F);

 for(k=3,r,
   c = GrMultModIdeal(grpring , c , a[k], F);
 );

 return(c);

}

addhelp(GrMultModIdeal , "GrMultModIdeal(grpring,a,b=0,F): Given a group ring grpring in GrInit and two elements a and b in this group ring, this computes the product of a and b in the group ring modulo the ideal F. a,b and F must be contained in the maximal order. a can also be a vector of group ring elements. Than b must be omitted or zero. Than this computes the product of all elements in the vector a.");





/* ****************************** GrPow ******************************
 *
 * GrPow(grpring,lambda,pow,{flag=0}): Given a group ring in GrInit
 * and an element lambda in the group ring, this computes lambda to 
 * the pow-th power.
 * 
 * lambda can also be a s-component vector 
 * lambda=[lambda_1,...,lambda_s] and pow=[n_1,...,n_s] a vector of 
 * integers. Than the result is the product 
 * prod_{i=1}^n lambda_i^{n_i}. If flag is non-zero, the result is 
 * a s-component vector like [lambda_1^{n_1},...,lambda_s^{n_s}].
 * 
 * ***************************************************************** */

GrPow(grpring , lambda , pow , flag=0) = 
{

 local(n,v,id,i,k,p);

 \\ the lambdas can be given in matrix representation
 if(type(pow) == "t_INT",  \\ 
   lambda = [lambda]; pow = [pow];
 
 , \\ else

  if(type(lambda) != "t_VEC",
     lambda = Vec(lambda);
  );
 );

 n = length(lambda);                \\ lambda is a n-component vector with vector as entries
 if(n != length(pow) , error("incompatible vector length\n") );

 id = vector(length(lambda[1]));    \\ identity
 id[1] = Mod(1 , grpring.nfd.pol);

 for(i=1,n,
   if((p=pow[i]) != 0,
     lambda[i] = GrMult( grpring , vector(abs(p) , k , lambda[i]) );
     if(p < 0, lambda[i] = GrEltInv(grpring , lambda[i]));
     ,
     \\ else
     lambda[i] = id;
   );
 );

 if(flag , return(lambda));

 return( GrMult(grpring , lambda) );

}

addhelp(GrPow , "GrPow(grpring,lambda,pow,{flag=0}): Given a group ring in GrInit and an element lambda in the group ring, this computes lambda to the pow-th power. lambda can also be a s-component vector lambda=[lambda_1,...,lambda_s] and pow=[n_1,...,n_s] a vector of integers. Than the result is the product prod_{i=1}^n lambda_i^{n_i}. If flag is non-zero, the result is a s-component vector like [lambda_1^{n_1},...,lambda_s^{n_s}].");



/* ****************************** GrPow ModIdeal******************************
 *
 * GrPowModIdeal(grpring,lambda,pow,F,{flag=0}): Given a group ring in GrInit
 * and an element lambda in the group ring, this computes lambda to 
 * the pow-th power modulo the ideal F. 
 * lambda and F must be contained in the maximal order.
 * 
 * lambda can also be a s-component vector 
 * lambda=[lambda_1,...,lambda_s] and pow=[n_1,...,n_s] a vector of 
 * integers. Than the result is the product 
 * prod_{i=1}^n lambda_i^{n_i}. If flag is non-zero, the result is 
 * a s-component vector like [lambda_1^{n_1},...,lambda_s^{n_s}].
 * 
 * ***************************************************************** */

GrPowModIdeal(grpring , lambda , pow , F, flag=0) = 
{

 local(n,v,id,i,k,p);

 \\ the lambdas can be given in matrix representation
 if(type(pow) == "t_INT",  \\ 
   lambda = [lambda]; pow = [pow];
 
 , \\ else

  if(type(lambda) != "t_VEC",
     lambda = Vec(lambda);
  );
 );

 n = length(lambda);                \\ lambda is a n-component vector with vector as entries
 if(n != length(pow) , error("incompatible vector length\n") );

 id = vector(length(lambda[1]));    \\ identity
 id[1] = Mod(1 , grpring.nfd.pol);

 for(i=1,n,
   if((p=pow[i]) != 0,
     lambda[i] = GrMult( grpring , vector(abs(p) , k , lambda[i]) );
     lambda[i] = RedModIdeal(grpring, lambda[i]~, F);
     if(p < 0, error("negative exponent in GrPowModIdeal\n"););
     ,
     \\ else
     lambda[i] = id;
   );
 );

 if(flag , return(lambda));

 return( GrMultModIdeal(grpring , lambda, 0, F) );

}

addhelp(GrPowModIdeal , "GrPowModIdeal(grpring,lambda,pow,F,{flag=0}): Given a group ring in GrInit and an element lambda in the group ring, this computes lambda to the pow-th power modulo the ideal F. pow must nut be negativ, lambda and F must be contained in the maximal order. lambda can also be a s-component vector lambda=[lambda_1,...,lambda_s] and pow=[n_1,...,n_s] a vector of integers. Than the result is the product prod_{i=1}^n lambda_i^{n_i}. If flag is non-zero, the result is a s-component vector like [lambda_1^{n_1},...,lambda_s^{n_s}].");



/* ****************************** GrPowModF ******************************
 *
 * As GrPow, but does the computation modulo the conductor F. This function
 * assumes that (R/F)^* is already computed.
 * 
 * ***************************************************************** */


GrPowModF(grpring , RqF, lambda , pow , flag=0) = 
{

 local(n,v,id,i,k,p,a);

 \\ the lambdas can be given in matrix representation
 if(type(pow) == "t_INT",  \\ 
   lambda = [lambda]; pow = [pow];
 
 , \\ else

  if(type(lambda) != "t_VEC",
     lambda = Vec(lambda);
  );
 );

 n = length(lambda);   \\ lambda is a n-component vector with vector as entries
 if(n != length(pow) , error("incompatible vector length\n") );

 for(i=1,n,
     a = RqFLog(grpring,RqF, lambda[i]);
     for (k=1, length(a),
         a[k] = (a[k] * pow[i]) % RqF[2][k];
     );
     lambda[i] = GrPow(grpring, RqF[3], a);
 );

 if(flag , return(lambda));

 return( GrMult(grpring , lambda) );

}


addhelp(GrPowModF, "GrPowModF(grpring , RqF, lambda , pow , flag=0) as GrPow, but modulo F. (R/F)^* must be computed first.");



/* ****************************** GrEltInv ******************************
 *
 * GrEltInv(grpring,lambda): Given a group ring grpring and an element
 * lambda. This compute the inverse element of lambda in the group ring. 
 * 
 * ******************************************************************** */

GrEltInv(grpring , lambda) = 
{

 local(r,aux,inv,i);

 r = length(grpring.Kch); 
 aux = Phi(grpring , lambda);                 \\ we will reverse the group ring element
					      \\ lambda due to the Wedderburn-decompositon

 for(i=1,r,

   trap(,                                     \\ try to invert the element, 
     error("non invertible element"),         \\ if it is not possible, return an error
     inv = aux[i]^(-1);
   );

   aux[i] = inv;                      \\ reversing
 );

 aux = PhiInv(grpring , aux);                 \\ and back to the group ring

 return(aux);

}

addhelp(GrEltInv , "GrEltInv(grpring,lambda): Given a group ring grpring and an element lambda. This compute the inverse element of lambda in the group ring.");

/* ****************************** GrAction ******************************
 *
 * GrAction(aut,relpol,lambda,th): Returns the action of a group ring
 * element lambda on an element th. aut are the galois automorphisms in
 * Gal(L|K) (GaloisAut), relpol is the relative defining polynomial for
 * L|K. 
 *
 * ******************************************************************** */

GrAction(aut , relpol , lambda , th) = 
{

 local(n,var,result);

 n = length(lambda); var = variable(relpol);
 result = 0;

 for(i=1,n,
   result += (lambda[i] * subst(th , var , aut[i])) % relpol;
 );

 return(result);

}

addhelp(GrAction , "GrAction(aut,relpol,lambda,th): The action of a group ring element lambda on an element th. aut are the galois automorphisms given by GaloisAut, relpol the relative defining polynomial for the field extension.");





/* ****************************** Phi ******************************
 *
 * Phi(grpring,lambda): Computes the representation of lambda in the
 * Wedderburn decomposition of the group ring grpring. The result is a
 * r-component vector, where r is the number of the K-irreducible
 * characters Kch. 
 * 
 * *************************************************************** */

Phi(grpring , lambda) =
{

 local (r,alpha,i);

 r = length(grpring[3]);                      \\ number of K-irreducible characters

 alpha = vector(r);

 for (i=1, r, 
\\   alpha[i] = Phirho(grpring[2] , grpring[3][i][1] , grpring[3][i][4] , lambda) );
  alpha[i] = Phirho(grpring , i , lambda) ); 

 return(alpha);

}

addhelp(Phi , "Phi(grpring,lambda): Computes the representation of lambda in the Wedderburn decomposition of the group ring grpring.");





/* ****************************** PhiInv ******************************
 *
 * PhiInv(grpring,alpha): Given a group ring grpring and an element
 * alpha in the r-component representation of the Wedderburn 
 * decomposition. This computes an corresponding element in the group
 * ring. 
 *
 * ****************************************************************** */

PhiInv(grpring , alpha) =
{

 local (r,n,lambda,rho,a,mu,g,s,t,i);

 r = length(grpring.Kch);                     \\ number of Kch-components

 n = length(grpring.grp.eltlist);             \\ n = group order
 lambda = vector(n);                          \\ will be the group ring element

 for (rho=1, r,
   a = lift(alpha[rho]);       \\ polynomial in x with coefficients in the number field K
   mu = vector(n);
   g = grpring[3][rho][4][1];  \\ relative polynomial of the rho-th component
   s = grpring[3][rho][3];     \\ s |--> zeta_m = x under chi_rho 

   t = 1;
   for (i=0, poldegree(g)-1,
     mu[t] = polcoeff(a, i);
     t = Gmult(grpring.grp, t, s);
   );

   lambda = lambda + GrMult(grpring, mu, grpring[3][rho][2]);
 );

 return(lambda);

}

addhelp(PhiInv , "PhiInv(grpring,alpha): Given a group ring grpring and an element alpha in the r-component representation of the Wedderburn decomposition. This computes an corresponding element in the group ring.");





/* ****************************** Phirho ******************************
 *
 * Phirho(grpring,comp,lambda): Computes the representation of the group
 * ring element lambda in the rho-th component of the K-irreducible
 * character chi. 
 *
 * ****************************************************************** */

\\Phirho(G, chi, E, lambda) =
Phirho(grpring,comp,lambda) =
{

 local(G,chi,var,m,g,a,i,j,s,t);

 G = grpring.grp;
 chi = grpring.Kch[comp].ch;
 var = variable(grpring.Kch[comp].Krhoabs.pol);


 m = zz_CharOrd(G, chi);
 \\ g = E[1];   /* das relative Polynom */
 g = grpring.Kch[comp].Krho[1];

     a = Mod(0, g);
 for (i=1, length(lambda),
   s = zz_Glisttobasis(G, i);
   t = sum(j=1, length(s), (m*chi[j]*s[j])/G[2][j]) % m;

   a = a + Mod(var^t, g) * lambda[i];
 );

 return(a);

}

addhelp(Phirho , "Phirho(grpring,comp,lambda): Computes the representation of the group ring element lambda in the rho-th component comp of the Wedderburn decomposition of grpring.");



/* ****************************** Groups ******************************
 *
 *
 *
 * ****************************************************************** */

/* ****************************** Ginit ******************************
 *
 * Ginit(gen,ord): Initializes a group. gen is a vector of 
 * generators and ord a vector of orders. The result is a 
 * 5-component vector like 
 * [ G.ggen , G.ord , G.eltlist , G.mt , G.mtbi ],
 * where G.ggen are the generators of the group and G.ord there orders.
 * G.eltlist is a list of the group elements, G.mt the multiplication 
 * table for these elements. G.mtbi is a multiplication table for the
 * bilinear form.
 *
 * The bilinear form is given by
 *
 *         { 1, if gh==id
 * <g,h> = {               g,h in G
 *         { 0, otherwise
 *
 * and is needed to compute a dual basis of a module in a group ring
 * (see DualModule). 
 * 
 * ***************************************************************** */

Ginit(gen , ord) = 
{

 local(G,list,n,T,v);

 \\ [ generators , odres , list of elements , multiplication table , mt for the bilinear form ]
 G = [gen, ord, 0, 0, 0];        \\ will contain the group
 list = zz_MakeList([gen, ord]);

 G[3] = list;
 n = length(list);  r = length(gen);
 T = matrix(n,n,i,j,0);

 for (i=1, n,
   for (j=1, n,
     v = vector(r, k, (list[i][k] + list[j][k]) % ord[k]);
     T[i,j] = zz_Gbasistolist(G, v);
   );
 );

 G[4] = T;                                   \\ the multiplication table of the group
 G[5] = zz_Ginit_biform(G[4]);               \\ multiplication table of the bilinear form

 return(G);

}

addhelp(Ginit , "Ginit(gen,ord): Initializes a group. gen is a vector of generators and ord a vector of orders. The result is a 5-component vector like [ G.ggen, G.ord, G.eltlist, G.mt, G.mtbi ].");





/* ****************************** Gmult ******************************
 * 
 * Gmult(G,g,h): Computes the element gh in the group G given by 
 * Ginit.
 *
 * ***************************************************************** */

Gmult(G, g, h) = 
{ 
 return(G[4][g,h]); 
}

addhelp(Gmult , "Gmult(G,g,h): Computes the element gh in the group G given by Ginit.");

Ginv(G, g) = 
{ 
 local(i);

 for (i=1,length(G.eltlist),
   if (G.mt[g,i] == 1, return(i))
 );
}





/* ****************************** GeltOrder ******************************
 * 
 * GeltOrder(G,elt): Computes the order of the element elt in the group G.
 *
 * *********************************************************************** */

GeltOrder(G , elt) =
{

 local(t,u);

 if( type(G) != "t_MAT", mt = G.mt);    \\ G can be the multiplication table 

 t = elt; u = 1;

 while(t != 1,
   t = mt[elt,t]; u++;
 );

 return(u);

}

addhelp(GeltOrder , "GeltOrder(G,elt): Computes the order of the element elt in the group G.");





/* ****************************** GaloisAut ******************************
 *
 * GaloisAut(grpring,relpol): Let grpring be a group ring in GrInit(K,G) 
 * with G = Gal(L|K) for a field extension L|K and rpol be the relative
 * defining polynomial for L|K. This computes the galois automorphisms
 * in Gal(L|K). 
 *
 * ********************************************************************* */

GaloisAut(grpring,relpol) = 
{

 local(var,apol,aut,c,ix,gen);

 var = variable(relpol);

 apol = rnfequation(grpring.nfd , relpol); 

 aut = zz_GaloisAut_aut(grpring.nfd , apol , relpol);   \\ compute all galois automorphisms

 if( (c=aut[1]) != var,       \\ if the first automorphism isn't the identity, 
   ix = vecfind(aut,var);     \\ search the identity map und change
   aut[ix] = c;
   aut[1] = var;
 );


 gen = zz_GaloisAut_gen(aut , grpring.grp.mt, relpol);           \\ now, find a group generator, return [ix,gen,ord]

 aut = zz_GaloisAut_sort(aut , gen , relpol);              \\ 

 return([aut,gen]);

}

addhelp(GaloisAut , "GaloisAut(grpring,relpol): Let grpring be a group ring in GrInit(K,G) with G = Gal(L|K) for a field extension L|K and relpol be the relative defining polynomial for L|K. This computes the galois automorphisms in Gal(L|K).");



/* ****************************** GaloisGen ******************************
 * 
 * GaloisGen(nf,relpol,raut=0): Computes the generators of the galois group
 * given by raut. If raut is zero, first compute the galois automorphisms,
 * then the generators. relpol is the relative defining polynomial for L|K.
 *
 * ********************************************************************* */

GaloisGen(nf , relpol , raut=0) = 
{

 local(var,apol,aut,c,ix,gen,mtab);

 var = variable(relpol);

 apol = rnfequation(nf, relpol); 

if(!raut,
 aut = zz_GaloisAut_aut(nf , apol , relpol);   \\ compute all galois automorphisms
 ,
 aut = raut;
);

 if( (c=aut[1]) != var,       \\ if the first automorphism isn't the identity, 
   ix = vecfind(aut,var);     \\ search the identity map und change
   aut[ix] = c;
   aut[1] = var;
 );


 mtab = zz_mtab(aut,relpol);  \\ generate the multiplication table for the galois automorphisms

 gen = zz_GaloisAut_gen(aut , mtab, relpol);   \\ search the generators

 return(gen);

}

addhelp(GaloisGen , "GaloisGen(nf,relpol,{raut=0}): Computes the generators of the galois group given by raut. If raut is zero, first compute the galois automorphisms, then the generators. relpol is the relative defining polynomial for L|K.");


/* ****************************** NewGaloisGen ******************************
 * 
 * NewGaloisGen(L): Computes the generators of the galois group
 * of the relative number field L.
 *
 * ********************************************************************* */

NewGaloisGen(L) = 
{

 local();

 relpol = L[1];
 var = variable(relpol);

 apol = L.pol;   \\ the absolute equation

 aut = New_zz_GaloisAut_aut(L);   \\ compute all galois automorphisms
 if( (c=aut[1]) != var,       \\ if the first automorphism isn't the identity, 
   ix = vecfind(aut,var);     \\ search the identity map und change
   aut[ix] = c;
   aut[1] = var;
 );


 mtab = zz_mtab(aut,relpol);  \\ generate the multiplication table for the galois automorphisms

 gen = zz_GaloisAut_gen(aut , mtab,relpol);   \\ search the generators

 return(gen);

}

addhelp(GaloisGen , "GaloisGen(nf,relpol,{raut=0}): Computes the generators of the galois group given by raut. If raut is zero, first compute the galois automorphisms, then the generators. relpol is the relative defining polynomial for L|K.");





/* ****************************** AutPow ******************************
 *
 * AutPow(aut,pow,relpol): Computes the automorphism aut^pow.
 *
 * ****************************************************************** */

AutPow(aut,pow,relpol) =
{

local(var,result);

var = variable(aut);

if (pow < 0, pow = pow % AutOrder(aut, relpol));
if(pow == 0, return(var));
if(pow == 1, return(aut));

result = aut;
for(i=2,pow,
 result = subst(result,var,aut) % relpol;
);

return(result);

}

addhelp(AutPow , "AutPow(aut,pow,relpol): aut^pow % relpol.");


AutOrder(aut,relpol) =
{

    local(var,iso, k);

    var = variable(aut);

    iso = aut; k=1;
    while (iso != var, 
        iso = subst(iso, var, aut) % relpol;
        k = k+1;
    );
    return(k);
}




/* ****************************** AllGroupElt ******************************
 * 
 * AllGroupElt(G,relpol): Given a group G with the galois generators in 
 * G.grp.ggen and the relative defining polynomial for L|K with Gal(L|K),
 * this computes all elements in Gal(L|K).
 *
 * *********************************************************************** */

AllGroupElt(G , relpol) =
{

local(var,nofelt,nofgen,aut,g, iso);

var = variable(G.ggen[1]);

nofelt = length(G.eltlist); nofgen = length(G.ggen);

aut = vector(nofelt);
for(i=1,nofelt,

 g = var; 
 for(j=1,nofgen,
   iso = AutPow(G.ggen[j] , G.eltlist[i][j] , relpol) % relpol;
   g = subst(g, var, iso) % relpol; 
 );

 aut[i] = g;

);

return(aut);

}

addhelp(AllGroupElt , "AllGroupElt(G,relpol): Given a group G with the galois generators in G.grp.ggen and the relative defining polynomial for L|K with Gal(L|K), this computes all elements in Gal(L|K).");







/* ****************************** Auxiliary functions ******************************
 * 
 *                              - NOT FOR PERSONAL USE -
 * 
 * ************************************************************************** */

/* ****************************** New_zz_GaloisAut_aut ******************************
 * 
 * New_zz_GaloisAut_aut(K,apol,rpol): Not for personal use! 
 * 
 * Given a number field K, an absolute (apol) and relative (rpol) defining 
 * polynomial for the field extension E|K, New_zz_GaloisAut_aut computes the 
 * relative galois automorphisms Gal(E|K). 
 *
 * **************************************************************************** */

\\ not for personal use
New_zz_GaloisAut_aut(L) =
{

 local(base_var, top_var, E, N, n, H, delta, alpha, beta, iso);

 base_var = variable(L[10].pol);
 top_var = variable(L.pol);
 E = nfinit(L.pol);

 N = nfgaloisconj(E,1);

 \\ now, N is a set of automorphisms, but some of them are false,
 \\ they don't fix K. so, we need to test them!

 n = length(N);                               \\ number of automorphisms

 H = [];  
 H_db = [];                                    \\ will be the result

 delta = lift( rnfeltreltoabs(L, top_var ) );   \\ L = K(delta)
 alpha = rnfeltup(L, base_var);
 beta = lift( alpha );   \\ K = Q(alpha), alpha as an element in L/K 
 for(i=1,n,    
   if(subst(beta , top_var , Mod(N[i], L.pol)) == alpha,\\ test each automorphism     
     \\ iso = subst(delta, top_var,  Mod(N[i], L.pol) );
     \\ H_db = concat(H_db, N[i]);
     iso = rnfeltabstorel(L, N[i]);
     \\ the next modification is necessary, since the root of the absolute polynomial
     \\ may not be the root of the relative polynomial (see Remarks concerning L[11]
     \\ in the description of rnfinit
     iso = iso - L[11][3]*Mod(base_var, L[10].pol); 
     H=concat(H,lift(iso));                  \\ if the automorphism fixes K save it
   );
 );
 H_db = H;

 return(H);                                   \\ return the automorphisms

}

addhelp(New_zz_GaloisAut_aut , "zz_GaloisAut_aut(K,apol,rpol): Not for personal use! This functions is used by GaloisAut which computes the galois group Gal(E|K).");

zz_GaloisAut_aut(K , apol , rpol) =
{

 local(var,E,N,n,H);

 var = variable(rpol);

 E = nfinit(apol);                            \\ initialize E|Q
 
 N = nfgaloisconj(E,1);
 \\  N = nfgaloisconj(E,1) % rpol;                \\ compute the group of conjugates from L|Q 

 \\ now, N is a set of automorphisms, but some of them are false,
 \\ they don't fix K. so, we need to test them!

 n = length(N);                               \\ number of automorphisms

 H = [];                                      \\ will be the result

 for(i=1,n,                                        
   if(subst(rpol , var , Mod(N[i],rpol)) == 0,\\ test each automorphism
     H=concat(H,N[i]);                        \\ if the automorphism fixes K save it
   );
 );

 return(H);                                   \\ return the automorphisms

}

addhelp(zz_GaloisAut_aut , "zz_GaloisAut_aut(K,apol,rpol): Not for personal use! This functions is used by GaloisAut which computes the galois group Gal(E|K).");





/* ****************************** zz_GaloisAut_gen ******************************
 * 
 * zz_GaloisAut_gen(aut,mt,relpol): 
 * Not for personal use! This function is used by 
 * GaloisAut to compute a generator of the galois automorphisms. aut contains 
 * all automorphisms and mt is a multiplication table of the group.
 *
 * **************************************************************************** */

\\ not for personal use
zz_GaloisAut_gen(aut , mt, relpol) = 
{

 local(n,L,v,c,i,j,k,H,S,gen,ord,U, iso, var);

 if(type(aut) != "t_COL", aut = aut~);

 n = length(aut);                             \\ number of automorphisms

 L = matrix(n,n^2-n*(n-1)/2); v = vectorv(n); \\ G ~ Z^n/L

 c=0;
 for(i=2,n,

   for(j=i,n,

    k=mt[i,j];                                \\ g_i*g_j = g_k

    v[i] += 1; v[j] += 1; v[k] =-1 ;

    L[,c++]=v;

    v[i]=0; v[j]=0; v[k]=0;

   );
 );

 for(i=1,n,

   L[i,c++] = GeltOrder(mt,i);              \\ compute the order of the elements

 );

 H = mathnf(L);                               \\ compute HNF
 S = matsnf(H,1);                             \\ computes smith normal form [U,V,S]
					      \\ such that S = UHV
 var = variable(relpol);
 U = S[1]^(-1);
 v = [];
 \\ v = aut~ * S[1]^(-1);
 for (i=1, length(U),
    iso = AutPow(aut[1], U[1,i], relpol);
    for (j=1, length(U[,i]),
        iso =  subst(iso, var, AutPow(aut[j], U[j,i], relpol)) % relpol;
    );
    v = concat(v, iso);
 );    

 gen=[]; ord=[]; 

 for(i=1,n,

    if(S[3][i,i] != 1,

      gen = concat(gen , v[i]);               \\ the generators
      ord = concat(ord , S[3][i,i]);          \\ the orders

     );
 );

return( [ gen , ord ] );                      \\ return [ generator , orders ]
 
}

addhelp(zz_GaloisAut_gen , "zz_GaloisAut_gen(aut,mt,relpol): Not for personal use! This function is used by GaloisAut to compute a generator of the galois automorphisms. aut contains all automorphisms and mt is a multiplication table of the group.");





/* ****************************** zz_Ginit_biform ******************************
 *
 * zz_Ginit_biform(mt): Given the multiplication table mt of the galois 
 * automorphisms, this computes the multiplication table for the bilinear form.
 *
 *         { 1, if gh==id
 * <g,h> = {               g,h in G
 *         { 0, otherwise
 *
 * Only necessary to compute a dual module in a group ring (see DualModule).
 * 
 * *************************************************************************** */

\\ not for personal use
zz_Ginit_biform(mt) = 
{

 local(n);
 n = matsize(mt)[1];

 for(i=1,n,
   for(j=1,n,
     if(mt[i,j] != 1, mt[i,j]=0);
   );
 );

 return(mt);

}

addhelp(zz_Ginit_biform , "zz_Ginit_biform(mt): Not for personal use! Only necessary to compute a dual module in a group ring (see DualModule).");





/* ****************************** zz_mtab ******************************
 *
 * zz_mtab(aut,rpol): Not for personal use! This function is used by
 * GaloisAut to generate a multiplication table for the galois 
 * automorphisms. 
 * 
 * Given the galois automorphisms aut form Gal(L|K) and the relative
 * defining polynomial for L|K this computes a multiplication table
 * for the automorphisms. 
 *
 * **************************************************************** */

zz_mtab(aut , rpol) = 
{

 local(var,n,MT,k,s,ix);

 var = variable(rpol); n = length(aut);       \\ number of automorphims
 MT = matrix(n,n);                            \\ MT will contain the multiplication table

 k=1;

 for(i=1,n,

   for(j=k,n,

    s=subst(aut[j] , var , aut[i]) % rpol;    \\ multiply the elements
					      \\ in the galois group

    ix = vecfind(aut , s);                    \\ which element is equal to s

    MT[i,j] = ix; MT[j,i] = ix;               \\ it's an abelian galois group

   );

   k+=1;

 );

 return(MT);

}

addhelp(zz_mtab , "zz_mtab(aut,rpol): Not for personal use! Computes the multiplication table for the galois automorphisms aut, aut=Gal(L|K). rpol is a relative defining polynomial for L|K.");





/* ****************************** zz_GaloisAut_sort ******************************
 *
 * zz_GaloisAut_sort(aut,gen,rpol): Not for personal use! This function is used
 * by GaloisAut to sort the automorphisms, such that
 *
 * aut = [ id, gen, gen^2, gen^3, ... , gen^{ord-1} ]
 * 
 * ***************************************************************************** */

\\ not for personal use
zz_GaloisAut_sort(aut , gen , rpol) = 
{

 local(ord,var,v);  

 ord = gen[2][1]; gen = gen[1][1];

 var = variable(gen); v = vector(ord); 
 v[1] = aut[1]; v[2] = gen;

 for(i=3,ord,
  v[i] = subst(v[i-1],var,gen) % rpol;
 );

 return(v);

}

addhelp(zz_GaloisAut_sort , "zz_GaloisAut_sort(aut,gen,rpol): Not for personal use. This function is used by GaloisAut to sort the automorphisms such that aut = [ id, gen, gen^2, gen^3, ... , gen^{ord-1} ].");





\\ not for personal use 
zz_Gbasistolist(G , g) =
{

 local(i,h);

 h = vector(length(g), i, g[i] % G[2][i]);

 for (i=1, length(G[3]), 
   if (G[3][i] == h, return(i));
 );

}

addhelp(zz_Gbasistolist , "zz_Gbasistolist(G,g): Not for personal use!");





\\ not for personal use 
zz_Glisttobasis(G , i) =
{
 return(G[3][i]);
}

addhelp(zz_Glisttobasis , "zz_Glisttobasis(G,g): Not for personal use!");





\\ not for personal use 
zz_CharOrd(G , chi) =
{

 local (m,i);

 m = 1;

 for (i=1, length(G[1]), 
   m = lcm(m, G[2][i] / gcd(G[2][i], chi[i]))
 );

 return(m);
}

addhelp(zz_CharOrd , "zz_CharOrd(G,chi): Not for personal use!");





\\ not for personal use 
zz_GrInit_CycloPolK(K , m , ivar=x)=
{

 local (f,g,i,v);

 f = polcyclo(m , ivar);
 g = sum(i=0, poldegree(f) , Mod(polcoeff(f, i),K.pol) * ivar^i);

 v = nffactor(K , g);

 return(v[1,1]);

}

addhelp(zz_GrInit_CycloPol , "zz_GrInit_CycloPol(K,m,{ivar=x}): Not for personal use!");





\\ not for personal use 
zz_MakeList(G) =
{

 local(r,low,high,i,c,ch);

 r = length(G[1]); low = vector(r); high = vector(r, i, G[2][i] - 1);

 c = InitMultiCounter(low, high);
 ch = [];

 while(c,
   ch = concat(ch, [c[1]]);
   c = IncMultiCounter(c);
 );

 return(ch);

}

addhelp(zz_MakeList , "zz_MakeList(G): Not for personal use!");





\\ not for personal use 
zz_GrInit_CompIdemSigma(G , chi , m , Krho , s) =
{  

 local(var,g,n,r,idem,i,t,j,c);

 if (m == 1, g = 1, g = -1);

 var = variable(Krho[1]);
 n = length(G[3]); r = length(G[1]);
 idem = vector(n);

 

 for (i=1, n,
   t = sum(j=1, r, (m*chi[j]*G[3][i][j]) / G[2][j]) % m;

   if (t == 1 & g == -1, g = i);

   c = sum(j=1, length(s), Mod(var^(s[j]*t), Krho[1]));
   idem[i] = rnfeltdown(Krho, c / n);

 );

 return([g, idem]);

}

addhelp(zz_GrInit_CompIdemSigma , "zz_GrInit_CompIdemSigma(G,chi,m,Krho,s): Not for personal use!");



/* ****************************** GrInit ******************************
 *
 * GrInit(K,G): Initialize a group ring with underlying number field K
 * in bnfinit and a group G in Ginit. This computes all necessary 
 * data to work with and within a group ring. 
 * The result is a 4-component vector like
 * [ number field K , group G , K irreducible characters Kch , maximal order ] 
 * KG[1] is the number field K (KG.nfd), KG[2] is the group (KG.grp) and
 * KG[3] is equal to the K irreducible characters (KG.Kch).
 * Kch is 
 * [ chi (KG.Kch.ch) , idemsigma[2] (KG.Kch.idem) , 
 *   idemsigma[1] (KG.Kch.g) , Krho (KG.Kch.Krho) , 
 *   Krhoabs (KG.Kch.Krhoabs) ]
 *
 * ****************************************************************** */

GrInit(K, G, ivar=x) =
{

 local(ch,r,i,Kch,chused,j,m,g,Krho,Krhoabs,s,idemsigma,MaxOrd);

 ch = zz_MakeList(G);
 r = length(G[1]); i = 1; Kch = []; chused = vector(length(ch));

 while (i <= length(ch),

   m = zz_CharOrd(G , ch[i]);
   g = zz_GrInit_CycloPolK(K, m, ivar);

   Krho = rnfinit(K , g);
   s = [1];

   for (j=2, m,
     if( Mod(subst(g, ivar, ivar^j), g) == Mod(0, g), s = concat(s , [j]) );
   );

   for (j=1, length(s),
     chused[ zz_Gbasistolist(G , s[j]*ch[i]) ] = 1;
   );

   for (j=1, length(s), s[j] = (-s[j]) % m); 
     idemsigma = zz_GrInit_CompIdemSigma(G , ch[i] , m , Krho , s);             \\ [g, idem]
     Krhoabs = bnfinit( rnfequation(K,Krho[1]) );
     Kch = concat(Kch, [ [ch[i] , idemsigma[2] , idemsigma[1] , Krho , Krhoabs] ]); 
     while (i <= length(ch) & chused[i] == 1, i= i+1);

   );


 \\ compute the maximal order MaxOrd in the group ring K[G]
 MaxOrd = CalcMaxOrd([K , G , Kch]);
 MaxOrd = ModuleInit([K , G , Kch] , MaxOrd, 1);         \\ in 'ModuleInitType'
 MaxOrd = concat(MaxOrd, [matid(length(MaxOrd.zbase))]); \\ add zhnf
 MaxOrd = concat(MaxOrd, [matid(length(MaxOrd.zbase))]); \\ add Uinv
 return( [ K , G , Kch , MaxOrd ] );             \\ return group ring structure

}

addhelp(GrInit , "GrInit(K,G): Initialize a group ring with number field K in bnfinit and a group G in Ginit. The result is a 4-component vector [ K , G , Kch , MaxOrd ], where K is KG.nfd, G is KG.grp, Kch is KG.Kch and the maximal order is KG.MaxOrd.");





/* ****************************** GrOKG ******************************
 *
 * GrOKG(grpring): Generates the order O_K[G] in the group ring 
 * grpring initialized bz GrInit(K,G).
 *
 * ***************************************************************** */

GrOKG(grpring) = 
{

 local(n, M, k);

 n = length(grpring.grp.eltlist);

 M = matid(n) * Mod(1,grpring.nfd.pol);

 return( [ M, vector(n,k,matid(length(grpring.nfd.zk))) ] );

}

addhelp(GrOKG , "GrOKG(grpring): Generates the order O_K[G] in the group ring grpring initialized bz GrInit(K,G).");






/* ****************************** PhirhoModule ******************************
 * 
 * PhirhoModule(grpirng, M, comp): Computes the ideal which corresponds
 * to the comp-th component of the module M in the Wedderburn-decomposition.
 * 
 * ************************************************************************ */

PhirhoModule(grpring , M, comp) =
{

local(ce,phirho,ki, arho);

   ce = 0;
   for(ki=1,length(M[1]),          
     phirho = Phirho(grpring, comp, M[1][,ki]);
     phirho = rnfeltreltoabs(grpring.Kch[comp].Krho, phirho);
     arho = rnfidealup(grpring.Kch[comp].Krho, M[2][ki]);
     arho = idealgentohnf(grpring.Kch[comp].Krhoabs, arho);
     arho = idealmul(grpring.Kch[comp].Krhoabs, arho, phirho);
     ce  = idealadd(grpring.Kch[comp].Krhoabs , ce , phirho ); 
   );

return(ce);

}

addhelp(PhirhoModule , "PhirhoModule(grpring,M,comp): ");













