\\  -*-gp-script-*-

/* ****************************** ClassField.gp ******************************
 *
 * ClassField.gp: This file contains routines to compute the relative 
 * polynomial for L|K, where K ist the ground field and L is a subfield of
 * K(p)|K, where p is a prime ideal. K(p) is the ray class field module p
 * over K.  
 *
 * The algorithm follows Cohen, Advanced Topics in Computational Number 
 * Theory, Alg. 6.3.27, p341.
 * 
 * The routines are:
 *
 *      - ClassField(rcgp,ideal,H)
 *      - ListR(rcgp,nf,ideal,H)
 *      - PhiStar(om,z) 
 *      - IdealIsPrimitive(nf,ideal)
 *      - IsIdealElt(nf,ideal,lambda,epsilon)
 *      - IdealList(nf,B) 
 *      - RedModSubGrp(x,H)
 *      - RoundToAlgebraic(K,P,{d=1},{acc=10^-5.}) 
 *      - IsInZK(K,beta,{acc=10^-5.})  
 *
 * Auxiliary functions: NOT FOR PERSONAL USE 
 *
 *      - zz_genPolynomial(nf,R,ideal,lambda,{flag=0})
 *      - zz_ComputeIndex(r,x,d) 
 *      - zz_Conjugates(K,c,mtab,f,{ivar=x})
 *      - yy_denom(v)
 *
 * time-stamp: <Mon, Oct 04, 2004 - 10:36>
 *
 * author:     markus endres (mail: info@mendres.org)
 *
 * ************************************************************************* */ 





/* ****************************** ClassField ******************************
 *
 * ClassField(rcgp,ideal,H): Given a ground field K=Q(sqrt{D}),
 * h_K=1, a conductor ideal and a subgroup H in the ray class group 
 * cl_ideal(K) (rcgp) modulo the ideal. This computes the class field L|K
 * which belongs to H. 
 *
 * rcgp must be initalized with bnrinit, than K is equal to rcgp[1].
 *
 * The strategy follows Cohen 2, Advanced Topics in Computational Number
 * Theory, Alg. 6.3.27, p341.
 *
 * ********************************************************************** */
 
ClassField(rcgp , ideal , H , ivar=x) = 
{

local(nf,var,omnf,f,f2,ntu,tu,u,a,b,lambda,N,R,P,v,mtab,cv,algcv);

 nf = rcgp[1]; var = variable(nf.pol);    
 
 omnf = Mod(nf.zk[2] , nf.pol);
 f = idealtwoelt(nf,ideal)[1];     \\ to get the positive generator f of ideal \cap Z
 f2 = f*2;

 \\ step 1: h_nf = 1, OK!
 \\ step 2: not yet implemented

 \\ step 3: choose lambda
 ntu = nf.tu[1];  \\ number of torsion units
 tu = vector(ntu); u = nf.tu[2]; tu[1] = u;

 for(i=2,ntu,
   tu[i] = nfeltmul(nf , u , tu[i-1]);
 );

 tu = nfbasistoalg(nf , tu);

 for(a=0,f2-1,
   for(b=0,f2-1,

     lambda = a*omnf + b;
     N = norm(lambda);

     if( Mod(N , f2) == 1,
      for(i=1,ntu,
        if( IsIdealElt(nf , ideal , lambda , tu[i]) == 1,
         break(2);
        );
     );

     break(2);
     );
   );

 );


 \\ step 4: compute a list R of primitive ideals c with (c,fO_K)=1
 R = ListR(rcgp , nf , ideal , H);

 \\ step 5: initialize P(X)
 v = zz_genPolynomial(nf , R , ideal , lambda);   \\ compute polynomial for L|K

 P = v[1];  cv = v[2];   mtab = R[4];

 if( P == 0, 
   v = zz_genPolynomial(nf , R , ideal , lambda , 1); \\ if P == 0, use norm, flag = 1
   P = v[1];  cv = v[2];
 );


 P = RoundToAlgebraic(nf , P); \\ round coefficients of P to algebraic integers
 
 P = subst(P,x,ivar); 

 if( polisirreducible(P),      \\ check if P is irreducible
   algcv = zz_Conjugates(K,cv,mtab,P,ivar);

   return([P, algcv]);                  \\ return P
  ,
   return(0);         
 );

 
}

addhelp(ClassField , "ClassField(rcgp,ideal,H,{ivar=x}): Given a ground field K=Q(sqrt{D}), h_K=1, a conductor ideal and a subgroup H in the ray class group cl_ideal(K) (rcgp) modulo the ideal. This computes the class field L|K which belongs to H. rcgp must be initalized with bnrinit, than K is equal to rcgp[1].");





/* ****************************** ListR ******************************
 *
 * ListR(rcgp,nf,ideal,H): Computes a list R of primitive ideals c
 * such that (c,idealO_K) = 1 and all ideal c together addict to 
 * cl_ideal(K), where K is the ground field, K = rcgp[1]. H subgroup.
 *
 * ***************************************************************** */

ListR(rcgp,nf,ideal,H) = 
{

 local(n,d,r,f,h,R,B,L,b,l,j,list,c,x1,m,y1,q,k,mtab);

 n = rcgp[5][1];  \\ rcgp.clgp[1] = cardinality of the ray class group 
 d = rcgp[5][2];  \\ rcgp.clgp[2] = vector of cyclic components
 r = length(rcgp[5][2]);

 f = idealtwoelt(nf , ideal)[1];        \\ f is the positive generator of ideal \cap Z
 h = prod(i=1,length(H) , H[i,i]);

 \\ step 1: initialize
 R = vector(h,i,[]); B = 10*n; L = vector(n); b = 0; l = 0; j = 1;

 while(l < n,
  list = ideallist(nf , B);             \\ list of ideals with norm j <= B

  while(l < n && j <= B, 

    if( gcd(f,j) == 1,

      k=1;

      \\ step 4
      while(l < n && k <= length(list[j]),         

    c = idealhnf(nf , list[j][k]);

    if( IdealIsPrimitive(nf , c) ,       \\ check if c is a primitive ideal

      x1 = bnrisprincipal(rcgp , c , 0); \\ flag 0: output only the exponents x_i

      m = 1 + zz_ComputeIndex(r ,x1 ,d); \\ compute 
                                             \\ x[r] + d[r]*(x[r_1] + d[r-1]*(....d[2]*(x[1]))...)

      if(L[m] == 0,

        L[m] = 1;
 
        y1 = RedModSubGrp(x1 , H);       \\ reduce modulo subgroup H   

        q = 1 + zz_ComputeIndex(r , y1 , vector(r,i,H[i,i]));

        R[q] = vecput(R[q] , c);

        l++;

      );    

    );

      k++;

      );  \\ end step 4

    );

  j++;

  );

  B = 2*B;

 );
 
 
 mtab = matrix(h, h); 
 v = vector(r,k,H[k,k]);
 c = InitMultiCounter( vector(r) , vector(r,k,H[k,k]-1));
 while (c,
     d = InitMultiCounter( vector(r) , vector(r,k,H[k,k]-1) );

     while (d,
         s = RedModSubGrp(c[1]~, H); s = 1 + zz_ComputeIndex(r,s, v);
         t = RedModSubGrp(d[1]~, H); t = 1 + zz_ComputeIndex(r,t, v);
         y1 = RedModSubGrp((c[1]+d[1])~, H); y1 = 1 + zz_ComputeIndex(r,y1, v);
         mtab[s,t] = y1;
         d = IncMultiCounter(d);
     );
     c = IncMultiCounter(c);
 );        

 return([ n , h , R, mtab ]);

}

addhelp(ListR , "ListR(rcgp,nf,ideal,H): Compute a list R of primitive ideals c such that (c,idealO_nf)=1 and all ideal c together are cl_ideal(nf) (rcgp). This function is used by ClassField.");





/* ****************************** PhiStar ******************************
 *
 * PhiStar(om,z): Given a complex element z and a lattice 
 * om = [om1,om2] in C, Im(om1/om2) > 0, this computes the function
 * 
 *    Phi*(om,z) = exp(-zz*(om)/2)sigma(om,z)
 *
 * where z = x1om1 + x2om2 and z*(om) = x1eta1 + x2eta2.
 *
 * eta1 and eta2 are the quasi-periods associated to the basis (om1,om2).
 * sigma(om,z) denotes the Weierstrass sigma-function. 
 *
 * ******************************************************************* */


PhiStar(om,z) = 
{

 local(M,X,qp,zstar,s);

 \\ compute the x_i in z = x1om2 + x2om2
 M = matrix(2,2);
 M[1,1] = real(om[1]); M[1,2] = real(om[2]);
 M[2,1] = imag(om[1]); M[2,2] = imag(om[2]);

 X = matsolve(M , [real(z),imag(z)]~);       \\ X = [x1,x2]

 qp = elleta(om);                            \\ compute the quasi-periods [eta1,eta2]

 zstar = X[1]*qp[1] + X[2]*qp[2];            \\ compute z*(L) = x1eta1 + x2eta2

 s = ellsigma(om , z);                       \\ compute the Weierstrass sigma function

 return( exp((-z*zstar)/2.) * s );

}

addhelp(PhiStar , "PhiStar(om,z): Given a complex element z and a lattice om = [om1,om2] in C, Im(om1/om2) > 0, this computes the function Phi*(om,z) = exp(-zz*(om)/2)sigma(om,z), where sigma(om,z) denotes the Weierstrass sigma-function.");





/* ****************************** IsIdealElt ******************************
 *
 * IsIdealElt(nf,ideal,lambda,epsilon): Checks, if the element 
 * lambda - epsilon is an element of the ideal ideal in the number
 * field nf.
 *
 * ********************************************************************** */

IsIdealElt(nf , ideal , lambda , epsilon) =
{

local(f,le);

f = idealtwoelt(nf , ideal)[1];  \\ positive generator
ideal = idealhnf(nf , ideal);    \\ we need the hnf of the ideal 

le = lambda - epsilon;
le = nfalgtobasis(nf , le); 

if( denominator(ideal^(-1) * le) == 1, return(1));

return(0);

}

addhelp(IsIdealElt , "IsIdealElt(nf,ideal,lambda,epsilon): Checks if the element lambda - epsilon is an element of the ideal ideal in the number field nf.");





/* ****************************** IdealList ******************************
 *
 * IdealList(nf,B): Let nf be a number field in nfinit and B a positive 
 * integer. This algorithm outputs a list of lists L such that for each 
 * n <= B, Ln is the list of all integral ideals of absolute norm equal 
 * to n.
 *
 * The algorithm is given by Cohen, Advanced Topics in Computational
 * Number Theory, Algorithm 2.3.23, p100.
 * 
 * This function is equal to ideallist in PARI/GP.
 *
 * ********************************************************************* */

IdealList(nf , B) =
{

 local(list,zk,p,dec,g,f,j,q,n,plist);

 \\ step 1: initialize
 list = listcreate(B);

 for(i=1,B,
   listput(list , listcreate(B));        \\ the list contains lists
 );

 zk = idealhnf(nf , nf.zk);

 listput(list[1] , zk);                  \\ the first entry is Z_K

 p = 2;

 while(p <= B,

  dec = idealprimedec(nf , p);           \\ factor pZ_K as pZ_K = \prod_{i=1}^g p_i^{e_i}

  g = length(dec);

  f = vector(g , i , dec[i][4]);         \\ f_i = f(p_i/p)

  j = 1;

   \\ next prime ideal
   while(j <= g,

    q = p^f[j]; n = q;

     \\ step 5: loop through all multiples of q
     while(n <= B, 

       for(i=1,length(list[n/q]),
       plist = idealmul(nf , dec[j] , list[n/q][i]);
       listput(list[n] , plist);
       );

       n += q;
     );

    j +=  1;
   );

   p = nextprime(p+1);    \\ next prime
 );

 return(list);

}

addhelp(IdealList , "IdealList(nf,B): Let nf be a number field in nfinit and B a positive integer. This algorithm outputs a list of lists L such that for each n <= B, Ln is the list of all integral ideals of absolute norm equal to n. This function is equal to ideallist in PARI/GP. For a better performance use ideallist.")





/* ****************************** ReduceModSubGrp ******************************
 *
 * ReduceModSubGrp(x,H): This function reduces x modulo a subgroup H.
 *
 * *************************************************************************** */

RedModSubGrp(x , H) =
{

 local(r,n,v);

 r = length(H); n = r;

 for(i=1,n,

   v = divrem(x[n] , H[n,n])[1];   

   x = x - v*H[,n]; 

   n -= 1;

 ); 

 return(x);

}


addhelp(RedModSubGrp , "RedModSubGrp(x,H): Reduce x modulo a subgroup H. Not for personal use.");





/* ****************************** IsInZK ******************************
 *
 * IsInZK(K, beta, {acc=10^-.5}): Following the strategy of Cohen 2,
 * Alg. 6.3.29, p343, checks if beta is element of Z_K. K is an 
 * imaginary quadratic field and beta a complex numerical approximation
 * beta = r + i*im. 
 *
 * ****************************************************************** */

IsInZK(K , beta , acc=10^-5.) = 
{

 local(D,r,im,a0,b0,a,b);

 D = K.disc;

 r = real(beta); im = imag(beta);

 \\ step 1: compute a and b
 a0 = 2*r; a = round(a0); 
 b0 = 2*im / sqrt( abs(D) ); b = round(b0);

 \\ step 2: check and terminate
 if( abs(a - a0) > acc || abs(b - b0) > acc || Mod(a-b*D,2) == 1, return(0));

 return([a,b]);

}

addhelp(IsInZK , "IsInZK(K,beta,{acc=10^-5.}): Checks if beta is in Z_K for an imaginary quadratic field K. beta is a complex numerical approximation beta = r + i*im.");





/* ****************************** RoundToAlgebraic ******************************
 *
 * RoundToAlgebraic(K,P): Round the coefficients of the polynomial P to 
 * algebraic integers in K.
 *  
 * The strategy follows Cohen 2 in Alg. 6.3.27, p341.
 *
 * **************************************************************************** */

RoundToAlgebraic(K , P , d=1 , acc=10^-5.) =
{

 local(var,deg,P2,coeff,ab);

 var = variable(K.pol);

 deg = poldegree(P);

 P2 = 0;
 for(i=0,deg,

    coeff = d * polcoeff(P,i);
    ab = IsInZK(K , coeff , acc);

    if( type(ab) != "t_INT",
      P2 = P2 + ( Mod( (ab[1]+ab[2]*var)/(2*d) , K.pol) ) * X^i;
     , \\ else
       error(" increase accuracy ");
    );

 );

 P2 = subst(P2 , X , x);

 return(P2);

}

addhelp(RoundToAlgebraic , "RoundToAlgebraic(K,P,{d=1},{acc=10^-5.}): Round the coefficients of the polynomial P to algebraic integers in K.");










/* ****************************** Auxiliary functions ******************************
 * 
 *                              - NOT FOR PERSONAL USE -
 * 
 * ******************************************************************************* */

/* ****************************** zz_genPolynomial ******************************
 *
 * zz_genPolynomial(K,R,ideal,lambda,{flag=0}): Not for personal use! This 
 * function is used by ClassField to compute a generating polynomial for a 
 * field extension L|K.
 *
 * This follows the strategy of Cohen 2, Alg. 6.3.27, p341.
 *
 * **************************************************************************** */

\\ not for personal use
zz_genPolynomial(K , R , ideal , lambda , flag=0) =
{

local(var,omK,P,n,h,t,c,fc,om,s,ConjVec);

 var = variable(K.pol);
 omK = Mod(K.zk[2] , K.pol);

 P = 1; 
 n = R[1]; h = R[2]; R = R[3];
 ConjVec = vector(h);

 if(!flag,   \\ use trace

  for(i=1,h,
    t = 0;
    for(j=1, n/h,                             \\ step 7:
      c = idealinv(K,R[i][j]);
      fc = idealmul(K,ideal,c); 
      fc = idealhnf(K,fc);
      om = [ fc[1,2] + fc[2,2]*subst(lift(omK) , var , sqrt(K.disc)) , fc[1,1] ];

      lambda = subst(lift(lambda) ,var , sqrt(K.disc));
      s = PhiStar(om , lambda) / PhiStar(om , 1); \\ step 8:

       t += s;
    );

   P = (X - t) * P;                           \\ step 9:
   ConjVec[i] = t;
  );

 , \\ else there is a flag, use the norm

  for(i=1,h,
    t = 1;
    for(j=1, n/h,                             \\ step 7:
      c = idealinv(K,R[i][j]);
      fc = idealmul(K,ideal,c); 
      fc = idealhnf(K,fc);
      om = [ fc[1,2] + fc[2,2]*subst(lift(omK) , var , sqrt(K.disc)) , fc[1,1] ];

      lambda = subst(lift(lambda) ,var , sqrt(K.disc));
      s = PhiStar(om , lambda) / PhiStar(om , 1); \\ step 8:

      t *= s;
    );

   P = (X - t) * P;                           \\ step 9:
   ConjVec[i] = t;
  );

 );

 return([P, ConjVec]);

}

addhelp(zz_genPolynomial , "zz_genPolynomial(K,R,ideal,lambda,{flag=0}): Not for personal use! This function is used by ClassField to compute a generating polynomial for a field extension L|K.");





/* ****************************** zz_ComputeIndex ******************************
 *
 * zz_ComputeIndex(r,x,d): Not for personal use! This function is used by ListR
 * to compute 
 *
 * x[r] + d[r]*(x[r_1] + d[r-1]*(....d[2]*(x[1]))...)
 *
 * *************************************************************************** */

\\ not for personal use
zz_ComputeIndex(r , x , d) =
{

 local(m);

 if(r == 1, return(x[1]));

 m = x[r] + d[r] * zz_ComputeIndex( r-1 , veckill(x,r) , veckill(d,r) );
 
 return(m);

}

addhelp(zz_ComputeIndex , "zz_ComputeIndex(r,x,d): Not for personal use! This function is used by ListR to compute x[r] + d[r]*(x[r_1] + d[r-1]*(....d[2]*(x[1]))...).");



\\ not for personal use
zz_Conjugates(K, c, mtab, f, ivar=x) =
{
 local(var,n,A,i,b,res,P,algc,OL,v,j,d);
 

 var = variable(K.pol);
 n = length(c);       \\ n = number of conjugates
 A = matrix(n,n,i,j,c[i]^(j-1));
 A = A^(-1);

 OL = rnfpseudobasis(K, f);
 OL = PseudoBasisToBasis(K, OL, ivar);
 v = [];
 for (i=1, length(OL),
    for (j=0, poldegree(OL[i]),
        v = concat(v, [nfalgtobasis(K, polcoeff(OL[i], j))]);
    );
 );
 d = yy_denom(v);
 
 algc = vector(n);
 for (i=1,n,
     b = vectorv(n,j,c[mtab[i,j]]);
     res = A*b;
     P=0; for (i=0,n-1,P=P+res[i+1]*var^i);
     algc[i] = RoundToAlgebraic(K, P, d, 10^-5.);
 );
 return(algc);
}
addhelp(zz_Conjugates,"zz_Conjugates(K,c,mtab,f,{ivar=x}): Not for personal use! Computes the conjugates.");




\\ not for personal use
yy_denom(v) =
{
    local(d,i,j);

    d = 1;
    for (i=1, length(v),
        for (j = 1, length(v[i]),
            d = lcm(d, denominator(v[i][j]));
        );
    );
    return(d);
}





