\\  -*-gp-script-*-

/* ****************************** Add.gp ******************************
 * 
 * Add.gp: Contains some additional functions we need for various
 * computations in GrInit.gp, ClassField.gp, AAO.gp, etc.
 *
 * Routines:
 *
 *      - StructType(t)
 *      - PseudoBasisToBasis(nf,pseudobasis)
 *      - vecfind(v,elt)
 *      - gcdvec(v)
 *      - lcmvec(v)
 *      - veckill(v,i)
 *      - veckillall(v,elt)
 *      - vecput(v,elt)
 *
 *      - InitMultiCounter(low,hihg,r,i)
 *      - IncMultiCounter(c,r,i,carry)
 *
 * 
 * time-stamp: <Mon, Oct 04, 2004 - 10:30>
 *
 * author:     markus endres (mail: parigp@mendres.org)
 *             werner bley (mail: bley@math.uni-augsburg.de)
 *
 * created: <Tue, Nov 02, 2004>
 *
 * ****************************************************************** */




/* ****************************** StructType ******************************
 *
 * StructType(t): Let t be a matrix or a vector with different entries. 
 * This returns a vector with the types of the entries of t. 
 * 
 * ********************************************************************** */

StructType(t) =
{

 local(n,v);

 if(type(t) != "t_VEC", return(type(t)) );

 n = length(t); v = vector(n);

 for(i=1,n,
   v[i] = type(t[i]);
 );

 return(v);

}

addhelp(StructType , "StructType(t): Returns the type of a structure.");





/* ****************************** PseudoBasisToBasis ******************************
 *
 * PseudoBasisToBasis(nf,pseudobasis): Computes the real basis of a given pseudo
 * basis from rnfinit(nf,pol). pseudobasis is a two-component row vector [A,I], 
 * where the first component is the relative integral pseudo basis expressed as 
 * polynomials with polmod coefficients in the number field nf and the second 
 * component is the ideal list of the pseudobasis in HNF. nf must have class number 1.
 *
 * ****************************************************************************** */

PseudoBasisToBasis(nf , pseudobasis , ivar=x) =
{

 local(n,v,q,alg);

 n = length(pseudobasis[1]);

 v = vector(n);

 if(type(pseudobasis[1]) == "t_MAT",          \\ first component is a matrix with
					      \\ algebraic numbers or polmods as entries

   if(type(pseudobasis[1][1,1]) != "t_POLMOD",\\ if we have algebraic numbers, 
					      \\ so we change it to polmods
     pseudobasis[1] = matbasistoalg(nf , pseudobasis[1]);  
   );

   \\ make a vector 
   for(i=1,n,
     for(j=1,n,
       v[i] += pseudobasis[1][j,i] * ivar^(j-1); \\ and than the real basis
     );
   );

   pseudobasis[1] = v;

 );

 for(i=1,n,
   q = bnfisprincipal(nf , pseudobasis[2][i])[2];
   alg = nfbasistoalg(nf , q);
   pseudobasis[1][i] = alg * pseudobasis[1][i];
 );

 return(pseudobasis[1]);

}

addhelp(PseudoBasisToBasis , "PseudoBasisToBasis(nf,pseudobasis,{ivar=x}): Computes the real basis of a given pseudo basis from rnfinit(nf,pol). pseuodbasis is a two-component row vector [A,I], where the first component is the relative integral pseudo basis expressed as polynomials with polmod coefficients in the number field nf and the second component is the ideal list of the pseudobasis in HNF.");





/* ****************************** vecfind ******************************
 *
 * vecfind(v,elt): Finds the position of the element elt in the vector v.
 *
 * ******************************************************************* */

vecfind(v , elt) =
{ 

 local(n); 

 n = length(v);

 for(k=1,n,
   if( v[k] == elt, return(k));
 );

 []  \\ not found

}

addhelp(vecfind,"vecfind(v,elt) Finds the position of the element elt in the vector v.");





/* ****************************** gcdvec ******************************
 *
 * gcdvec(v): Computes the gcd of all the elements in the vector v.
 *
 * ****************************************************************** */

gcdvec(v) = 
{

 local(n,g);

 n = length(v);

 if(n < 2, return(v[1]));

 g = gcd(v[1],v[2]);

 for(i=3,n, g = gcd(v[i],g) );

 return(g);

}

addhelp(gcdvec , "gcdvec(v): Computes the gcd of all the elements in the vector v.");





/* ****************************** lcmvec ******************************
 *
 * lcmvec(v): Computes the lcm of all the elements in the vector v.
 *
 * ****************************************************************** */

lcmvec(v) =
{

 local(n,l);

 n = length(v);

 if(n < 2, return(v[1]));

 l = lcm(v[1],v[2]);

 for(i=3,n, l = lcm(v[i],l));

 return(l);

}

addhelp(lcmvec , "lcmvec(v): Computes the lcm of all the elements in the vector v.");





/* ****************************** veckill ******************************
 *
 * veckill(v,i): Given a vector v and an index i, this removes the
 * i-th coordinate of v.
 *
 * ******************************************************************* */

veckill(v , i) =
{

 local(k);

 vector(length(v)-1 , k , if(k<i,v[k],v[k+1]))

}

addhelp(veckill,"veckill(v,i:) Given a vector v, and an index i, removes the i-th coordinate.");





/* ****************************** veckillall ******************************
 *
 * veckillall(v,elt): Given a vector v and an element elt in v. This 
 * deletes all elements elt from v.
 * 
 * ********************************************************************** */

veckillall(v , elt) = 
{

 local(aux);

 aux = [];

 for(i=1,length(v),
   if(v[i] != elt, aux = vecput(aux,v[i]));
 );

 return(aux);

}

addhelp(veckillall , "veckillall(v,elt): Given a vector v and an element elt in v. This deletes all elements elt from v.");





/* ****************************** vecput ******************************
 *
 * vecput(v,elt): This generates a 'dynamical vector' v. This means,
 * initalize v with v=[], and than put the elements elt into v with
 * v = vecput(v,elt). 
 *
 * ****************************************************************** */

vecput(v , elt) = 
{

 local(n,aux);

 n = length(v);

 if(n == 0, return([elt]));

 aux = vector(n+1);
 for(i=1,n,
   aux[i] = v[i];
 );

 aux[n+1] = elt;

 return(aux);

}

addhelp(vecput , "vecput(v,elt): Puts the element elt into the vector v.");





\\ not for personal use 
{InitMultiCounter(low, high, r, i) =
 if (length(low) != length(high),
  error("InitMultiCounter: length(low) != length(high)\n")
 );

 r = length(low);
 for (i = 1, r,
   if (low[i] > high[i],
   error("InitMultiCounter: low > high\n")
   )
 );

 return([low, low, high]);
}





\\ not for personal use 
{IncMultiCounter(c, r, i, carry) =
  r = length(c[1]);
  carry = true;
  i = 1;

  while (carry && i <= r,
    if(c[1][i] < c[3][i],
      c[1][i]++; carry = false
       ,
      c[1][i] = c[2][i]; i++
    )
 );

 if (carry, return(false), return(c));
}




\\ necessary for the unstable CVS PARI-GP version
idealgentohnf(nf, ideal) = 
{

 local(z);

 z = nfalgtobasis(nf, ideal);
 z[1] = Mat(z[1]);
 mathnf(concat(z));

}
