/********************************************************************************************
 * This file contains routines which are needed to compute the Mazur-Tate pairing as described in
 * the paper
 *       W. Bley, D. Macias Castillo : Congruences for critical values of higher derivatives of twisted Hasse-Weil L-functions, III 
 * It is built on many routines written by Chris Geishauser for his Master Thesis
 *       C. Geishauser, Computation of 2-extensions of dual Selmer groups, LMU München, 2018
 * In particular, the computation of the generalized Selmer group is due to him. The algorithm is based
 * on the work of Schaefer and Stoll, How to do a p-descent on an elliptic curve, Transactions of the AMS, 356 (3).
 *
 *
 * Most important intrinsics are:
 *     ComputePhiMatrix, MazurTatePairing and MyThreeSelmerGroup
 *
 **********************************************************************************************/     


GeneralSelmerGroup := recformat<Sel : GrpAb,
                                SelmerToField : Map,
			        SelmerInvers : Map,
			        SelmerRep : HomGrp,
			        SigmaL1 : SetEnum,
			        xmap : Map
                               >;


/******************************************ConHom****************************************************
 * 
 * This function computes the connecting homomorphism F: E(K)/3E(K) -> A^star/(A^star)^3, where K is
 * a number field or the completion of a number field and A is the global or local étale Algebra to a 
 * Galois invariant set X \subset E[3], spanning E[3].
 * The étale algebra decomposes into a product of number fields, where each number field is associated
 * to a representative P of an orbit of X. R \in E(K) equal P \in X is equivalent to R being a root of a 
 * irreducible factor of f (f is the polynomial defining A).
 * 
 ****************************************************************************************************/
function ConHom(E,R,f)


b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
K:=Parent(R[1]);
Ky<y>:=PolynomialRing(K);
D:=-4*a^3-27*b^2;

x:=(1/(4*K!a*K!D))*(-6*K!b*Ky!D+3*Ky!D*y*y-72*Ky!b*y^4-9*y^6);
fD:=(x^3-Ky!a*x-2*Ky!b)^2-4*Ky!b*y^2;

T:=Factorization(f);
n:=#T;
List:=[];


if 3*R ne E!0 then
    //fR:=(3*x^2+a)*R[1]-2*y*R[2]-x^3+a*x+2*b;
    fR:=2*y*R[2]-(3*x^2+Ky!a)*R[1]+x^3-Ky!a*x-2*Ky!b;
    for i:=1 to n do
        Append(~List, [fR*fD,T[i][1]]);
    end for;
return List;

elif R eq E!0 then
    for i:=1 to n do
        Append(~List, [Ky!1,T[i][1]]);
    end for;
return List;

else 
    for i:=1 to n do
        
        if y-R[2] eq T[i][1] then
            //fnegP:= (3*x^2+a)*R[1]-2*y*(-(R[2]))-x^3+a*x+2*b;
            fnegP:=2*y*(-R[2])-(3*x^2+Ky!a)*R[1]+x^3-Ky!a*x-2*Ky!b;
            Append(~List, [fnegP^2*fD^2,T[i][1]]);
        else 
            //fR:=(3*x^2+a)*R[1]-2*y*R[2]-x^3+a*x+2*b;
            fR:=2*y*R[2]-(3*x^2+Ky!a)*R[1]+x^3-Ky!a*x-2*Ky!b;
            Append(~List, [fR*fD,T[i][1]]);
        end if;
    end for;
return List;
end if;
end function;

/**************************************HilbertSymbol******************************************
 * 
 * K denotes a local field such that the residue characteristic l of K does not divide n.
 * Moreover the n-th roots of unity need to live in K.
 * 
 * *****************************************************************************************/

intrinsic HilbertSymbol(K:: FldPad ,n:: RngIntElt ,a:: FldPadElt, b:: FldPadElt) -> FldPadElt
{}

    Qx<x>:=PolynomialRing(Rationals());
    O:=Integers(K);
    F,m:=ResidueClassField(O);
    q:=#F;
    if IsDivisibleBy(n, Characteristic(F)) then
        print "residue characteristic divides n!";
    end if;
    if #Roots(x^n-1, K) ne n then
        print "K does not contain n-th roots of unity!";
    end if;
    if Parent(a) ne K or Parent(b) ne K then
        print "a or b is not an element of K!";
    end if;
    va:=Valuation(a);
    vb:=Valuation(b);
    y:= (-1)^(va*vb)*(b^va div a^vb);

    /* Replaced using the intrinsic HenselLift in Sept 2021 since this take much too long for q large. */
/* 
    z:=0;
    for T in Roots(x^(q-1)-1,K) do
        if m(y) eq m(T[1]) then
            z:= T[1];
            break T;
        end if;
    end for;
*/

    KX<x> := PolynomialRing(K);
    z := HenselLift(x^(q-1)-1, K!m(y), Precision(y));
    return z^(Integers()!((q-1)/n));
end intrinsic;


/*************************************ConstrAlgebra****************************************
 * 
 * This function computes the etale algebra A for an elliptic curve E and the G-set X=E[3]\O.
 * We ALWAYS assume a neq 0, so that the defining polynomial is separable.
 * A=K[X]/(f) and we also return the map K[X]->K[X]/(f)
 * 
 * *****************************************************************************************/

intrinsic ConstrAlgebra(E:: CrvEll) -> RngUPolRes, Map
{}

b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
D:=-4*a^3-27*b^2;
K:=Parent(b);
Ky<y>:=PolynomialRing(K);

f:=y^8+8*b*y^6-(2/3)*D*y^4-(1/27)*D^2;
A,q:=quo< Ky | f >;
return A,q;
end intrinsic;


//Computes the etale algebra B of E/K as explained in Section 7.3

intrinsic ConstrAlgebraB(E:: CrvEll) -> RngUPolRes, Map
{}

b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
D:=-4*a^3-27*b^2;
K:=Parent(b);
Ky<y>:=PolynomialRing(K);

f:=y^8+2*a*y^4-4*b*y^2-(1/3)*a^2;
B,q:=quo< Ky | f >;
return B,q;
end intrinsic;


/****************************************3TorsBase**********************************************
 * 
 * This intrinsic will be used for an admissible place v and a completion K_v.
 * Assume that E[3] \subset E(K), where E is defined over K. 
 * This function computes a basis of E[3]. We again assume that a neq 0.
 * 
 * *********************************************************************************************/

intrinsic ThreeTorsBase(E :: CrvEll) -> SeqEnum
{}

b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
D:=-4*a^3-27*b^2;
K:=Parent(b);
Ky<y>:=PolynomialRing(K);

f:=y^8+8*b*y^6-(2/3)*D*y^4-(1/27)*D^2;

//Find first basis element S:
ys:=Roots(f,K)[1][1];
xs:=(1/(4*a*D))*(-6*b*D+3*D*ys^2-72*b*ys^4-9*ys^6);
S:=E![xs,ys];

// Find second basis element T (enough if y_T not eq +-ys)
yt:=0;

for i:=2 to #Roots(f,K) do
    if Roots(f,K)[i][1] ne -ys and Roots(f,K)[i][1] ne ys then
        yt:=Roots(f,K)[i][1];
        break i;
    end if;
end for;

xt:=(1/(a*4*D))*(-6*b*D+3*D*yt^2-72*b*yt^4-9*yt^6);
T:=E![xt,yt];


return [S,T];
end intrinsic;


/*************************************** LocalSymbol *************************************************
 * 
 * E is an elliptic curve defined over the number field K. v is a place of K.
 * A is the global étale algebra of the set X=E[3]\{O}, h \in K[X] should represent an element in A.
 * A is given in the form K[X]/(f).
 * We assume that v has the following properties: E(K_v) contains a basis of E[3], v does not 
 * divide 3. The assumption implies \mu_3 \subset K_v. Q is a point on E(K_v).
 * Computes the map Sel^3(E/K,Sigma)\subset A/A^3 ----> (E(K_v)/3E(K_v))^v
 *                                              h |---->  [ Q|-> (res_v(h), Fv(Q))_{A,v} ]
 * where F_v is the local connecting homomorphism and (.,.)_{A,v} is the local Tate pairing
 * 
 * **********************************************************************************************/
intrinsic LocalSymbol (E :: CrvEll, K :: FldNum, w :: PlcNumElt, alpha_P :: FldNumElt, alpha_Q :: FldNumElt, SelGrp::Rec) -> FldFin
{}
    local Kx, L1, L, SelmerToField, tilde_xi, pol_xi, j, eta;

    Kx<z>:=PolynomialRing(K);

    L := Codomain(SelGrp`SelmerToField);

    pol_alpha_Q := &+[alpha_Q[j]*z^(j-1) : j in [1..Degree(L)]];
    pol_alpha_P := &+[alpha_P[j]*z^(j-1) : j in [1..Degree(L)]];

    Kw,iota := Completion(K,w: Precision:=100);
    Ew:=EllipticCurve([iota(Eltseq(E)[4]),iota(Eltseq(E)[5])]);
    B:=ThreeTorsBase(Ew);
    S:=B[1];T:=B[2];

    KwX<y>:=PolynomialRing(Kw);
    pol_alpha_P_w := &+[iota(Coefficient(pol_alpha_P, j))*y^j : j in [0..Degree(pol_alpha_P)]];
    pol_alpha_Q_w := &+[iota(Coefficient(pol_alpha_Q, j))*y^j : j in [0..Degree(pol_alpha_Q)]];

    tP:=[Evaluate(pol_alpha_P_w, S[2]), Evaluate(pol_alpha_P_w, T[2])];
    tQ:=[Evaluate(pol_alpha_Q_w, S[2]), Evaluate(pol_alpha_Q_w, T[2])];

    h1:=HilbertSymbol(Kw,3, tP[1],tQ[2]);
    h2:=HilbertSymbol(Kw,3, tP[2],tQ[1]);
    hil:=h1*h2^(-1);
    // print "Hilbert symbol hil = ", hil, " Test: hil^3 = ", hil^3;
    e:=ThreeWeilPair(Ew,T,S, Kw);
    // k := [j : j in [0..2] | e^j eq hil][1];
    k := [j : j in [0..2] | 0 eq Integers()!(e^j - hil) mod Prime(Kw)][1];

    return GF(3)!k;
end intrinsic;

/****************************** MazurTatePairing ***********************************
 * P and Q are points on an elliptic curve E over Q
 * EK is the base change of E
 * w is a finite place of K
 * SelGrp is the generalized Selmer group computed by MyThreeSelmerGroup
 *
 * Computes the Mazur-Tate pairing <P,Q> as described in the last sections of etncV.
 * Note that the Mazur-Tate pairing is evaluated by computing the Bertolini-Darmon pairing.
 * Note however, that we actually compute -1 times the Bertolini-Darmon pairing.
 *
*******************************************************************************************/
intrinsic MazurTatePairing(P::PtEll, Q::PtEll, EK :: CrvEll, ws :: SeqEnum, SelGrp::Rec) -> FldFin
{}
    local Kx, L1, f, L, tilde_xi, pol_xi, j, eta;

    print "Compute MazurTatePairing for P ", P, " and Q = ", Q;
    Gal := Domain(SelGrp`SelmerRep);
    E := Curve(P);
    K := CoefficientField(EK);
    L := Codomain(SelGrp`SelmerToField);
    f := DefiningPolynomial(L);

    delta:=ConHom(EK,EK!Q,f);      //delta[1][1] is an element in K[X]
    deltapol:=delta[1][1];
    alpha_Q:=Evaluate(deltapol, L.1);

    xi_P := KummerMap(EK!(P), f, SelGrp);
    eta_P := NormEquation(xi_P, SelGrp`SelmerRep);
    res := [];
    for i:= 0 to 2 do
         print "i = ", i;
         b := SelGrp`SelmerToField( GaloisAction(eta_P, Gal.1^i, SelGrp`SelmerRep) );
         // Append(~res, LocalSymbol(E, K, w, b, alpha_Q, SelGrp));
	 Append(~res, &+[LocalSymbol(E, K, w, b, alpha_Q, SelGrp): w in ws]);
    end for;

    FG := GroupAlgebra(GF(3), Gal);
    I := ideal<FG | [FG!Gal.1 - 1]>;
    xi := FG ! res;
    lambda := FG!Gal.1 - 1;
    for t:=0 to 2 do
        if xi - t*lambda in I^2 then
	    return t;
	end if;
    end for;
    printf "Could not find a mu -- something is completely wrong, so I quit";
    assert false;
    return 0;
end intrinsic;


/******************************PolAction*******************************************
 * 
 * pol is a polynomial in K[X] and sigma is supposed to be an automorphism of K.
 * 
 * ******************************************************************************/

intrinsic PolAction (pol :: RngUPolElt, sigma :: Map) -> RngUPolElt
{}
K:=CoefficientRing(Parent(pol));
Kx<x>:=PolynomialRing(K);
seq:=[];
for i:=0 to Degree(pol) do
    Append(~seq,sigma(Coefficient(pol,i)));
end for;

h:=0;
for i:=1 to Degree(pol)+1 do
    h:=h+seq[i]*x^(i-1);
end for;
return h;
end intrinsic;


/*******************************AlgebraAction************************************
 * 
 * a is a tupel in A=prod_i K[X]/(f_i), where f_i are the irreducible factors
 * of the defining polynomial f.
 * sigma is supposed to be an automorphism of K.
 * q is a tupel of quotient maps, consisting of maps q_i: K[x]-> K[X]/(f_i)
 * computes the action of sigma on A.
 *
 * Not used anywhere! (24.9.19 by Werner Bley)
 * 
 * *****************************************************************************/

intrinsic AlgebraAction (a ::SeqEnum, q :: SeqEnum, sigma :: Map)-> SeqEnum
{}

seq1:=[];
seq2:=[];
seq3:=[];
// lift the entries to polynomials in K[X]
for i:=1 to #a do
    bool, s:=HasPreimage(a[i],q[i]);
    if not bool then
        print "something is wrong in AlgebraAction with the element a or map q";
    end if;
    Append(~seq1, s);
end for;    


// Searches for j such that sigma(f_j) = f_i
for i:=1 to #a do
    for j:=1 to #a do
        if PolAction(Modulus(Parent(a[j])),sigma) eq Modulus(Parent(a[i])) then
            Append(~seq2, PolAction(seq1[j],sigma));
            break j;
        end if;
    end for;
end for;

//Map the sequence of polynomials back to an element in A.

for i:=1 to #a do
    Append(~seq3, q[i](seq2[i]));
end for;

return seq3;

end intrinsic;



/************************************ThreeWeilPair******************************************
 * 
 * IF S,T are two 3-torsion points on E, uses [Silverman, XI.8.2] to compute the
 * Weil-Pairing e_3(S,T).
 * This is a preliminary version, where we assume that the coordinates of S,T lie in the 
 * base field of E. K is a local field. This will be just used for completions K_v where
 * v is an admissible place.
 * 
 * *****************************************************************************************/

intrinsic ThreeWeilPair(E :: CrvEll,S :: PtEll,T :: PtEll, K :: FldPad) -> FldPadElt
{}

if 3*S ne E!0 or 3*T ne E!0 then
    print "S or T is not a 3-torsion point!";
end if;

O:=Integers(K);
Kxy<x,y>:=PolynomialRing(K,2);
b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
xs:=S[1];
ys:=S[2];
xt:=T[1];
yt:=T[2];

fS:=(3*xs^2+a)*x-2*ys*y-xs^3+a*xs+2*b;
fT:=(3*xt^2+a)*x-2*yt*y-xt^3+a*xt+2*b;
pi:=UniformizingElement(O);

nice:=false;

//Searches for a random point on E which is not a 3-torsion point
//Increases the precision with every step

while nice eq false do
    x1 := pi^(2*Random(2))*Random(O);
    if IsSquare(x1^3+a*x1+b) eq true then
        nice,y1:=IsSquare(x1^3+a*x1+b);
        Q:=E![x1,y1];
        if 3*Q eq E!0 then
        nice:=false;
        end if;
    end if;
    K`DefaultPrecision:=K`DefaultPrecision+1;
end while;


R1:=S+Q;
R2:=T-Q;
negQ:=-Q;

e:=Evaluate(fT,[R1[1],R1[2]])*Evaluate(fS,[negQ[1],negQ[2]])*
Evaluate(fT,[Q[1],Q[2]])^(-1)*Evaluate(fS,[R2[1],R2[2]])^(-1);

return e;
end intrinsic;    

/***************************************newBDLocalBasis********************************************
 *
 * Given an elliptic curve E, a numberfield K and a place v of K, this function computes
 * a basis of E(Kv)/pE(Kv).
 * Note that there are some assumptions on the place v. This intrinsic is based on the special
 * situation we are working with. For example we have
 * E(Kv)/pE(Kv) \simeq E(Fv)/pE(Fv), where Fv is the residue class field of Kv
 * P denotes a free generator of E(K), which is not trivial in E(Kv)/pE(Kv).
 * See also Remark 8.2
 *
 * *********************************************************************************************/
intrinsic newBDLocalBasis (E :: CrvEll, K :: FldNum, automap :: Map, v :: RngIntElt, P :: PtEll, p :: RngIntElt) -> SeqEnum
{}

    w0 := Decomposition(K, v)[1][1];
    Kw0, i:=Completion(K,w0);
    O:=Integers(Kw0);
    b:=Eltseq(E)[5];
    a:=Eltseq(E)[4];
    aw0:=i(a);
    bw0:=i(b);
    Ew0:=EllipticCurve([aw0,bw0]);
    // Map P \in E(K) into E(Kv)
    Pw0:=Ew0![i(P[1]),i(P[2]),i(P[3])];

    F,j:=ResidueClassField(O);
    Ef:=ChangeRing(Ew0,F);

    Pf:=Ef!Pw0;

    // y:=0;
    pi:=UniformizingElement(O);
    cnt := 0;
    good:=false;
    while not good do
        // cnt := cnt+1; print "cnt = ", cnt;
        // Find a random point Q in E(Kv)
        // cnt := cnt+1; print "cnt = ", cnt;
        nice := false;
        while not nice do
            x := pi^(2*Random(2))*Random(O);
            if IsSquare(x^3+aw0*x+bw0) eq true then
                nice,y:=IsSquare(x^3+aw0*x+bw0);
                Q:=Ew0![x,y];
            end if;
            Kw0`DefaultPrecision:=Kw0`DefaultPrecision+1;
        end while;

        // Test wether Q complements P to a basis of E(Kv)/pE(Kv)    
        // In order to do this, we work in the group E(Fv)/pE(Fv) \simeq E(Kv)/pE(Kv)    

        Qf:=Ef!Q; 

        G,phi:=AbelianGroup(Ef);

        Pfinv := [g : g in G | phi(g) eq Pf][1];
        Qfinv := [g : g in G | phi(g) eq Qf][1];
        l := 3;
        Gmodp, pr := quo<G | [l*G.1, l*G.2]>;
        S := sub<Gmodp | [pr(Pfinv), pr(Qfinv)]>;
        if S eq Gmodp then
            good := true;
        end if;
    end while;

    res := [];
    OK := Order(Ideal(w0));
    for sigma in Domain(automap) do
        w := Place(ideal< OK | [automap(sigma)(z) : z in Generators(Ideal(w0))]>);
        Kw, i:=Completion(K,w);
        O:=Integers(Kw);
        b:=Eltseq(E)[5];
        a:=Eltseq(E)[4];
        aw:=i(a);
        bw:=i(b);
        Ew:=EllipticCurve([aw,bw]);
        // Map P \in E(K) into E(Kv)
        Pw:=Ew![i(P[1]),i(P[2]),i(P[3])];
        Qw := Ew ! Q;   /* ERROR ???????????????????????? */
        Append(~res, [* w, Pw, Qw, Ew *]);
    end for;


    return res;
end intrinsic;

intrinsic GaloisAction(sigma :: Map, m :: List, SigmaKIndexSet :: SeqEnum, SigmaKEw :: SeqEnum) -> List
{}
    local OK, sigmam, w, j, i;

    OK := Order(Ideal(SigmaKIndexSet[1]));
    sigmam := m;
    for w in SigmaKIndexSet do
        sigmaw := Place(ideal< OK | [sigma(z) : z in Generators(Ideal(w))]>);
	j := Index(SigmaKIndexSet, sigmaw);
	i := Index(SigmaKIndexSet, w);
	sigmam[i] := SigmaKEw[i] ! m[j];
        // Append(~sigmam, SigmaKEw[i] ! m[j]);
    end for;
    return sigmam;
end intrinsic;

intrinsic '+'(a :: List, b :: List) -> List
{}
    return [* a[i]+b[i] : i in [1..#b] *];
end intrinsic;


/*****************************************ConstructS****************************************
 * 
 * For an elliptic curve E/Q, computes the subset S of M_K as defined in Proposition 3.12. 
 * S consists of the places v above p (for our prime p) and the places v such that the
 * Tamagawa number is divisible by p.
 * 
 * *****************************************************************************************/
intrinsic ConstructS (E :: CrvEll, K :: FldNum, p :: RngIntElt)-> SeqEnum
{}

list:=[];
E:=BaseChange(E,K);

for I in Decomposition(K,p) do
    Append(~list, Ideal(I[1]));
end for;

LocalInfo:=LocalInformation(E);
for T in LocalInfo do
    c:=T[4];
    if IsDivisibleBy(c,p) and not T[1] in list then
        Append(~list, T[1]);
    end if;
end for;
return list;
end intrinsic;


//same intrinsic as above, but for the rationals Q instead of a number field

intrinsic ConstructS (E :: CrvEll, p :: RngIntElt)-> SeqEnum
{}

list:=[p];
Q:=Rationals();

for v in BadPrimes(E) do
    if IsDivisibleBy(TamagawaNumber(E,v),p) and (v ne p) then
        Append(~list, v);
    end if;
end for;
return list;
end intrinsic;

//This is the map \overline{\theta} in the thesis, see also Section 7.4

intrinsic sigmagminusg(K :: FldNum, a :: FldNumElt) -> FldNumElt
{}

K<alpha>:=K;

//Compute sigma g minus g in the case that the etale algebra is a field K
b:=0;
for i:=1 to Degree(K) do
    if IsEven(i-1) then
        b:=b+a[i]*(alpha^(i-1));
    else 
        b:=b+(-1)*a[i]*(alpha^(i-1));
    end if;
end for;
return a*b;
end intrinsic;


/**************************************IsCubeLocally*****************************************
 * 
 * Given a local field and an element of this field, checks whether this element is a cube 
 * by using a version of Hensel's Lemma:
 * An element of valuation 0 is a cube iff it is a cube in the quotient ring O/ P^(2*v(3)+1)
 * where P denotes the maximal ideal of valuation ring O of K.
 * 
 * ******************************************************************************************/
intrinsic IsCubeLocally(K :: FldPad, a :: FldPadElt)-> BoolElt
{}

if (Valuation(a) mod 3) ne (0 mod 3) then
    return false;
end if;

pi:=UniformizingElement(K);
O:=Integers(K);

b:=a*pi^(-Valuation(a));        //b has Valuation 0

if Valuation(K!3) eq 0 then      
//enough to check if b is a cube in the residue field since x^3-b is separable (Hensel Lemma)
    F,q:=ResidueClassField(O);          
    Fx<x>:=PolynomialRing(F);
    if HasRoot(x^3-q(O!b)) then
        return true;
    else 
        return false;
    end if;
else
    e:=Valuation(K!3);
    R:=quo<O | pi^(2*e+1)>;
    Ox<x>:=PolynomialRing(O);   
//Can't work properly in the quotient R somehow, why not just HasRoot in K?
    if HasRoot(x^3-O!b) then
        return true;
    else 
        return false;
    end if;
end if;

end intrinsic;


/***************************************LocalFieldModThree*********************************
 * 
 * Given a local field K, this function tries to describe K^\star/(K^\star)^3 by giving an
 * abstract F_3 vector space and a map h: K -> V.
 * 
 * Actually the MAGMA intrinsic pSelmerGroup does the job, however, it runs into an error for the
 * same examples as the function LocalFieldModThree2 written by Geishauser.
 *
 * TO DO: Use a new MAGMA version and see if the bug is fixed.
 * DONE: This seems to work now (22.09.2021) 
 * 
 * ***************************************************************************************/
intrinsic LocalFieldModThree3(K:: FldPad) -> ModTupFld, Map
{}
   X, mX := pSelmerGroup(3, K);  
   V:=KSpace(GF(3),#Generators(X));
   h := map< K->V | x:->V ! Eltseq(mX(x))>;
   return V, h;
end intrinsic;


/************************************EllLocalDim*****************************************
 * 
 * Computes the dimension of E(Kv)/3E(Kv) for a number field K and a place v in K.
 * See also Lemma 6.4.
 * 
 * *************************************************************************************/

intrinsic EllLocalDim(E :: CrvEll, K :: FldNum, v :: RngOrdIdl)-> RngIntElt
{}

Kv2,i:=Completion(K,v);
Kv:=ChangePrecision(Kv2,400);
E:=BaseChange(E,K);
b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
av:=Kv!i(a);
bv:=Kv!i(b);
Ev:=EllipticCurve([av,bv]);
Av,qv:=ConstrAlgebra(Ev);
fv:=Modulus(Av);
n:=1;                                   // n is going to be the cardinality of E(Kv)[3]
for i:=1 to #Factorisation(fv) do
    if Degree(Factorisation(fv)[i][1]) eq 1 then
        n:=n+1;
    end if;
end for;

if Valuation(3,Place(v)) eq 0 then
    return Valuation(n,3);

else return Valuation(n*3^Degree(Kv),3);
end if;

end intrinsic;

//same intrinsic as above, but for the rationals Q

intrinsic EllLocalDim(E :: CrvEll, v :: RngIntElt)-> RngIntElt
{}

Qv:=pAdicField(v,400);
b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
av:=Qv!a;
bv:=Qv!b;
Ev:=EllipticCurve([av,bv]);
Av,qv:=ConstrAlgebra(Ev);
fv:=Modulus(Av);
n:=1;                                   // n is going to be the cardinality of E(Kv)[3]
for i:=1 to #Factorisation(fv) do
    if Degree(Factorisation(fv)[i][1]) eq 1 then
        n:=n+1;
    end if;
end for;

if v ne 3 then
    return Valuation(n,3);

else return Valuation(n,3)+1;
end if;

end intrinsic;

/************************************MyKernels***********************************************
 * 
 * This intrinsic is used for the computation of Selmer groups: We first compute A(S,3) and
 * cut this group down by local restrictions. We put the result into the intrinsic MyKernels.
 * MyKernels then intersects the remaining group with ker(\overline(u)) \cap ker(\overline(\theta))
 * See also Corollary 5.27.
 * 
 * *****************************************************************************************/

intrinsic MyKernels(V :: ModTupFld, m :: Map, E :: CrvEll)-> ModTupFld 
{}

if #V eq 1 then
    return V;
end if;

// DimOfImage := Dimension(V) - 2*#SigmaK;

K:=Codomain(m);
BasisImage:=[];
n:=Dimension(V);
BasisRepImage:=[* *];
Kx<x>:=PolynomialRing(K);

B,qb:=ConstrAlgebraB(E);
fb:=Modulus(B);
if not IsIrreducible(fb) then
    print "fb is not irreducible!";
end if;
NFB<z>:=NumberField(fb);

be:=Eltseq(E)[5];
ae:=Eltseq(E)[4];
D:=-4*ae^3-27*be^2;


By<y>:=PolynomialRing(NFB);
tb:=-(z^4+ae)/(2*z);
c0:=(1/3)*ae^2*z+z^2*(z^4+2*ae)*tb+z^3*tb^2+tb^3;
c1:=z^2*(z^4+2*ae)+2*z^3*tb+3*tb^2;
c2:=z^3+3*tb;


qc:=y^3-c2*y^2+c1*y-c0;

C,mc:=quo<By | qc>;


sc:=function(s,K,mc)         //s is in K
    seq:=Eltseq(s);
    c:=0;
    for i:=1 to Degree(K) do
        c:=c+seq[i]*y^(i-1);
    end for;
    c:=mc(c);
    char:=CharacteristicPolynomial(c);
    ev:=Evaluate(char,0);
return -ev;
end function;


h2:=map< V->NFB | v:-> sc(m(v),K,mc)>;

//Now we repeat the upper calculation with h replaced by h2
BasisImage2:=[];
BasisRepImage2:=[* *];

// print "A loop of length ", n;
for i:=1 to n do
    // printf "%o ", i;
    v:=BasisElement(V,i);
    a:=h2(v);
    // if HasRoot(y^3-h2(v)) then
    if HasRoot(y^3-a) then
        Append(~BasisRepImage2,[GF(3)!0 : i in [1..n]]);
        continue i;
    elif #BasisImage2 eq 0 then
        Append(~BasisImage2, a);
        // Append(~BasisImage2, h2(v));
        Append(~BasisRepImage2,[i eq 1 select GF(3)!1 else 0 : i in [1..n]]);
        continue i;
    else 
        M:=CartesianPower([0..2],#BasisImage2);
	// print "#M = ", #M;
    
        independent:=true;
        // a:=h2(v);
        for tup in M do
            combi:=1;
            for j:=1 to #BasisImage2 do
                combi:=combi*BasisImage2[j]^tup[j];
            end for;
            if HasRoot(y^3-combi*(a^-1)) then
                independent:=false;
                representation:= TupleToList(tup) cat Seqlist([GF(3)!0 : k in [1..n-#BasisImage2] ]);
                Append(~BasisRepImage2, representation);
                break tup;
            end if;
        end for;
        
        if independent eq true then
            Append(~BasisImage2,a);
            Append(~BasisRepImage2, [i eq #BasisImage2 select GF(3)!1 else 0 : i in [1..n]]);
        end if;
    end if;
end for;

MSeq2:=[];
for i:=1 to n do
    for j:=1 to n do
        Append(~MSeq2, BasisRepImage2[i][j]);
    end for;
end for;
M2:=Matrix(GF(3),n,n, MSeq2);


//Construct second kernel
ker:=Kernel(M2);
s:=Dimension(V);
NeedSpace:=KSpace(GF(3),s);
mapNeedSpace:=map< NeedSpace->V | v:-> &+[v[i]*BasisElement(V,i) : i in [1..s]] >;
ker1:=[];

for i:=1 to Dimension(ker) do
    Append(~ker1, mapNeedSpace(BasisElement(ker,i)));
end for;
    
W:=sub<V | ker1>;


return W;

end intrinsic;

/* This is the last intersection which has to be done in the computation of the (generalized) Selmer group, see
 * steps 7 and 8 in Schaefer/Stoll, p-Descent on elliptic curves, page 1225. Geishauser has done this without computing
 * B(SB1, 3). However, computing B(SB1, 3) and then using linear algebra for the computation
 * of the kernel is much faster.
*/
intrinsic LastIntersection(h :: Map, EK::CrvEll, S::SetEnum, L1ToL::Map, AS3IntoField::Map)-> GrpAb
{}
    U := Domain(h);
    L := Codomain(h);
    if #U eq 1 then
        return U;
    end if;

    Lx<x>:=PolynomialRing(L);

    B,qb:=ConstrAlgebraB(EK);
    f_B:=Modulus(B);
    if not IsIrreducible(f_B) then
        print "f_B is not irreducible!";
    end if;
    L_B<z>:=NumberField(f_B);
    OL_B:=Integers(L_B);
    L1_B:=AbsoluteField(L_B);
    OL1_B:=Integers(L1_B);
    bool,L1ToL_B:=IsIsomorphic(L1_B,L_B);


    SB1 := PlacesOverS(SetToSequence(S), OL_B, OL1_B, L1ToL_B);
    print "computing BS3..";

    BS3,FieldIntoBS3,mBas, Bas:=pSelmerGroup(3,SB1:Raw);
    BS3IntoField := map<BS3->L1_B | alpha:->PowerProduct(Eltseq(Bas), [c mod 3 : c in Eltseq(mBas(alpha))])>;

    be:=Eltseq(EK)[5];
    ae:=Eltseq(EK)[4];
    D:=-4*ae^3-27*be^2;

    By<y>:=PolynomialRing(L_B);
    tb:=-(z^4+ae)/(2*z);
    c0:=(1/3)*ae^2*z+z^2*(z^4+2*ae)*tb+z^3*tb^2+tb^3;
    c1:=z^2*(z^4+2*ae)+2*z^3*tb+3*tb^2;
    c2:=z^3+3*tb;


    qc:=y^3-c2*y^2+c1*y-c0;

    C,mc:=quo<By | qc>;


    sc:=function(s,L,mc)         //s is in L \simeq K[x]/(f), and note that f divides qc
        seq:=Eltseq(s);
        c:=0;
        for i:=1 to Degree(L) do
            c:=c+seq[i]*y^(i-1);
        end for;
        c:=mc(c);
        char:=CharacteristicPolynomial(c);
        ev:=Evaluate(char,0);
    return -ev;
    end function;

    h2:=hom< U->BS3 | u:-> FieldIntoBS3( (L1ToL_B^-1)(sc( L1ToL( AS3IntoField(u) ), L, mc)))>;
    print "Compute the kernel";
    W := Kernel(h2);
    return W;
end intrinsic;

/*********************************MyThreeSelmerGroup***********************************************
 * 
 * EQ is an elliptic curve defined over Q 
 * EK is the same curve but defined over some number field K.
 * Sigma is a finite set of prime ideals in M_K.
 *
 * This function computes the generalized Selmer group Sel^3(E/K, Sigma). It returns the group as an abstract group
 * together with maps from the Selmer group into the number field L and vice versa.
 * 
 * We assume that the étale algebra A = L is a field.
 * 
 * This intrinsic is essentially written by Chris Geishauser with some improvement by Werner Bley.
 *
 * *************************************************************************************************/
intrinsic MyThreeSelmerGroup(EQ :: CrvEll, EK :: CrvEll, Sigma :: SeqEnum)-> GrpAb, Map, Map, SetEnum, Map
{}

K:=BaseField(EK);
fprintf "Cannon.m", "K = %o\n\n", K;
Kz<z1>:=PolynomialRing(K);
A,q:=ConstrAlgebra(EK);
f:=Modulus(A);
if not IsIrreducible(f) then
    print "the defining polynomial for A is not irreducible";
end if;
L:=NumberField(f);
OL:=Integers(L);
//We need L1 to use certain magma functions such as pSelmerGroup
L1:=AbsoluteField(L);
OL1:=Integers(L1);
bool,L1ToL:=IsIsomorphic(L1,L);
// Construct the set S of bad primes 
S:=ConstructS(EQ,K,3);


S:=SequenceToSet(S) join Set(Sigma);
SA1 := PlacesOverS(SetToSequence(S), OL, OL1, L1ToL);
print "computing AS3..";

/* It is important to use Raw here! One can then define AS3IntoField as below and read all exponents mod 3
   which speeds up the computation a lot!.
*/
AS3,FieldIntoAS3,mB, B:=pSelmerGroup(3,SA1:Raw);
AS3IntoField := map<AS3->L1 | alpha:->PowerProduct(Eltseq(B), [c mod 3 : c in Eltseq(mB(alpha))])>;

//Construct AS3 as vector space to be able to use linear algebra
    
s:=#Generators(AS3);
VSAS3:=KSpace(GF(3),s);
VSToAS3:=map<VSAS3->AS3 | v:-> &+[(Integers()!Coordinates(VSAS3,v)[i])*AS3.i : i in [1..s] ]>;
AS3ToVS:=map<AS3->VSAS3 | x:-> &+[Eltseq(x)[i]*BasisElement(VSAS3,i) : i in [1..s] ]>;
V0:=VSAS3;

//check local restrictions

for v in (S diff Set(Sigma)) do
    fprintf "Cannon.m", "v = %o\n\n", v;
    Kv2,iota:=Completion(K,v);
    Kv2`DefaultPrecision:=400;
    Kv:=ChangePrecision(Kv2,400);
    Ov:=Integers(Kv);
    pi:=UniformizingElement(Ov);
    b:=Eltseq(EQ)[5];
    a:=Eltseq(EQ)[4];
    av:=iota(a);
    bv:=iota(b);
    Ev:=EllipticCurve([av,bv]);
    Av,qv:=ConstrAlgebra(Ev);
    fv:=Modulus(Av);
    Dv:=-4*av^3-27*bv^2;
    Fac:=Factorisation(fv);

    //Construct the algebra Av:=Kv[x]/(f) as a product of local fields E_i

    listFields:=[];
    listVSMap:=[];
    listRingToLoc:=[];
    Z:=Integers();
    Kvz<z>:=PolynomialRing(Kv);
    for i:=1 to #Fac do
        LocField1:=LocalField(Kv,Fac[i][1]);
        LocField,maprl:=RamifiedRepresentation(LocField1);
        //Construct the maps E_i^\star / (E_i^\star)^3 as abstract vector space together
        //with maps E_i -> E_i^\star / (E_i^\star)^3
        //V,mapE:=LocalFieldModThree2(LocField);
	fprintf "Cannon.m", "LocField = %o\n\n", LocField;
	V,mapE:=LocalFieldModThree3(LocField); /* TO DO: returns an error in certain cases. This is a MAGMA bug!!!! 
	                                          The error first ocurs for E = 89a1 and p=19 in the MAGMA intrinsic
						  pSelmerGroup  
					       */
        Append(~listFields,LocField);
        listRingToLoc:=listRingToLoc cat [maprl];
        listVSMap:=listVSMap cat [mapE];
    end for;    

    //Now we construct the subspace of E(Kv) in \prod E_i^\star / (E_i^\star)^3
    //We search for random points in E(Kv) and map them into \prod E_i^\star / (E_i^\star)^3
    // until the image generates a subspace of the right dimension
    
    //First construct \prod E_i^\star / (E_i^\star)^3 as abstract vector space
    //Compute dimension of cartesian product space
    dimVS:=0;
    for i:=1 to #Fac do
        dimVS:=dimVS+Dimension(Codomain(listVSMap[i]));
    end for;
    
    //Maybe there is a shorter way to do the next step
    //Construct product space and maps into this space
    MapsIntoProdSpace:=[];
    ProdSpace:=KSpace(GF(3),dimVS);
    
    fctspace:=function(v,V,i,VS) //maps a vector in E_i^\star / (E_i^\star)^3 into ProdSpace
        embv:=VS!0;
        coord:=Coordinates(V,v);
        for j:=1 to Dimension(V) do
            embv:=embv+coord[j]*BasisElement(VS,i+j-1);
        end for;
        return embv;
    end function;
    
    l:=1;
    for i:=1 to #Fac do
        EiModThree:=Codomain(listVSMap[i]);
        EintoProdE:=map<EiModThree->ProdSpace | v:-> fctspace(v,EiModThree,l,ProdSpace)>;
        l:=l+Dimension(EiModThree);
        MapsIntoProdSpace:=MapsIntoProdSpace cat [EintoProdE];
    end for;
    
    //Now we are finally able to compute the image of E(Kv) in ProdSpace
    //first we map potential 3-torsion points into CarProd
    //Compute 3-torsion:
    TorsList:=[];
    if #TorsList ge 1 then
        print "Torslist not empty!";
    end if;
    
    // P is a point in Ev
    //maplist1 is a list of maps from Kv[x] to E_i for all i
    //maplist2 is a list of maps from E_i to E_i^\star / (E_i^\star)^3 for all i
    //maplist3 is a list of maps from E_i^\star / (E_i^\star)^3 to ProdSpace
    //maplist4 is supposed to contain the maps RingToLoc
    //This function computes the connecting homomorphism of a point P in E(Kv)
    EvInCarProd:=function(P,maplist1,maplist2,maplist3,maplist4,Ev,fv)
        ProdPol:=ConHom(Ev,Ev!P,fv);                   //a sequence of polynomials in Kv[x]
        ProdInEi:=[];
        for i:=1 to #Fac do           //Fix some field E_i
            mapEi2:=maplist2[i];
            Ei:=Domain(mapEi2);
            mapEi3:=maplist3[i];
            for j:=1 to #Fac do       //Find the corresponding polynomial in ProdPol
                if DefiningPolynomial(Domain(maplist4[i])) eq ProdPol[j][2] then
                   Append(~ProdInEi,mapEi3(mapEi2(maplist1[i](ProdPol[j][1]))));
                end if;
            end for;
        end for;
        return &+(ProdInEi);
    end function;
    
    //construct maplist1
    maplist1:=[];
    for i:=1 to #Fac do
        Ei:=listFields[i];
        mapEi:=listRingToLoc[i];
        REi:=Domain(mapEi);
 //       Eiz<z>:=PolynomialRing(Ei);
 //       alphai:=Roots(Eiz!DefiningPolynomial(Ei))[1][1];
        mapIntoE:=map<Kvz->Ei | g:->    &+[Coefficient(g,j)*mapEi(REi.1)^j : j in [0..Degree(g)]] >;
        maplist1:=maplist1 cat [mapIntoE];
    end for;
    
    //Construct our subspace W \simeq E(Kv)/3E(Kv) in ProdSpace
    W:=sub<ProdSpace | [ EvInCarProd(P,maplist1,listVSMap,MapsIntoProdSpace, listRingToLoc,Ev,fv) : P in TorsList] >;
    LocalDim:=EllLocalDim(EQ,K,v);
 
    //Now search for random points in E(Kv) to find remaining generators
    while LocalDim ne Dimension(W) do

        hopex := pi^(2*Random(2))*Random(Ov);
        if IsSquare(hopex^3+av*hopex+bv) eq true then
            nice,hopey:=IsSquare(hopex^3+av*hopex+bv);
            Point:=Ev![hopex,hopey];
            im:=EvInCarProd(Point,maplist1,listVSMap,MapsIntoProdSpace,listRingToLoc,Ev,fv);
            W:=sub<ProdSpace | W,im>;
        end if;
    end while;
    
    //Use linear algebra (or rather matrix multiplication) to make the computation (hopefully) feasible
    print "mapping basis elements and checking local conditions, second please..";
    
    //Consruct the image of the basis vectors of VSAS3 in Prodspace
    ImageBasis:=[];
    s:=Dimension(VSAS3);
    print "we map"; s; "elements    ";
    for k:=1 to s do
        printf "%o ", k;
        //map the abstract basis vectors into L
        bk:=BasisElement(VSAS3,k);
        bkA:=VSToAS3(bk);
        fieldEl1:=L1!(AS3IntoField(bkA));
	//fieldEl1:=L1!((FieldIntoAS3^-1)(bkA));
        //Construct the corresponding element in K[x]/(f) or rather a representative
        //and map this element into Kv[x]
        polEl:=Kvz!0;
        fieldEl:=L1ToL(fieldEl1);
        for j:=1 to Degree(L) do
            polEl:=polEl+Kv!(iota(fieldEl[j]))*z^(j-1);
        end for;
        //Bring the polynomial into ProdSpace
        LocalImageSequence:=[];
        for i:=1 to #Factorisation(fv) do
            mapEi2:=listVSMap[i];
            mapEi3:=MapsIntoProdSpace[i];
            LocalImagei:=mapEi3(mapEi2(maplist1[i](polEl)));
            Append(~LocalImageSequence,LocalImagei);
        end for;
        LocalImage:=&+(LocalImageSequence);
        Append(~ImageBasis,LocalImage);
    end for;
    //Set up the matrix
    MSeq:=[];
    
    for i:=1 to s+Dimension(W) do
        
        if i le s then
            for j:=1 to Dimension(ProdSpace) do
            Append(~MSeq, ImageBasis[i][j]);
            end for;
        else
            for j:=1 to Dimension(ProdSpace) do
            Append(~MSeq, -BasisElement(W,i-s)[j]);
            end for;
        end if;
            
    end for;
    M:=Matrix(GF(3),s+Dimension(W),Dimension(ProdSpace),MSeq);

    print "check which elements in AS3 lie in W";    
    ker:=Kernel(M);
    
    //This space exists because VSAS3 lives (maybe) in VS of bigger dimension than s
    NeedSpace:=KSpace(GF(3),s);
    mapNeedSpace:=map< NeedSpace->VSAS3 | v:-> &+[v[i]*BasisElement(VSAS3,i) : i in [1..s]] >;
    ker1:=[];
    //We just want the first s coordinates
    for i:=1 to Dimension(ker) do
        Append(~ker1, mapNeedSpace(NeedSpace![BasisElement(ker,i)[j] : j in [1..s]]));
    end for;
    
    VSAS3:=sub<VSAS3 | ker1>;
        
end for;

h:=map<VSAS3->L | v:-> L1ToL(AS3IntoField(VSToAS3(v)))>;
// h:=map<VSAS3->L | v:-> L1ToL((FieldIntoAS3^-1)(VSToAS3(v)))>; 
print "Done! Cutting down remaining group by testing global conditions..";


/* We now compute the intersection of U with ker(\bar\theta_g) where \bar\theta_g : A^times/A^{\times p} -> A^times/A^{\times p}.
   Here U is the subgroup that we already found by considering the local conditions. yhom will be the homomorphism 
   \bar\theta_g (see [Geis, page 53]). Note that yhom maps into AS3 !!!
*/ 
U := sub<AS3 | [VSToAS3(BasisElement(VSAS3, i)) : i in [1..Dimension(VSAS3)]]>; 
yhom := hom<U->AS3 | u:->FieldIntoAS3((L1ToL^-1)(sigmagminusg(L,L1ToL(AS3IntoField(u)))))>;   
U := Kernel(yhom);

h:=map<U->L | u:-> L1ToL(AS3IntoField(u))>;

/* It finally remains to intersect U1 with ker(\bar u) (see [Geis, page 53, Cor.5.27]).
*/

SelmerGroup := LastIntersection(h, EK, S, L1ToL, AS3IntoField);

SelmerMap:=map<SelmerGroup -> L | v:-> L1ToL(AS3IntoField(v))>;
SelmerMapInvers:=map< L -> AS3 | l:-> FieldIntoAS3((L1ToL^-1)(l))>;

return SelmerGroup, SelmerMap, SelmerMapInvers, SA1, L1ToL; 

end intrinsic;


/*************************************AdmPlace*************************************************
 * 
 * This is part of the realization of Chapter 8. In particular we impose its assumptions.
 * For instance Sel(E/K)\simeq E(K)/3E(K) and the free generators of E(K) lie in E(Q).
 * E/Q is an elliptic curve, K is a number field of degree 3 over Q.
 * The function finds for every free generator P of E(Q) an admissible place v such that
 * P is not trivial in E(Qv)/3E(Qv).
 * One should first check whether Gal(Q(E[3])/Q) contains the automorphism E[3] -> E[3] : Q |-> -Q
 * since that would ensure the existence of infinitely many such v's. 
 * We make a small check at the beginning by checking whether
 * (48/ #Gal(Q(E[3])/Q)) is divisble by 2. 48 is the size of Aut(E[3]). 
 * The returned list consists of pairs [*v,P*] such that v is admissible for P.
 *
*********************************************************************************************/

// intrinsic AdmPlace(E :: CrvEll, K :: FldNum) -> List
intrinsic AdmPlace(E :: CrvEll, K :: FldNum, Ps :: SeqEnum) -> List
{}
EQ:=E;
QX<x>:=PolynomialRing(Rationals());
b:=Eltseq(E)[5];
a:=Eltseq(E)[4];
D:=-4*a^3-27*b^2;
f:=x^8+8*b*x^6-(2/3)*D*x^4-(1/27)*D^2;

seq:=[];

L:=SplittingField(f);
if not IsDivisibleBy(Integers()!(48/Degree(L)),2) then
    print "infinitely many admissible sets will exist!";
end if;

FreeGen := Ps;
listplaces:=[* *];
//We now find for every generator P in FreeGen an admissible place
v:=1;

for P in FreeGen do

    bool:=false;
    
while bool eq false do
    
    v:=NextPrime(v);
    
    if v in BadPrimes(E) then
        continue;
    end if;
    if v eq 3 then
        continue;
    end if;
    if not #Decomposition(K,v) eq 3 then
        continue;
    end if;
    if not #Roots(f,pAdicField(v)) eq 8 then
       continue;
    end if;
    if v in BadPrimes(E) or v eq 3 or not #Decomposition(K,v) eq 3 or not #Roots(f,pAdicField(v)) eq 8 then
        continue;
    end if;
//Check whether the local restriction map is injective on <P>, i.e. if P in E(Qv)/3E(Qv) is nontrivial.
// the assumptions on v imply E(Qv)/3E(Qv) \simeq E(Fv)/3E(Fv), where Fv is the residue field of Qv.    

Qv:=pAdicField(v,200);
Ov:=Integers(Qv);
pi:=UniformizingElement(Ov);
b:=Eltseq(EQ)[5];
a:=Eltseq(EQ)[4];
av:=Qv!a;
bv:=Qv!b;
Ev:=EllipticCurve([av,bv]);

//map P into E(Qv)
Pv:=Ev![Qv!(P[1]),Qv!(P[2]),Qv!(P[3])];

F,j:=ResidueClassField(Ov);
Ef:=ChangeRing(Ev,F);    
    
Pf:=Ef!Pv;
    
    
G,phi:=AbelianGroup(Ef);    

//Construct G/3G.
seq:=[];    
    for g in G do
        Append(~seq, 3*g);
    end for;
    
    Red,r:=quo< G | seq>;
    
    Pg:=0;
    for g in G do
        if phi(g) eq Pf then
            Pg:=g;
        end if;
    end for;
    
    if r(Pg) eq r(G!0) then
        continue;
    end if;

    Append(~listplaces, [*v, P*]);
    
    bool:=true;
    
end while;
end for;

return listplaces;

end intrinsic;


/***************************************galois_action_selmer********************************
 * 
 * selEl is an element in the general Selmer group, sigma is an automorphism in Gal(K/Q).
 * SelmerToField: GeneralSel -> L1 and SelmerInvers: L1 -> GeneralSel are the maps given in the
 * intrinsic f4map. This function computes the galois action of sigma on selEl.
 * This intrinsic is tailored to the intrinsic f4Map and you can find the maps SelmerToField, 
 * SelmerInvers and L1ToL defined in f4Map.
 * 
 * ***************************************************************************************/
intrinsic galois_action_selmer(selEl :: GrpAbElt, sigma :: Map, SelmerToField :: Map, SelmerInvers:: Map, L1ToL :: Map) -> GrpAbElt 
{}

L:=Codomain(L1ToL);
//map SelEl into L1
fieldel1:=SelmerToField(selEl);
//we need fieldel1 to be an element of L
fieldel:=L1ToL(fieldel1);
K:=BaseField(L);
Kx<z>:=PolynomialRing(K);
//find the expression of fieldel in L = K[X]/(f) as a polynomial
pol:=0;
for j:=1 to Degree(L) do
    pol:=pol+fieldel[j]*z^(j-1);
end for;
//apply sigma on the coefficients of pol, which is done by PolAction   
sigma_pol:=PolAction(pol, sigma);
//map sigma_pol into L
sigma_pol_in_L:=Evaluate(sigma_pol, L.1);
//and finally we map sigma_pol_in_L back to the general Selmer group
sigma_selEl:=SelmerInvers((L1ToL^-1)(sigma_pol_in_L));

return sigma_selEl;

end intrinsic;


/**************************** GaloisAction *************************************************
 *
 * This is an adaption of Geishauser's galois_action_selmer. Whereas he returns sigma(selEl) as an element
 * of the supergroup which we now call Eps (Eps is constructed from units and class group while computing the 
 * Sigma-Selmer group) we now return sigma(selEl) as an element of the Sigma-Selmer group.
 *
********************************************************************************************/
intrinsic GaloisAction(selEl :: GrpAbElt, sigma :: Map, SelGrp::Rec) -> GrpAbElt
{}
    local L, fieldel1, fieldel, K, Kx, pol, j, sigma_pol, sigma_pol_in_L, sigma_selEl, Fp, Eps, Sel, GeneralSelmerGens,
          A, s, t, B, b, x;

    SelmerToField := SelGrp`SelmerToField; SelmerInvers := SelGrp`SelmerInvers; 
    SigmaL1 := SelGrp`SigmaL1; xmap := SelGrp`xmap;

    L:=Codomain(SelGrp`SelmerToField);
    fieldel:=SelmerToField(selEl);

    K:=BaseField(L);
    Kx<z>:=PolynomialRing(K);
    //find the expression of fieldel in L = K[X]/(f) as a polynomial
    pol:=0;
    for j:=1 to Degree(L) do
        pol:=pol+fieldel[j]*z^(j-1);
    end for;
    //apply sigma on the coefficients of pol, which is done by PolAction   
    sigma_pol:=PolAction(pol, sigma);
    //map sigma_pol into L
    sigma_pol_in_L:=Evaluate(sigma_pol, L.1);
    assert CheckValuations(sigma_pol_in_L, SigmaL1, xmap);

    sigma_selEl:=SelmerInvers(sigma_pol_in_L);

    Eps := Codomain(SelmerInvers);
    Sel := Domain(SelmerToField);
    Fp := GF(Exponent(Sel));
    GeneralSelmerGens := [g : g in Generators(Sel)];
    A := [Eltseq(Eps!v) : v in GeneralSelmerGens];
    s := #GeneralSelmerGens;
    t := #Generators(Eps);
    B := Matrix(Fp, s, t, A);
    b := Matrix(Fp, 1, t, Eltseq(sigma_selEl));
    x := Solution(B, b);
    sigma_selEl := &+[(Integers()!x[1,i])*(GeneralSelmerGens[i]) : i in [1..s]];

    return sigma_selEl;
end intrinsic;


/********************************* ComputeSelmerRep *****************************************
 * Computes the representation induced by the generaliszed Selmer group Sel_p(E/K, \Sigma).
 * Here K/Q is a Galois extension with goup Gal. All input is computed by Geishauser's MyThreeSelmerGroup.
 *
*********************************************************************************************/
intrinsic ComputeSelmerRep(SelGrp::Rec, automap::Map) -> HomGrp
{}
    local l, Gal, C, s, i, c, Gl, SelmerRep;

    Gal := Domain(automap);
    C := [];
    s := #Generators(SelGrp`Sel);
    for i:= 1 to s do
        c := Eltseq( GaloisAction(SelGrp`Sel.i, automap(Gal.1), SelGrp) );
        Append(~C, c);
    end for;

    F := GF(Exponent(SelGrp`Sel));
    Gl := GL(s, F);
    SelmerRep := hom<Gal->Gl | [Gal.1 -> (Gl ! Transpose(Matrix(F, s, s, C)))] >;

    return SelmerRep;
end intrinsic;


/********************************* GaloisAction *****************************************
 * Computes the action of sigma in Gal (as an abstract permutation group) on an element selEl
 * of the Sigma-Selmer group (computed by MyThreeSelmerGroup and represented as an abstract abelian group).
 *
 * It is assumed that the representation induced by the Sigma-Selmer group is already computed and given by the
 * homomorphism SelmerRep : Gal(K/Q) -> Gl_s(Fp). 
 *
*********************************************************************************************/
intrinsic GaloisAction(selEl::GrpAbElt, sigma::GrpPermElt, SelmerRep::Map) -> GrpAbElt
{}

    F := GF(#Domain(SelmerRep));
    s := #Eltseq( selEl );
    x := Matrix(F, s, 1, Eltseq( selEl ));
    y := SelmerRep(sigma)*x;

    return Parent(selEl) ! Eltseq(y);
end intrinsic;


/********************************* Trace *****************************************
 * Computes the trace (or norm) of an element selEl of the Sigma-Selmer group.
 *
 * It is assumed that the representation induced by the Sigma-Selmer group is already computed and given by the
 * homomorphism SelmerRep : Gal(K/Q) -> Gl_s(Fp). 
 *
*********************************************************************************************/
intrinsic Trace(selEl::GrpAbElt, SelmerRep::Map) -> GrpAbElt
{}

    return &+[GaloisAction(selEl, g, SelmerRep) : g in Domain(SelmerRep)];
end intrinsic;

/********************************* KummerMap *****************************************
 * Computes the image of a point Q in E(K) in the Sigma-Selmer group
 * which is computed by MyThreeSelmerGroup and represented as an abstract abelian group.
 *
*********************************************************************************************/
intrinsic KummerMap(Q::PtEll, f::RngUPolElt, SelGrp::Rec) -> GrpAbElt
{}
    local EK, L, delta, deltapol, deltaL, deltaSel, Eps, Fp, GeneralSelmerGens, A, s, t, B, b, x, kummer_of_Q;

    Sel := SelGrp`Sel; SelmerInvers := SelGrp`SelmerInvers; 
    SigmaL1 := SelGrp`SigmaL1; xmap := SelGrp`xmap;
    
    L := Codomain(SelGrp`SelmerToField);
    EK := Curve(Q);

    delta:=ConHom(EK,Q,f);      //deltapol[1][1] is an element in K[X]
    deltapol:=delta[1][1];
    deltaL:=Evaluate(deltapol, L.1);
    deltaSel:=SelmerInvers(deltaL);

    Eps := Codomain(SelmerInvers);
    Fp := GF(Exponent(Sel));
    GeneralSelmerGens := [g : g in Generators(Sel)];
    A := [Eltseq(Eps!v) : v in GeneralSelmerGens];
    s := #GeneralSelmerGens;
    t := #Generators(Eps);
    B := Matrix(Fp, s, t, A);
    b := Matrix(Fp, 1, t, Eltseq(deltaSel));
    x := Solution(B, b);
    kummer_of_Q := &+[(Integers()!x[1,i])*(GeneralSelmerGens[i]) : i in [1..s]];

    return kummer_of_Q;
end intrinsic;

/********************************* NormEquation *****************************************
 * Given an element xi in the Sigma-Selmer group fixed by Gal(K/Q), so xi in Sel^Gal, this routine computes
 * a preimage under the trace (or norm).
 *
 * It is assumed that the representation induced by the Sigma-Selmer group is already computed and given by the
 * homomorphism SelmerRep : Gal(K/Q) -> Gl_s(Fp). 
 *
*********************************************************************************************/
intrinsic NormEquation(xi::GrpAbElt, SelmerRep::HomGrp) -> GrpAbElt
{}
    local Sel, F, s, A, b, x, eta;

    Sel := Parent(xi);
    F := GF(Exponent(Sel));
    s := #Generators(Sel);
    A := Matrix(F, s,s, [Eltseq( Trace(Sel.i, SelmerRep) ) : i in [1..s]]);
    b := Matrix(F,1,s, Eltseq(xi));
    x := Solution(A, b);
    eta := Sel ! Eltseq(x);
    
    return eta;
end intrinsic;


/************************** ComputePhiMatrix ************************************************
 * E is an elliptic curve over Q
 * K a number field of degree l = 3
 * P a generator of E(K)
 * automap a Map given e.g. by AutomorphismGroup and describes gal(K/Q).
 * 
 * This intrinsic computes the matrix Phi as in etncIV in the following case: E(K)_3 \simeq \Z_3^r, r > 0.
 *
*******************************************************************************/ 
intrinsic ComputePhiMatrix(E :: CrvEll, K :: FldNum, Ps :: SeqEnum, automap :: Map) -> AlgMatElt
{}

    //first find an admissible set Sigma of places of the rationals Q
    Gal := Domain(automap);
    SigmaPairs:=AdmPlace(E,K,Ps);
    
    //For every w in Sigma(K) find a basis of E(K_w)/3E(K_w)
    //One basis element is always a free generator P of E/Q
    Basis:=[* *];
    SigmaK:=[];

    for T in SigmaPairs do
        v:=T[1];
        P:=T[2];

        vBasis := newBDLocalBasis(E, K, automap, v, P , 3);
        assert TestBDLocalBasis(E, K, vBasis);
        for vB in vBasis do
            Append(~Basis, T cat vB);
            Append(~SigmaK, Ideal(vB[1]));
        end for;
    end for;

    // debug
    assert CheckBasis(Basis);

    //Compute the generalized Selmer group Sel^3(E/K, Sigma(K))
    EK:=BaseChange(E,K);
    print "We compute the general Selmer group, this may take up to two minutes or so.";
    SelGrp := rec<GeneralSelmerGroup | >;
    time SelGrp`Sel, SelGrp`SelmerToField, SelGrp`SelmerInvers, SelGrp`SigmaL1, SelGrp`xmap:=MyThreeSelmerGroup(E,EK, SigmaK);
    
    print "Compute the matrix Phi; must compute ", (#Ps)^2, "Mazur-Tate pairings";
    SelGrp`SelmerRep := ComputeSelmerRep(SelGrp, automap);
    ws := [Basis[i,3] : i in [1..#Basis] | i mod 3 eq 1];
    // w := Basis[1,3];
    // mu := MazurTatePairing(P, P, EK, w, SelGrp);
    Phi := Matrix(#Ps, #Ps, [MazurTatePairing(P, Q, EK, ws, SelGrp) : P,Q in Ps]);
    
    return Phi;
end intrinsic;

/*************** written by Werner Bley ********************************/


intrinsic '*'(aut :: Map, P :: PtEll) -> PtEll
{}
    return Parent(P) ! [aut(s) : s in Eltseq(P)];
end intrinsic;


intrinsic CheckValuations(alpha :: FldNumElt, SigmaL1 :: SetEnum, xmap::Map) -> BoolElt
{}
     local fac, f;

     // print "In CheckValuations";
     L1 := Domain(xmap);
     OL1 := MaximalOrder(L1);
     fac := Factorization(xmap(alpha)*OL1);
     // print [f[2] : f in fac | not f[1] in SigmaL1];
     for f in fac do
         if (not f[1] in SigmaL1) and (f[2] mod 3 ne 0) then
             return false;
         end if;
     end for;
     return true;
end intrinsic;


intrinsic CheckBasis(Basis :: List) -> BoolElt
{}
     local fac, f;

     // print "In CheckBasis";

     for b in Basis do
         Pw := b[4];
	 Qw := b[5];
	 Ew := b[6];

         Lw := BaseField(Ew);
         OLw := Integers(Lw);
         F, j := ResidueClassField(OLw);
         Ef:=ChangeRing(Ew,F);
         Pf := Ef ! Pw;
         Qf := Ef ! Qw;
         G,phi:=AbelianGroup(Ef);
         gP := [g : g in G | phi(g) eq Pf][1];
         gQ := [g : g in G | phi(g) eq Qf][1];
         Gmodp, pr := quo<G | [3*G.1, 3*G.2]>;
         S := sub<Gmodp | [pr(gP), pr(gQ)]>;
         if not S eq Gmodp then
	     print "ERROR in CheckBasis";
	     return false;
	 end if;
     end for;
     // print "CheckBasis: Basis okay";
     return true;
end intrinsic;

intrinsic TestBDLocalBasis(E :: CrvEll, K :: FldNum, vBasis :: SeqEnum) -> BoolElt
{}
    for data in vBasis do
        w := data[1]; Pw := data[2]; Qw := data[3]; Ew := data[4];
        Kw, iota := Completion(K, w);
        O:=Integers(Kw);
        bw:=iota( Eltseq(E)[5] );
        aw:=iota( Eltseq(E)[4] );
        Ew:=EllipticCurve([aw,bw]);

        F,j:=ResidueClassField(O);
        Ef:=ChangeRing(Ew,F);
        Pf:=Ef!Pw;
        Qf:=Ef!Qw;

        G,phi:=AbelianGroup(Ef);
        Pfinv := [g : g in G | phi(g) eq Pf][1];
        Qfinv := [g : g in G | phi(g) eq Qf][1];
        l := 3;
        Gmodp, pr := quo<G | [l*G.1, l*G.2]>;
        S := sub<Gmodp | [pr(Pfinv), pr(Qfinv)]>;
        if #S ne #Gmodp then
              return false;
        end if;
     end for;
     return true;
end intrinsic;


intrinsic PlacesOverS(S::SeqEnum, OL::RngOrd, OLabs::RngOrd, phi::Map) -> SeqEnum
{}

    SL:=[];
    for I in S do
        for IL in Decomposition(OL,I) do
            Append(~SL, IL[1]);
        end for;
    end for;

    //Map the places into OL1
    SL1:=[];
    for I in SL do
        Gen:=Generators(I);
        seq:=[];
        for i:=1 to #Gen do
            Append(~seq, (phi^(-1))(Gen[i]));
        end for;
        Ideal:=ideal<OLabs | seq>;
        Append(~SL1, Ideal);
    end for;

    SA1:=SequenceToSet(SL1);
    return SA1;
end intrinsic;

    

